use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

#[derive(Debug,PartialEq)]
enum GameFlavour {
    CobraCommand,
    NinjaRevenge,
}

const TILE_W: usize = 24;
const TILE_H: usize = 20;

struct RawTileDecoder {
    fr:         FileReader<BufReader<File>>,
    audio:      bool,
    pal:        [u16; 16],
    data:       Vec<u8>,
    ablk_size:  usize,
    flavour:    GameFlavour,
}

impl RawTileDecoder {
    fn render_frame(&mut self) -> Vec<u16> {
        const WIDTH: usize = TILE_W * 8;
        let mut frame = vec![0; WIDTH * TILE_H * 8];

        let src = if self.flavour != GameFlavour::NinjaRevenge { &self.data } else { &self.data[8..] };

        for (strip, tiles) in frame.chunks_exact_mut(WIDTH * 8)
                .zip(src.chunks_exact(TILE_W * 32)) {
            for (x, tile) in tiles.chunks_exact(32).enumerate() {
                let dpos = x * 8;
                for (line, src) in strip.chunks_exact_mut(WIDTH).zip(tile.chunks_exact(4)) {
                    for (dst, &b) in line[dpos..].chunks_exact_mut(2).zip(src.iter()) {
                        dst[0] = self.pal[usize::from(b >> 4)];
                        dst[1] = self.pal[usize::from(b & 0xF)];
                    }
                }
            }
        }

        frame
    }
}

impl InputSource for RawTileDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  TILE_W * 8,
                    height: TILE_H * 8,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: if self.ablk_size == 2000 { 6 } else { 10 },
                }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: if self.ablk_size == 2000 { 12000 } else { 16000 },
                    sample_type: AudioSample::U8,
                    channels:    2,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        br.peek_byte().map_err(|_| DecoderError::EOF)?;
        if self.audio {
            let mut audio = vec![0; self.ablk_size * 2];
            for pair in audio.chunks_exact_mut(2) {
                pair[0] = br.read_byte()?;
            }
            br.read_skip(0x800 - self.ablk_size)?;
            for pair in audio.chunks_exact_mut(2) {
                pair[1] = br.read_byte()?;
            }
            br.read_skip(0x800 - self.ablk_size)?;
            self.audio = false;
            for el in audio.iter_mut() {
                if *el < 0x80 {
                    *el = 0x80 - *el;
                }
            }
            Ok((1, Frame::AudioU8(audio)))
        } else {
            for el in self.pal.iter_mut() {
                let clr = br.read_u16be()?;
                let b = (clr >> 8) & 0xF;
                let g = (clr >> 4) & 0xF;
                let r =  clr       & 0xF;
                *el = (r << 11) | ((r >> 3) << 10)
                    | (g <<  6) | ((g >> 3) <<  5)
                    | (b <<  1) |  (b >> 3);
            }
            br.read_skip(0x20)?;
            br.read_buf(&mut self.data)?;
            let frame = self.render_frame();
            self.audio = true;
            Ok((0, Frame::VideoRGB16(frame)))
        }
    }
}

pub fn open_cc(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    open_generic(name, GameFlavour::CobraCommand)
}

pub fn open_ninja(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    open_generic(name, GameFlavour::NinjaRevenge)
}

fn open_generic(name: &str, flavour: GameFlavour) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let (vblk_size, ablk_size) = match flavour {
            GameFlavour::CobraCommand   => (0x5000, 0x7D0),
            GameFlavour::NinjaRevenge   => (0x4000, 0x640),
        };

    br.seek(SeekFrom::Start(0))?;

    Ok(Box::new(RawTileDecoder {
        fr,
        flavour,
        pal: [0; 16],
        audio: true,
        data: vec![0; vblk_size - 0x40],
        ablk_size,
    }))
}
