use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct RLFDecoder {
    fr:      FileReader<BufReader<File>>,
    width:   usize,
    height:  usize,
    nframes: u32,
    cur_frm: u32,
    time:    u32,
    vdata:   Vec<u8>,
    frame:   Vec<u16>,
    methods: Vec<[u8; 4]>,
}

impl RLFDecoder {
    fn unpack_hrle(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = 0;
        while br.left() > 0 && pos < self.frame.len() {
            let op = usize::from(br.read_byte()?);
            if (op & 0x80) == 0 {
                let len = op + 2;
                validate!(pos + len <= self.frame.len());
                let clr = br.read_u16le()?;
                for el in self.frame[pos..][..len].iter_mut() {
                    *el = clr;
                }
                pos += len;
            } else {
                let len = 256 - op;
                validate!(pos + len <= self.frame.len());
                for el in self.frame[pos..][..len].iter_mut() {
                    *el = br.read_u16le()?;
                }
                pos += len;
            }
        }
        validate!(br.left() < 4);

        Ok(())
    }
    fn unpack_dhle(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = 0;
        while br.left() > 0 && pos < self.frame.len() {
            let op = usize::from(br.read_byte()?);
            if (op & 0x80) == 0 {
                let len = op + 1;
                validate!(pos + len <= self.frame.len());
                pos += len;
            } else {
                let len = 256 - op;
                validate!(pos + len <= self.frame.len());
                for el in self.frame[pos..][..len].iter_mut() {
                    *el = br.read_u16le()?;
                }
                pos += len;
            }
        }
        validate!(br.left() < 4);

        Ok(())
    }
}

impl InputSource for RLFDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    15,
                tb_num: self.time,
                tb_den: 10000,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        if self.cur_frm >= self.nframes {
            return Err(DecoderError::EOF);
        }
        let tag = br.read_tag()?;
        let size = br.read_u32le()?;
        validate!(&tag == b"MARF");
        validate!(size > 0x1C);
        br.read_u32le()?; // always 1?
        br.read_u32le()?; // always 1?
        let method = br.read_tag()?;
        validate!(self.methods.contains(&method));
        let hdrsize = br.read_u32le()?;
        validate!(hdrsize == 0x1C);
        br.read_u32le()?; // always 1?
        self.vdata.resize(size as usize - 0x1C, 0);
        br.read_buf(&mut self.vdata)?;
        match &method {
            b"ELRH" => self.unpack_hrle().map_err(|_| DecoderError::InvalidData)?,
            b"ELHD" => self.unpack_dhle().map_err(|_| DecoderError::InvalidData)?,
            _ => return Err(DecoderError::NotImplemented),
        }
        self.cur_frm += 1;
        Ok((0, Frame::VideoRGB16(self.frame.clone())))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    let size = br.read_u32le()?;
    validate!(&tag == b"FELR" && size == 20);
    br.read_u32le()?; // always 1?
    br.read_u32le()?; // always 1?
    let nframes = br.read_u32le()?;
    validate!(nframes > 0);

    let tag = br.read_tag()?;
    let size = br.read_u32le()? as usize;
    validate!(&tag == b"FNIC" && size >= 48);
    br.read_u32le()?; // always 1?
    let nentries = br.read_u32le()? as usize;
    validate!(nentries > 0 && size == 16 + nentries * 32);

    let mut methods = Vec::with_capacity(nentries);
    for _ in 0..nentries {
        let two = br.read_u32le()?;
        validate!(two == 2);
        br.read_skip(24)?; // method name
        let method_tag = br.read_tag()?;
        validate!(!methods.contains(&method_tag));
        methods.push(method_tag);
    }

    let tag = br.read_tag()?;
    let size = br.read_u32le()? as usize;
    validate!(&tag == b"FNIM" && size == 32);
    let tag2 = br.read_tag()?;
    validate!(&tag2 == b"OEDV");
    br.read_u32le()?; // always 1?
    br.read_u32le()?;
    br.read_u32le()?;
    let width = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!(width > 0 && width <= 1024);
    validate!(height > 0 && height <= 768);

    let tag = br.read_tag()?;
    let size = br.read_u32le()? as usize;
    validate!(&tag == b"EMIT" && size == 16);
    br.read_u32le()?; // always 1?
    let time = br.read_u32le()?;

    Ok(Box::new(RLFDecoder {
        fr,
        width, height, time, methods, nframes,
        cur_frm: 0,
        vdata:   Vec::new(),
        frame:   vec![0; width * height],
    }))
}
