use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct SHDecoder {
    fr:         FileReader<BufReader<File>>,
    frame:      Vec<u8>,
    data:       Vec<u8>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    fps:        u32,
    version:    u32,
    cur_frm:    usize,
    frm_info:   Vec<(u32, u32)>,
    arate:      u32,
    abits:      u8,
    channels:   u8,
    adeltas:    [i32; 16],
    compressed: bool,
    pred:       i32,
}

impl SHDecoder {
    fn expand_audio(&mut self) -> DecoderResult<Vec<u8>> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        let mut dst = Vec::new();
        while br.left() > 0 {
            let op = br.read_byte()?;
            let hi_nib = op >> 4;
            let lo_nib = op & 0xF;
            if lo_nib == 0xF {
                if hi_nib < 8 {
                    self.pred = i32::from(br.read_byte()?);
                    dst.push(self.pred as u8);
                } else {
                    let run_len = (usize::from(hi_nib & 7) << 8) | usize::from(br.read_byte()?);
                    let run_val = self.pred.max(0).min(255) as u8;
                    for _ in 0..run_len * 2 {
                        dst.push(run_val);
                    }
                }
            } else {
                self.pred += self.adeltas[usize::from(lo_nib)];
                dst.push(self.pred.max(0).min(255) as u8);

                if hi_nib != 0xF {
                    self.pred += self.adeltas[usize::from(hi_nib)];
                    dst.push(self.pred.max(0).min(255) as u8);
                } else {
                    self.pred = i32::from(br.read_byte()?);
                    dst.push(self.pred as u8);
                }
            }
        }

        Ok(dst)
    }
    fn unpack_frame(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = 0;
        while br.left() > 0 {
            let op = br.read_byte()? as usize;
            match op >> 6 {
                0 => {
                    if op == 0x00 { continue; }
                    let len = op;
                    validate!(pos + len <= self.frame.len());
                    br.read_buf(&mut self.frame[pos..][..len])?;
                    pos += len;
                },
                1 => {
                    let len = op & 0x3F;
                    validate!(pos + len <= self.frame.len());
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                },
                _ => {
                    let skip = op & 0x7F;
                    validate!(pos + skip <= self.frame.len());
                    pos += skip;
                },
            }
        }

        Ok(())
    }
}

impl InputSource for SHDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: self.fps,
            }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo {
                sample_rate: self.arate,
                sample_type: if self.abits > 8 { AudioSample::S16 } else { AudioSample::U8 },
                channels:    self.channels,
            }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frm >= self.frm_info.len() {
            return Err(DecoderError::EOF);
        }

        let mut br = ByteReader::new(&mut self.fr);
        let (ftype, size) = self.frm_info[self.cur_frm];
        let size = size as usize;
        self.cur_frm += 1;

        match ftype {
            0 => {
                self.data.resize(size, 0);
                br.read_buf(&mut self.data)?;
            },
            1 => {
                let audio = if self.compressed {
                        validate!(size <= 16782);
                        self.data.resize(size, 0);
                        br.read_buf(&mut self.data)?;
                        self.expand_audio().map_err(|_| DecoderError::InvalidData)?
                    } else {
                        validate!(size <= 24148);
                        let mut audio = vec![0; size];
                        br.read_buf(&mut audio)?;
                        audio
                    };
                return Ok((1, Frame::AudioU8(audio)));
            },
            2 => {
                validate!(size > self.pal.len());
                br.read_vga_pal(&mut self.pal)?;
                self.data.resize(size - self.pal.len(), 0);
                br.read_buf(&mut self.data)?;
            },
            _ => return Err(DecoderError::InvalidData),
        }
        // TODO: handle JPEG-compressed data as well
        self.unpack_frame().map_err(|err|
                if err == DecoderError::ShortData { DecoderError::InvalidData } else { err })?;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(tag == [b'D', b'V', 0, 0]);

    let nframes = br.read_u32le()? as usize;
    validate!((1..=5000).contains(&nframes));

    let width  = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!((1..=640).contains(&width) && (1..=480).contains(&height));
    let version = br.read_u32le()?;
    if version == 3 {
        println!("JPEG compression not supported");
        return Err(DecoderError::NotImplemented);
    }
    let fps = br.read_u32le()?;
    validate!((1..=30).contains(&fps));

    let arate = br.read_u32le()?;
    let abits = br.read_u16le()?;
    let channels = br.read_u16le()?;
    if arate > 0 {
        if abits != 8 {
            return Err(DecoderError::NotImplemented);
        }
        validate!(channels == 1 || channels == 2);
    }
    br.read_u16le()?; // looped animation
    br.read_u16le()?; // unknown
    let has_pal = br.read_u16le()? != 0;
    let is_aligned = br.read_u16le()? != 0;
    let compressed = br.read_u16le()? != 0;
    br.read_u16le()?; // unknown

    let mut frm_info = Vec::with_capacity(nframes);
    for _ in 0..nframes {
        let ftype = br.read_u32le()?; // 0 - video, 1 - audio, 2 - palette+video
        let size = br.read_u32le()?;
        frm_info.push((ftype, size));
    }

    let mut pal = [0; 768];
    if has_pal {
        br.read_vga_pal(&mut pal)?;
    }

    let mut adeltas = [0; 16];
    if compressed {
        for el in adeltas.iter_mut() {
            *el = br.read_u32le()? as i32;
        }
    }

    if is_aligned {
        let pos = br.tell();
        br.seek(SeekFrom::Start((pos + 0x7FF) & !0x7FF))?;
    }

    Ok(Box::new(SHDecoder {
        fr,
        pal,
        frame: vec![0; width * height],
        data: Vec::with_capacity(65536),
        width, height, fps,
        version,
        frm_info,
        cur_frm: 0,
        arate, compressed,
        abits: abits as u8,
        channels: channels as u8,
        adeltas,
        pred: 0,
    }))
}
