use std::fs::File;
use std::io::BufReader;
use std::collections::VecDeque;
use crate::io::byteio::*;
use crate::input::util::lzss::*;
use super::super::*;

const SND_RATE: u32 = 22050;
const SND_BLOCK: usize = 1270;

struct SpidyAniDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    udata:      Vec<u8>,
    flags:      u16,
    width:      usize,
    height:     usize,
    nframes:    u16,
    cur_frame:  u16,
    aqueue:     VecDeque<Vec<u8>>,
    audio:      bool,
    has_audio:  bool,
}

impl SpidyAniDecoder {
    fn unpack_rle0(&mut self, yoff: usize, height: usize) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = yoff * self.width;
        let end_pos = (yoff + height) * self.width;
        while pos < end_pos {
            let op = br.read_byte()?;
            match op {
                0x00..=0x7F => {
                    self.frame[pos] = op;
                    pos += 1;
                },
                0x80 => {
                    let skip_len = br.read_u16le()? as usize;
                    validate!(skip_len > 0 && pos + skip_len <= end_pos);
                    pos += skip_len;
                },
                0x81..=0xBF => {
                    let skip_len = usize::from(op & 0x3F);
                    validate!(pos + skip_len <= end_pos);
                    pos += skip_len;
                },
                0xC0 => {
                    let run_len = br.read_u16le()? as usize;
                    validate!(run_len > 0 && pos + run_len <= end_pos);
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..run_len].iter_mut() {
                        *el = clr;
                    }
                    pos += run_len;
                },
                0xC1..=0xFF => {
                    let run_len = usize::from(op & 0x3F);
                    validate!(pos + run_len <= end_pos);
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..run_len].iter_mut() {
                        *el = clr;
                    }
                    pos += run_len;
                }
            }
        }

        Ok(())
    }
    fn unpack_rle1(&mut self, yoff: usize, height: usize) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = yoff * self.width;
        let end_pos = (yoff + height) * self.width;
        while pos < end_pos {
            let op = br.read_byte()?;
            match op {
                0x00 => {
                    let copy_len = br.read_u16le()? as usize;
                    validate!(copy_len > 0 && pos + copy_len <= end_pos);
                    br.read_buf(&mut self.frame[pos..][..copy_len])?;
                    pos += copy_len;
                },
                0x01..=0x7F => {
                    let copy_len = usize::from(op);
                    validate!(pos + copy_len <= end_pos);
                    br.read_buf(&mut self.frame[pos..][..copy_len])?;
                    pos += copy_len;
                },
                0x80 => {
                    let skip_len = br.read_u16le()? as usize;
                    validate!(skip_len > 0 && pos + skip_len <= end_pos);
                    pos += skip_len;
                },
                0x81..=0xBF => {
                    let skip_len = usize::from(op & 0x3F);
                    validate!(pos + skip_len <= end_pos);
                    pos += skip_len;
                },
                0xC0 => {
                    let run_len = br.read_u16le()? as usize;
                    validate!(run_len > 0 && pos + run_len <= end_pos);
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..run_len].iter_mut() {
                        *el = clr;
                    }
                    pos += run_len;
                },
                0xC1..=0xFF => {
                    let run_len = usize::from(op & 0x3F);
                    validate!(pos + run_len <= end_pos);
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..run_len].iter_mut() {
                        *el = clr;
                    }
                    pos += run_len;
                }
            }
        }

        Ok(())
    }
}

impl InputSource for SpidyAniDecoder {
    fn get_num_streams(&self) -> usize { if self.has_audio { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: SND_BLOCK as u32,
                    tb_den: SND_RATE,
                }),
            1 if self.has_audio => StreamInfo::Audio(AudioInfo {
                    sample_rate: SND_RATE,
                    sample_type: AudioSample::U8,
                    channels:    1,
                }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        let mut cur_line = 0;
        let use_lzss = (self.flags & 0x20) != 0;
        let use_rle0 = (self.flags & 0x04) == 0;
        loop {
            if self.audio && !self.aqueue.is_empty() {
                let abuf = self.aqueue.pop_front().unwrap();
                return Ok((1, Frame::AudioU8(abuf)));
            }
            if self.cur_frame >= self.nframes {
                return Err(DecoderError::EOF);
            }
            let tag = br.read_tag()?;
            match &tag {
                b"Pal " => {
                    br.read_vga_pal(&mut self.pal)?;
                },
                b"Hang" if self.has_audio => {
                    let mut abuf = vec![0; SND_BLOCK];
                    br.read_buf(&mut abuf)?;
                    // convert to unsigned
                    for el in abuf.iter_mut() {
                        *el ^= 0x80;
                    }
                    self.aqueue.push_back(abuf);
                },
                b"Data" => {
                    let unp_len = if use_lzss {
                            br.read_u16le()? as usize
                        } else { 0 };
                    let mut len = br.read_u16le()? as usize;
                    let width = br.read_u16le()? as usize;
                    let height = br.read_u16le()? as usize;
                    validate!(width == self.width && height > 0 && cur_line + height <= self.height);

                    let packed = use_lzss && len != 0xFFFF;
                    if use_lzss && !packed {
                        len = unp_len;
                    }
                    self.data.resize(len, 0);
                    br.read_buf(&mut self.data)?;
                    if packed {
                        self.udata.resize(unp_len, 0);
                        match lzss_unpack_ext(&self.data, &mut self.udata, 0x00) {
                            Ok(()) => {},
                            Err(DecoderError::ShortData) => {
                                // if not caused by padding then return an error
                                if self.data[self.data.len() - 1] != 0 {
                                    return Err(DecoderError::InvalidData);
                                }
                            },
                            Err(_) => return Err(DecoderError::InvalidData),
                        }
                        std::mem::swap(&mut self.data, &mut self.udata);
                    }
                    if use_rle0 {
                        self.unpack_rle0(cur_line, height).map_err(|_| DecoderError::InvalidData)?;
                    } else {
                        self.unpack_rle1(cur_line, height).map_err(|_| DecoderError::InvalidData)?;
                    }

                    cur_line += height;
                    if cur_line == self.height {
                        self.audio = self.has_audio;
                        self.cur_frame += 1;
                        return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                    }
                    br = ByteReader::new(&mut self.fr);
                },
                _ => {
                    println!("Unknown/unexpected tag @ {:X}", br.tell() - 4);
                    return Err(DecoderError::InvalidData);
                },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut magic = [0; 9];
    br.read_buf(&mut magic)?;
    validate!(&magic == b"SpidyAni2");
    let flags = br.read_u16le()?;
    validate!((flags & 1) == 1);
    let nframes = br.read_u16le()?;
    validate!(nframes > 0);
    let width  = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!((1..=640).contains(&width) && (1..=480).contains(&height));
    br.read_u16le()?; // language

    let tag = br.read_tag()?;
    validate!(&tag == b"Pal ");
    let mut pal = [0; 768];
    br.read_vga_pal(&mut pal)?;

    let tag = br.peek_tag()?;
    let has_audio = &tag == b"Hang";

    Ok(Box::new(SpidyAniDecoder {
        fr,
        pal,
        data: Vec::new(),
        udata: Vec::new(),
        frame: vec![0; width * height],
        width, height,
        flags,
        nframes,
        cur_frame: 0,
        aqueue: VecDeque::with_capacity(4),
        audio: false,
        has_audio,
    }))
}
