use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use crate::io::intcode::*;
use super::super::*;

trait ReadTalismanHuff {
    fn read_huff(&mut self, lengths: &[u8]) -> BitReaderResult<u8>;
}

impl<'a> ReadTalismanHuff for BitReader<'a> {
    fn read_huff(&mut self, lengths: &[u8]) -> BitReaderResult<u8> {
        let prefix = self.read_code(UintCodeType::LimitedZeroes((lengths.len() - 1) as u32))? as usize;
        let base = lengths[..prefix].iter().fold(0u8, |acc, &len| acc + (1 << len));
        let add = self.read(lengths[prefix])? as u8;
        Ok(base.saturating_add(add))
    }
}

#[allow(dead_code)]
struct TalismanDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    frm_no:     u32,
    nframes:    u32,
    frame:      Vec<u8>,
    data:       Vec<u8>,
    unp_buf:    Vec<u8>,
    has_cb:     bool,
    has_i:      bool,
    syms:       Vec<[u8; 256]>,
    cur_lut:    [u8; 256],
    audio:      Vec<u8>,
    asize:      usize,
    do_audio:   bool,
    dump_count: usize
}

impl TalismanDecoder {
    fn unpack_codebook(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        for i in 0..256 {
            let b = br.read_byte()?;
            match b {
                0x00..=0x7F => {
                    validate!(b < 8);
                    let nsyms = br.read_byte()? as usize;
                    br.read_buf(&mut self.syms[i][..nsyms])?;
                    self.cur_lut[i] = b;
                },
                0x80..=0xFE => {
                    validate!((b & 0x7F) < 8);
                    let idx = br.read_byte()? as usize;
                    validate!(idx < i);
                    self.syms[i] = self.syms[idx];
                    self.cur_lut[i] = b & 0x7F;
                },
                0xFF => continue,
            }
        }
        validate!(br.left() == 0);

        Ok(())
    }
    fn unpack_buf(&mut self) -> DecoderResult<()> {
        validate!(self.data.len() >= 9);

        let unp_size = read_u32le(&self.data)? as usize;
        validate!(unp_size > 0);

        for _ in 0..16 {
            self.data.push(0);
        }

        self.unp_buf.clear();
        self.unp_buf.reserve(unp_size);
        let mut last = self.data[4];
        self.unp_buf.push(last);

        let mut br = BitReader::new(&self.data[5..], BitReaderMode::LE16MSB);
        for _ in 1..unp_size {
            let idx = br.read_huff(HUFF_TABLES[self.cur_lut[last as usize] as usize])? as usize;
            let sym = self.syms[usize::from(last)][idx];
            self.unp_buf.push(sym);
            last = sym;
        }

        Ok(())
    }
    fn decode_frame(&mut self) -> DecoderResult<()> {
        let src = &self.unp_buf;
        validate!(src.len() >= 5);
        let clr_offset = read_u32le(src)? as usize;
        validate!(clr_offset <= src.len());

        let mut mr = MemoryReader::new_read(&src[4..][..clr_offset]);
        let mut ops = ByteReader::new(&mut mr);
        let mut mr = MemoryReader::new_read(&src[clr_offset + 4..]);
        let mut clrs = ByteReader::new(&mut mr);

        let mut pos = 0;
        let stride = self.width;
        let mut skip_run = 0;
        for blk_y in (0..self.height).step_by(2) {
            for blk_x in (0..self.width).step_by(2) {
                if skip_run > 0 {
                    skip_run -= 1;
                    continue;
                }

                let op = ops.read_byte()?;
                if op == 0xFE {
                    loop {
                        let val = ops.read_byte()? as usize;
                        skip_run += val;
                        if val != 0xFF {
                            break;
                        }
                    }
                    validate!(skip_run > 0);
                    skip_run -= 1;
                    continue;
                }
                if op == 0xFF {
                    clrs.read_buf(&mut self.frame[pos + blk_x..][..2])?;
                    clrs.read_buf(&mut self.frame[pos + blk_x + stride..][..2])?;
                    continue;
                }
                let mv = if op < 0xF7 { op } else { ops.read_byte()? };

                let src_x = blk_x as isize + isize::from((mv as i8) << 4 >> 4);
                let src_y = blk_y as isize + isize::from((mv as i8) >> 4);
                validate!(src_x >= 0 && src_y >= 0);
                let src_x = src_x as usize;
                let src_y = src_y as usize;
                validate!(src_x + 2 <= self.width && src_y + 2 <= self.height);
                let src_addr = src_x + src_y * stride;
                let mut blk = [self.frame[src_addr],          self.frame[src_addr + 1],
                               self.frame[src_addr + stride], self.frame[src_addr + stride + 1]];

                match op {
                    0xF7..=0xFA => {
                        let pix = clrs.read_byte()?;
                        blk[usize::from(0xFA - op)] = pix;
                    },
                    0xFB => {
                        let blk2 = [blk[1], blk[3], blk[0], blk[2]];
                        blk = blk2;
                    },
                    0xFC => {
                        let blk2 = [blk[3], blk[2], blk[1], blk[0]];
                        blk = blk2;
                    },
                    0xFD => {
                        let blk2 = [blk[2], blk[0], blk[3], blk[1]];
                        blk = blk2;
                    },
                    0xFE => unreachable!(),
                    0xFF => unreachable!(),
                    _ => {},
                }
                self.frame[pos + blk_x..][..2].copy_from_slice(&blk[..2]);
                self.frame[pos + blk_x + stride..][..2].copy_from_slice(&blk[2..]);
            }
            pos += stride * 2;
        }
        Ok(())
    }
}

impl InputSource for TalismanDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 25,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    channels:    1,
                    sample_type: AudioSample::U8,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            if self.do_audio && !self.audio.is_empty() {
                self.do_audio = false;
                if self.audio.len() < self.asize * 2 {
                    let mut ret = Vec::new();
                    std::mem::swap(&mut ret, &mut self.audio);
                    return Ok((1, Frame::AudioU8(ret)));
                } else {
                    let mut ret = vec![0; self.asize];
                    ret.copy_from_slice(&self.audio[..self.asize]);
                    self.audio.drain(..self.asize);
                    return Ok((1, Frame::AudioU8(ret)));
                }
            }
            if self.frm_no >= self.nframes {
                return Err(DecoderError::EOF);
            }
            let chunk_type = br.read_u32le()?;
            let chunk_size = br.read_u32le()? as usize;

            match chunk_type {
                0x1234 => return Err(DecoderError::InvalidData),
                0x4321 => {
                    validate!(chunk_size == 0x10);
                    let _next_offset = br.read_u32le()?;
                    let _zero = br.read_u32le()?;
                    let _buf_size = br.read_u32le()?;
                    let _nframes = br.read_u32le()?;

                    self.has_cb = false;
                    self.has_i = false;
                },
                0x1111 => {
                    self.data.resize(chunk_size, 0);
                    br.read_buf(&mut self.data)?;
                    self.unpack_codebook().map_err(|_| DecoderError::InvalidData)?;
                    self.has_cb = true;
                    br = ByteReader::new(&mut self.fr);
                },
                0x5544 => {
                    validate!(chunk_size == 768);
                    br.read_buf(&mut self.pal)?;
                },
                0x2001 => {
                    validate!(self.has_cb);
                    self.frm_no += 1;
                    self.data.resize(chunk_size, 0);
                    br.read_buf(&mut self.data)?;

                    for el in self.frame.iter_mut() {
                        *el = 0;
                    }
                    self.unpack_buf().map_err(|_| DecoderError::InvalidData)?;
                    self.decode_frame().map_err(|_| DecoderError::InvalidData)?;
                    self.has_i = true;
                    self.do_audio = true;

                    let pal = self.pal;
                    let frame = self.frame.clone();
                    return Ok((0, Frame::VideoPal(frame, pal)));
                },
                0x2110 => {
                    validate!(self.has_cb && self.has_i);
                    self.frm_no += 1;
                    self.data.resize(chunk_size, 0);
                    br.read_buf(&mut self.data)?;

                    self.unpack_buf().map_err(|_| DecoderError::InvalidData)?;
                    self.decode_frame().map_err(|_| DecoderError::InvalidData)?;
                    self.do_audio = true;

                    let pal = self.pal;
                    let frame = self.frame.clone();
                    return Ok((0, Frame::VideoPal(frame, pal)));
                },
                0x2332 | 0x2553 => {
                    self.frm_no += 1;
                    br.read_skip(chunk_size)?;
                    self.do_audio = true;

                    let pal = self.pal;
                    let frame = self.frame.clone();
                    return Ok((0, Frame::VideoPal(frame, pal)));
                },
                0x3456 => {
                    self.data.resize(chunk_size, 0);
                    br.read_buf(&mut self.data)?;
                    self.unpack_buf().map_err(|_| DecoderError::InvalidData)?;
                    std::mem::swap(&mut self.audio, &mut self.unp_buf);
                    self.do_audio = true;
                    br = ByteReader::new(&mut self.fr);
                },
                0xABCD => {
                    br.read_skip(chunk_size)?;
                },
                _ => {
                    br.read_skip(chunk_size)?;
                },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let chunk_type = br.read_u32le()?;
    let chunk_size = br.read_u32le()? as usize;
    validate!(chunk_type == 0x1234 && chunk_size == 0x14);
    let width = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!(width > 0 && width <= 640 && height > 0 && height <= 480);
    br.read_u32le()?; // always 0?
    br.read_u32le()?; // always 300000?
    let nframes = br.read_u32le()?;
    validate!(nframes > 0);

    let mut syms = Vec::with_capacity(256);
    for _ in 0..256 {
        syms.push([0; 256]);
    }

    Ok(Box::new(TalismanDecoder {
        fr,
        pal: [0; 768],
        width, height, nframes,
        frm_no: 0,
        data: Vec::new(),
        frame: vec![0; width * height],
        unp_buf: Vec::new(),
        has_cb: false,
        has_i: false,
        syms,
        cur_lut: [8; 256],
        audio: Vec::new(),
        asize: 880,
        do_audio: false,
        dump_count: 0
    }))
}

const HUFF_TABLES: [&[u8]; 8] = [
    &[ 3, 4, 4, 5, 6, 7 ],
    &[ 4, 4, 5, 5, 5, 5, 5, 6 ],
    &[ 3, 3, 5, 5, 5, 5, 5, 5, 6 ],
    &[ 3, 3, 3, 3, 5, 5, 5, 5, 5, 6 ],
    &[ 0, 3, 3, 4, 5, 5, 5, 5, 5, 5, 6 ],
    &[ 2, 4, 4, 5, 5, 5, 5, 5, 6 ],
    &[ 1, 4, 4, 5, 5, 5, 5, 5, 6 ],
    &[ 1, 3, 4, 5, 5, 5, 5, 5, 5, 6 ],
];
