use std::fs::File;
use std::io::{ErrorKind, Read, Seek};
use crate::io::byteio::*;
use crate::input::util::lzss::lz_copy;
use super::super::*;

struct AnimBufReader {
    file:       File,
    buf:        [u8; 0x1000],
    filepos:    u64,
    pos:        usize,
    fill:       usize,
    demo:       bool,
    eof:        bool,
}

impl AnimBufReader {
    fn new(mut file: File, demo: bool) -> Self {
        file.seek(SeekFrom::Start(0)).unwrap();
        Self {
            file,
            filepos: 0,
            buf:     [0; 0x1000],
            pos:     0,
            fill:    0,
            demo,
            eof:     false,
        }
    }
    fn refill(&mut self) -> std::io::Result<()> {
        if !self.demo {
            self.file.seek(SeekFrom::Current(4))?;
            self.filepos += 4;
        }
        self.pos  = 0;
        self.fill = 0;
        while !self.eof && self.fill < self.buf.len() {
            match self.file.read(&mut self.buf[self.fill..]) {
                Ok(0) => {
                    self.eof = true;
                    break;
                },
                Ok(size) => {
                    self.fill += size;
                },
                Err(_err) => {
                    self.eof = true;
                    break;
                },
            }
        }
        if self.fill > 0 {
            Ok(())
        } else {
            Err(std::io::Error::from(ErrorKind::UnexpectedEof))
        }
    }
}

impl Read for AnimBufReader {
    fn read(&mut self, dst: &mut [u8]) -> std::io::Result<usize> {
        loop {
            if self.fill > 0 {
                let copy_len = dst.len().min(self.fill);
                dst[..copy_len].copy_from_slice(&self.buf[self.pos..][..copy_len]);
                self.pos     += copy_len;
                self.fill    -= copy_len;
                self.filepos += copy_len as u64;
                return Ok(copy_len);
            }
            self.refill()?;
        }
    }
}

impl Seek for AnimBufReader {
    fn seek(&mut self, pos: SeekFrom) -> std::io::Result<u64> {
        match pos {
            SeekFrom::Start(_offs) => unimplemented!(),
            SeekFrom::Current(offs) => {
                if offs >= 0 {
                    let mut to_skip = offs as usize;
                    while to_skip > 0 {
                        if self.fill > 0 {
                            let skip_len = to_skip.min(self.fill);
                            self.pos     += skip_len;
                            self.fill    -= skip_len;
                            to_skip      -= skip_len;
                            self.filepos += skip_len as u64;
                        }
                        self.refill()?;
                    }
                    Ok(self.filepos)
                } else {
                    unimplemented!()
                }
            },
            SeekFrom::End(_offs) => unimplemented!(),
        }
    }
}

struct HybridReader<'a> {
    src:        &'a [u8],
    pos:        usize,
    bitbuf:     u16,
    bits:       u8,
}

impl<'a> HybridReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        let bitbuf = read_u16le(src).unwrap_or_default();
        Self {
            src,
            pos: 2,
            bitbuf,
            bits: 16,
        }
    }
    fn read_byte(&mut self) -> DecoderResult<u8> {
        if self.pos < self.src.len() {
            let ret = self.src[self.pos];
            self.pos += 1;
            Ok(ret)
        } else {
            Err(DecoderError::ShortData)
        }
    }
    fn read_bit(&mut self) -> DecoderResult<bool> {
        let ret = (self.bitbuf & 1) != 0;
        self.bitbuf >>= 1;
        self.bits    -= 1;
        if self.bits == 0 {
            self.bitbuf  = u16::from(self.read_byte()?);
            self.bitbuf |= u16::from(self.read_byte()?) << 8;
            self.bits = 16;
        }
        Ok(ret)
    }
    fn read_bits(&mut self, nbits: u8) -> DecoderResult<u8> {
        let mut ret = 0;
        validate!(nbits < 8);
        for _ in 0..nbits {
            ret <<= 1;
            if self.read_bit()? {
                ret |= 1;
            }
        }
        Ok(ret)
    }
}

struct AnimDecoder {
    fr:      FileReader<AnimBufReader>,
    width:   usize,
    height:  usize,
    pal:     [u8; 768],
    frame:   Vec<u8>,
    vdata:   Vec<u8>,
    ubuf:    Vec<u8>,
}

impl AnimDecoder {
    fn diet_unpack(src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<()> {
        validate!(src.len() > 0x12);
        validate!(&src[6..9] == b"dlz");
        let mut br = HybridReader::new(&src[0x11..]);

        dst.clear();
        loop {
            if br.read_bit()? {
                dst.push(br.read_byte()?);
            } else {
                let bit = br.read_bit()?;
                let mut offs_hi = 0xFF;
                let offs_lo = br.read_byte()?;
                let mut len = 2;
                if !bit {
                    if !br.read_bit()? { // 8-bit offset or escape code
                        if offs_lo == 0xFF {
                            if !br.read_bit()? {
                                return Ok(());
                            } else {
                                continue;
                            }
                        }
                    } else { // 11-bit offset
                        offs_hi = ((offs_hi << 3) | br.read_bits(3)?) - 1;
                    }
                } else { // larger copy lengths
                    offs_hi = (offs_hi << 1) | (br.read_bit()? as u8);
                    if !br.read_bit()? {
                        let mut l2 = 2;
                        for _ in 0..3 {
                            if br.read_bit()? {
                                break;
                            }
                            offs_hi = (offs_hi << 1) | (br.read_bit()? as u8);
                            l2 <<= 1;
                        }
                        offs_hi -= l2;
                    }

                    let mut esc = true;
                    for _ in 0..4 {
                        len += 1;
                        if br.read_bit()? {
                            esc = false;
                            break;
                        }
                    }
                    if esc {
                        if br.read_bit()? {
                            len += 1;
                            if br.read_bit()? {
                                len += 1;
                            }
                        } else if !br.read_bit()? {
                            len = 9 + usize::from(br.read_bits(3)?);
                        } else {
                            len = 17 + usize::from(br.read_byte()?);
                        }
                    }
                }
                let offset = -(((u16::from(offs_hi) << 8) | u16::from(offs_lo)) as i16);
                let pos = dst.len();
                dst.resize(pos + len, 0);
                lz_copy(dst, pos, offset as usize, len);
            }
        }
    }
    fn rle_unpack(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = 0;
        while pos < self.frame.len() {
            let skip = usize::from(br.read_byte()?);
            let op = br.read_byte()?;
            if skip == 0 && op == 0 {
                break;
            }
            pos += skip;
            match op {
                0x00 => {},
                0x01..=0x7F => {
                    let len = usize::from(op);
                    validate!(pos + len <= self.frame.len());
                    br.read_buf(&mut self.frame[pos..][..len])?;
                    pos += len;
                },
                _ => {
                    let len = 256 - usize::from(op);
                    validate!(pos + len <= self.frame.len());
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                },
            }
        }
        Ok(())
    }
}

impl InputSource for AnimDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        let mode    = br.read_byte()?;
        let has_pal = br.read_byte()? != 0;
        let _hdr2   = br.read_byte()?;
        let _hdr3   = br.read_byte()?;
        let width   = br.read_u16le()? as usize;
        let height  = br.read_u16le()? as usize;
        let xoff    = br.read_u16le()? as usize;
        let yoff    = br.read_u16le()? as usize;
        let frm_size = br.read_u32le()? as usize;
        if mode == 0xFF {
            return Err(DecoderError::EOF);
        }
        if width != self.width || height != self.height || xoff != 0 || yoff != 0 {
            return Err(DecoderError::NotImplemented);
        }
        validate!((1..=self.width).contains(&width) && (1..=self.height).contains(&height));
        validate!((1..=1048576).contains(&frm_size));
        if has_pal {
            br.read_buf(&mut self.pal)?;
        }
        self.vdata.resize(frm_size, 0);
        br.read_buf(&mut self.vdata)?;
        if mode == 0 {
            Self::diet_unpack(&self.vdata, &mut self.ubuf)
                .map_err(|_| DecoderError::InvalidData)?;
            validate!(self.ubuf.len() == self.frame.len());
            self.frame.copy_from_slice(&self.ubuf);
        } else {
            self.rle_unpack().map_err(|_| DecoderError::InvalidData)?;
        }
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let mut file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(&mut file);
    let mut br = ByteReader::new(&mut fr);

    let tag = br.peek_tag()?;
    let demo = &tag != b"CHCK";
    if !demo {
        br.read_skip(4)?;
    }
    let _flags = br.read_u32le()?;
    let width   = br.read_u16le()? as usize;
    let height  = br.read_u16le()? as usize;
    validate!((1..=640).contains(&width) && (1..=480).contains(&height));
    br.seek(SeekFrom::Start(4))?;

    Ok(Box::new(AnimDecoder {
        fr: FileReader::new_read(AnimBufReader::new(file, demo)),
        width, height,
        pal: [0; 768],
        frame: vec![0; width * height],
        vdata: Vec::new(),
        ubuf: Vec::new(),
    }))
}
