//! Deflate format (RFC 1951) support.

use crate::input::util::lzss::lz_copy;
use crate::io::bitreader::*;
use crate::io::codebook::*;
use super::*;

/// A list specifying general compression errors.
#[derive(Debug)]
pub enum DecompressError {
    /// Compressed stream header contains errors.
    InvalidHeader,
    /// Compressed stream checksum does not match unpacked contents.
    CRCError,
    /// Compressed stream contains a combination of bits that cannot be decoded.
    InvalidData,
    /// Compressed stream ended prematurely.
    ShortData,
    /// There is more data than what can fit output buffer.
    OutputFull,
    /// Provided function argument is invalid.
    InvalidArgument,
    /// Compressed stream contains features that cannot be decoded.
    Unsupported,
    /// Underlying input reader had a problem.
    IOError,
}

/// A specialised `Result` type for codebook operations.
pub type DecompressResult<T> = Result<T, DecompressError>;

impl From<ByteIOError> for DecompressError {
    fn from(e: ByteIOError) -> Self {
        match e {
            ByteIOError::EOF => DecompressError::ShortData,
            _ => DecompressError::IOError,
        }
    }
}

impl From<BitReaderError> for DecompressError {
    fn from(e: BitReaderError) -> Self {
        match e {
            BitReaderError::BitstreamEnd => DecompressError::ShortData,
            _ => DecompressError::InvalidData,
        }
    }
}

impl From<CodebookError> for DecompressError {
    fn from(_: CodebookError) -> Self {
        DecompressError::InvalidData
    }
}

const NUM_LITERALS: usize = 287;
const NUM_DISTS:    usize = 32;

struct FixedLenCodeReader {}

impl CodebookDescReader<u16> for FixedLenCodeReader {
    fn bits(&mut self, idx: usize) -> u8  {
        if      idx < 144 { 8 }
        else if idx < 256 { 9 }
        else if idx < 280 { 7 }
        else              { 8 }
    }
    #[allow(clippy::identity_op)]
    fn code(&mut self, idx: usize) -> u32 {
        let base = idx as u32;
        let bits = self.bits(idx);
        if      idx < 144 { reverse_bits(base + 0x30, bits) }
        else if idx < 256 { reverse_bits(base + 0x190 - 144, bits) }
        else if idx < 280 { reverse_bits(base + 0x000 - 256, bits) }
        else              { reverse_bits(base + 0xC0 - 280, bits) }
    }
    fn sym (&mut self, idx: usize) -> u16 { idx as u16 }
    fn len(&mut self) -> usize { NUM_LITERALS + 1 }
}

#[derive(Clone,Copy,Default)]
struct BitReaderState {
    pos:            usize,
    bitbuf:         u32,
    bits:           u8,
}

struct CurrentSource<'a> {
    src:            &'a [u8],
    br:             BitReaderState,
}

impl<'a> CurrentSource<'a> {
    fn new(src: &'a [u8], br: BitReaderState) -> Self {
        let mut newsrc = Self { src, br };
        newsrc.br.pos = 0;
        newsrc.refill();
        newsrc
    }
    fn reinit(src: &'a [u8], br: BitReaderState) -> Self {
        let mut newsrc = Self { src, br };
        newsrc.refill();
        newsrc
    }
    fn refill(&mut self) {
        while (self.br.bits <= 24) && (self.br.pos < self.src.len()) {
            self.br.bitbuf |= u32::from(self.src[self.br.pos]) << self.br.bits;
            self.br.bits += 8;
            self.br.pos += 1;
        }
    }
    fn skip_cache(&mut self, nbits: u8) {
        self.br.bitbuf >>= nbits;
        self.br.bits    -= nbits;
    }
    fn read(&mut self, nbits: u8) -> BitReaderResult<u32> {
        if nbits == 0 { return Ok(0); }
        if nbits > 16 { return Err(BitReaderError::TooManyBitsRequested); }
        if self.br.bits < nbits {
            self.refill();
            if self.br.bits < nbits { return Err(BitReaderError::BitstreamEnd); }
        }
        let ret = self.br.bitbuf & ((1 << nbits) - 1);
        self.skip_cache(nbits);
        Ok(ret)
    }
    fn read_bool(&mut self) -> BitReaderResult<bool> {
        if self.br.bits == 0 {
            self.refill();
            if self.br.bits == 0 { return Err(BitReaderError::BitstreamEnd); }
        }
        let ret = (self.br.bitbuf & 1) != 0;
        self.skip_cache(1);
        Ok(ret)
    }
    fn peek(&mut self, nbits: u8) -> u32 {
        if nbits == 0 || nbits > 16 { return 0; }
        if self.br.bits < nbits {
            self.refill();
        }
        self.br.bitbuf & ((1 << nbits) - 1)
    }
    fn skip(&mut self, nbits: u32) -> BitReaderResult<()> {
        if u32::from(self.br.bits) >= nbits {
            self.skip_cache(nbits as u8);
        } else {
            unreachable!();
        }
        Ok(())
    }
    fn skip_bytes(&mut self, nbytes: usize) -> BitReaderResult<()> {
        self.align();
        let cached = usize::from(self.br.bits / 8);
        if nbytes <= cached {
            self.skip((nbytes as u32) * 8)?;
        } else {
            self.skip((cached as u32) * 8)?;
            self.br.bits = 0;
            self.br.bitbuf = 0;
            self.br.pos += nbytes - cached;
            if self.br.pos > self.src.len() {
                return Err(BitReaderError::BitstreamEnd);
            }
            self.refill();
        }
        Ok(())
    }
    fn align(&mut self) {
        let b = self.br.bits & 7;
        if b != 0 {
            self.skip_cache(b);
        }
    }
    fn left(&self) -> isize {
        ((self.src.len() as isize) - (self.br.pos as isize)) * 8 + (self.br.bits as isize)
    }
    fn tell(&self) -> usize {
        self.br.pos - usize::from(self.br.bits / 8)
    }
}

impl<'a, S: Copy> CodebookReader<S> for CurrentSource<'a> {
    fn read_cb(&mut self, cb: &Codebook<S>) -> CodebookResult<S> {
        let mut esc = true;
        let mut idx = 0;
        let mut lut_bits = cb.lut_bits;
        let orig_br = self.br;
        while esc {
            let lut_idx = (self.peek(lut_bits) as usize) + idx;
            if cb.table[lut_idx] == TABLE_FILL_VALUE { return Err(CodebookError::InvalidCode); }
            let bits = cb.table[lut_idx] & 0x7F;
            esc  = (cb.table[lut_idx] & 0x80) != 0;
            idx  = (cb.table[lut_idx] >> 8) as usize;
            let skip_bits = if esc { u32::from(lut_bits) } else { bits };
            if (skip_bits as isize) > self.left() {
                self.br = orig_br;
                self.refill();
                return Err(CodebookError::MemoryError);
            }
            self.skip(skip_bits).unwrap();
            lut_bits = bits as u8;
        }
        Ok(cb.syms[idx])
    }
}

enum InflateState {
    Start,
    BlockStart,
    BlockMode,
    StaticBlockLen,
    StaticBlockInvLen(u32),
    StaticBlockCopy(usize),
    FixedBlock,
    FixedBlockLengthExt(usize, u8),
    FixedBlockDist(usize),
    FixedBlockDistExt(usize, usize, u8),
    FixedBlockCopy(usize, usize),
    FixedBlockLiteral(u8),
    DynBlockHlit,
    DynBlockHdist,
    DynBlockHclen,
    DynLengths(usize),
    DynCodeLengths,
    DynCodeLengthsAdd(usize),
    DynBlock,
    DynBlockLengthExt(usize, u8),
    DynBlockDist(usize),
    DynBlockDistExt(usize, usize, u8),
    DynCopy(usize, usize),
    DynBlockLiteral(u8),
    End,
}

/// The decompressor for deflated streams (RFC 1951).
pub struct Inflate {
    br:             BitReaderState,
    fix_len_cb:     Codebook<u16>,

    buf:            [u8; 65536],
    bpos:           usize,
    output_idx:     usize,
    full_pos:       usize,

    state:          InflateState,
    final_block:    bool,
    hlit:           usize,
    hdist:          usize,
    dyn_len_cb:     Option<Codebook<u32>>,
    dyn_lit_cb:     Option<Codebook<u32>>,
    dyn_dist_cb:    Option<Codebook<u32>>,
    len_lengths:    [u8; 19],
    all_lengths:    [u8; NUM_LITERALS + NUM_DISTS],
    cur_len_idx:    usize,
}

const LENGTH_ADD_BITS: [u8; 29] = [
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1,
    1, 1, 2, 2, 2, 2, 3, 3, 3, 3,
    4, 4, 4, 4, 5, 5, 5, 5, 0
];
const LENGTH_BASE: [u16; 29] = [
     3,   4,   5,   6,   7,   8,   9,  10,  11,  13,
    15,  17,  19,  23,  27,  31,  35,  43,  51,  59,
    67,  83,  99, 115, 131, 163, 195, 227, 258
];
const DIST_ADD_BITS: [u8; 30] = [
    0,  0,  0,  0,  1,  1,  2,  2,  3,  3,
    4,  4,  5,  5,  6,  6,  7,  7,  8,  8,
    9,  9, 10, 10, 11, 11, 12, 12, 13, 13
];
const DIST_BASE: [u16; 30] = [
       1,    2,    3,    4,    5,    7,    9,    13,    17,    25,
      33,   49,   65,   97,  129,  193,  257,   385,   513,   769,
    1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577
];
const LEN_RECODE: [usize; 19] = [
    16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15
];
const REPEAT_BITS: [u8; 3] = [ 2, 3, 7 ];
const REPEAT_BASE: [u8; 3] = [ 3, 3, 11 ];

macro_rules! read_bits {
    ($self: expr, $csrc: expr, $bits: expr) => ({
        if $csrc.left() < $bits as isize {
            $self.br = $csrc.br;
            return Err(DecompressError::ShortData);
        }
        $csrc.read($bits).unwrap()
    })
}

macro_rules! read_cb {
    ($self: expr, $csrc: expr, $cb: expr) => ({
        let ret = $csrc.read_cb($cb);
        if let Err(CodebookError::MemoryError) = ret {
            $self.br = $csrc.br;
            return Err(DecompressError::ShortData);
        }
        match ret {
            Ok(val) => val,
            Err(_)  => {
                $self.state = InflateState::End;
                return Err(DecompressError::InvalidData);
            },
        }
    })
}

impl Inflate {
    /// Creates a new instance of `Inflate` struct.
    pub fn new() -> Self {
        let mut cr = FixedLenCodeReader {};
        let fix_len_cb = Codebook::new(&mut cr, CodebookMode::LSB).unwrap();
        Self {
            br:             BitReaderState::default(),
            fix_len_cb,

            buf:            [0; 65536],
            bpos:           0,
            output_idx:     0,
            full_pos:       0,

            state:          InflateState::Start,
            final_block:    false,
            dyn_len_cb:     None,
            dyn_lit_cb:     None,
            dyn_dist_cb:    None,
            hlit:           0,
            hdist:          0,
            len_lengths:    [0; 19],
            all_lengths:    [0; NUM_LITERALS + NUM_DISTS],
            cur_len_idx:    0,
        }
    }
    fn put_literal(&mut self, val: u8) {
        self.buf[self.bpos] = val;
        self.bpos = (self.bpos + 1) & (self.buf.len() - 1);
        self.full_pos += 1;
    }
    fn lz_copy(&mut self, offset: usize, len: usize, dst: &mut [u8]) -> DecompressResult<()> {
        let mask = self.buf.len() - 1;
        if offset > self.full_pos {
            return Err(DecompressError::InvalidData);
        }
        let cstart = (self.bpos.wrapping_sub(offset)) & mask;
        for (i, out) in dst[..len].iter_mut().enumerate() {
            self.buf[(self.bpos + i) & mask] = self.buf[(cstart + i) & mask];
            *out = self.buf[(cstart + i) & mask];
        }
        self.bpos = (self.bpos + len) & mask;
        self.full_pos += len;
        Ok(())
    }
    /// Sets custom history for decoding an update for already decoded data.
    pub fn set_dict(&mut self, dict: &[u8]) {
        let len = dict.len().min(self.buf.len());
        let start = dict.len() - len;
        self.buf[..len].copy_from_slice(&dict[start..]);
        self.bpos = len;
        self.full_pos = len;
    }
    /// Reports whether decoder has finished decoding the input.
    pub fn is_finished(&self) -> bool {
        matches!(self.state, InflateState::End)
    }
    /// Reports the current amount of bytes output into the destination buffer after the last run.
    pub fn get_current_output_size(&self) -> usize { self.output_idx }
    /// Reports the total amount of bytes decoded so far.
    pub fn get_total_output_size(&self) -> usize { self.bpos }
    /// Tries to decompress input data and write it to the output buffer.
    ///
    /// Since the decompressor can work with arbitrary input and output chunks its return value may have several meanings:
    /// * `Ok(len)` means the stream has been fully decoded and then number of bytes output into the destination buffer is returned.
    /// * [`DecompressError::ShortData`] means the input stream has been fully read but more data is needed.
    /// * [`DecompressError::OutputFull`] means the output buffer is full and should be flushed. Then decoding should continue on the same input block with `continue_block` parameter set to `true`.
    ///
    /// [`DecompressError::ShortData`]: ../enum.DecompressError.html#variant.ShortData
    /// [`DecompressError::OutputFull`]: ../enum.DecompressError.html#variant.OutputFull
    pub fn decompress_data(&mut self, src: &[u8], dst: &mut Vec<u8>, continue_block: bool) -> DecompressResult<usize> {
        self.decompress_data_internal(src, dst, continue_block, false)
    }
    /// Tries to decompress whole input chunk to the output buffer.
    pub fn decompress_block(&mut self, src: &[u8], dst: &mut Vec<u8>) -> DecompressResult<usize> {
        self.decompress_data_internal(src, dst, false, true)
    }
    #[allow(clippy::comparison_chain)]
    fn decompress_data_internal(&mut self, src: &[u8], dst: &mut Vec<u8>, continue_block: bool, do_one_block: bool) -> DecompressResult<usize> {
        if src.is_empty() {
            return Err(DecompressError::InvalidArgument);
        }
        let mut csrc = if !continue_block {
                CurrentSource::new(src, self.br)
            } else {
                CurrentSource::reinit(src, self.br)
            };
        self.output_idx = dst.len();
        // check for zlib stream header
        if let (&InflateState::Start, true) = (&self.state, src.len() > 2) {
            let cm    = src[0] & 0xF;
            let cinfo = src[0] >> 4;
            let hdr   = (u16::from(src[0]) << 8) | u16::from(src[1]);
            if cm == 8 && cinfo <= 7 && (hdr % 31) == 0 {
                csrc.skip(16).unwrap();
            }
        }
        'main: loop {
            match self.state {
                InflateState::Start | InflateState::BlockStart => {
                    if csrc.left() == 0 {
                        if do_one_block {
                            return Ok(self.output_idx);
                        }
                        self.br = csrc.br;
                        return Err(DecompressError::ShortData);
                    }
                    self.final_block = csrc.read_bool().unwrap();
                    self.state = InflateState::BlockMode;
                },
                InflateState::BlockMode => {
                    let bmode = read_bits!(self, csrc, 2);
                    match bmode {
                        0 => {
                            csrc.align();
                            self.state = InflateState::StaticBlockLen;
                        },
                        1 => { self.state = InflateState::FixedBlock; },
                        2 => { self.state = InflateState::DynBlockHlit; },
                        _ => {
                            self.state = InflateState::End;
                            return Err(DecompressError::InvalidHeader);
                        },
                    };
                },
                InflateState::StaticBlockLen => {
                    let len = read_bits!(self, csrc, 16);
                    self.state = InflateState::StaticBlockInvLen(len);
                },
                InflateState::StaticBlockInvLen(len) => {
                    let inv_len = read_bits!(self, csrc, 16);
                    if (len ^ inv_len) != 0xFFFF {
                        self.state = InflateState::End;
                        return Err(DecompressError::InvalidHeader);
                    }
                    self.state = InflateState::StaticBlockCopy(len as usize);
                },
                InflateState::StaticBlockCopy(len) => {
                    for i in 0..len {
                        if csrc.left() < 8 {
                            self.br = csrc.br;
                            self.state = InflateState::StaticBlockCopy(len - i);
                            return Err(DecompressError::ShortData);
                        }
                        let val = csrc.read(8).unwrap() as u8;
                        self.put_literal(val);
                    }
                    self.state = InflateState::BlockStart;
                }
                InflateState::FixedBlock => {
                    let val = read_cb!(self, csrc, &self.fix_len_cb);
                    if val < 256 {
                        self.put_literal(val as u8);
                        dst.push(val as u8);
                        self.output_idx += 1;
                    } else if val == 256 {
                        if self.final_block {
                            self.state = InflateState::End;
                            return Ok(self.output_idx);
                        } else {
                            self.state = InflateState::BlockStart;
                        }
                    } else {
                        let len_idx = (val - 257) as usize;
                        if len_idx >= LENGTH_BASE.len() {
                            self.state = InflateState::End;
                            return Err(DecompressError::InvalidData);
                        }
                        let len_bits = LENGTH_ADD_BITS[len_idx];
                        let add_base = LENGTH_BASE[len_idx] as usize;
                        if len_bits > 0 {
                            self.state = InflateState::FixedBlockLengthExt(add_base, len_bits);
                        } else {
                            self.state = InflateState::FixedBlockDist(add_base);
                        }
                    }
                },
                InflateState::FixedBlockLiteral(sym) => {
                    self.put_literal(sym);
                    dst.push(sym);
                    self.output_idx += 1;
                    self.state = InflateState::FixedBlock;
                },
                InflateState::FixedBlockLengthExt(base, bits) => {
                    let add = read_bits!(self, csrc, bits) as usize;
                    self.state = InflateState::FixedBlockDist(base + add);
                },
                InflateState::FixedBlockDist(length) => {
                    let dist_idx = reverse_bits(read_bits!(self, csrc, 5), 5) as usize;
                    if dist_idx >= DIST_BASE.len() {
                        self.state = InflateState::End;
                        return Err(DecompressError::InvalidData);
                    }
                    let dist_bits = DIST_ADD_BITS[dist_idx];
                    let dist_base = DIST_BASE[dist_idx] as usize;
                    if dist_bits == 0 {
                        self.state = InflateState::FixedBlockCopy(length, dist_base);
                    } else {
                        self.state = InflateState::FixedBlockDistExt(length, dist_base, dist_bits);
                    }
                },
                InflateState::FixedBlockDistExt(length, base, bits) => {
                    let add = read_bits!(self, csrc, bits) as usize;
                    self.state = InflateState::FixedBlockCopy(length, base + add);
                },
                InflateState::FixedBlockCopy(length, dist) => {
                    dst.resize(self.output_idx + length, 0);
                    let ret = self.lz_copy(dist, length, &mut dst[self.output_idx..]);
                    if ret.is_err() {
                        self.state = InflateState::End;
                        return Err(DecompressError::InvalidData);
                    }
                    self.output_idx += length;
                    self.state = InflateState::FixedBlock;
                }
                InflateState::DynBlockHlit => {
                    self.hlit = (read_bits!(self, csrc, 5) as usize) + 257;
                    if self.hlit >= 287 {
                        self.state = InflateState::End;
                        return Err(DecompressError::InvalidHeader);
                    }
                    self.state = InflateState::DynBlockHdist;
                }
                InflateState::DynBlockHdist => {
                    self.hdist = (read_bits!(self, csrc, 5) as usize) + 1;
                    self.state = InflateState::DynBlockHclen;
                },
                InflateState::DynBlockHclen => {
                    let hclen = (read_bits!(self, csrc, 4) as usize) + 4;
                    self.cur_len_idx = 0;
                    self.len_lengths = [0; 19];
                    self.all_lengths = [0; NUM_LITERALS + NUM_DISTS];
                    self.state = InflateState::DynLengths(hclen);
                },
                InflateState::DynLengths(len) => {
                    for i in 0..len {
                        if csrc.left() < 3 {
                            self.br = csrc.br;
                            self.state = InflateState::DynLengths(len - i);
                            return Err(DecompressError::ShortData);
                        }
                        self.len_lengths[LEN_RECODE[self.cur_len_idx]] = csrc.read(3).unwrap() as u8;
                        self.cur_len_idx += 1;
                    }
                    let mut len_codes = [ShortCodebookDesc { code: 0, bits: 0 }; 19];
                    lengths_to_codes(&self.len_lengths, &mut len_codes)?;
                    let mut cr = ShortCodebookDescReader::new(&len_codes);
                    let ret = Codebook::new(&mut cr, CodebookMode::LSB);
                    if ret.is_err() {
                        self.state = InflateState::End;
                        return Err(DecompressError::InvalidHeader);
                    }
                    self.dyn_len_cb = Some(ret.unwrap());
                    self.cur_len_idx = 0;
                    self.state = InflateState::DynCodeLengths;
                },
                InflateState::DynCodeLengths => {
                    if let Some(ref len_cb) = self.dyn_len_cb {
                        while self.cur_len_idx < self.hlit + self.hdist {
                            let ret = csrc.read_cb(len_cb);
                            let val = match ret {
                                    Ok(val) => val,
                                    Err(CodebookError::MemoryError) => {
                                        self.br = csrc.br;
                                        return Err(DecompressError::ShortData);
                                    },
                                    Err(_) => {
                                        self.state = InflateState::End;
                                        return Err(DecompressError::InvalidHeader);
                                    },
                                };
                            if val < 16 {
                                self.all_lengths[self.cur_len_idx] = val as u8;
                                self.cur_len_idx += 1;
                            } else {
                                let idx = (val as usize) - 16;
                                if idx > 2 {
                                    self.state = InflateState::End;
                                    return Err(DecompressError::InvalidHeader);
                                }
                                self.state = InflateState::DynCodeLengthsAdd(idx);
                                continue 'main;
                            }
                        }
                        let (lit_lengths, dist_lengths) = self.all_lengths.split_at(self.hlit);

                        let mut lit_codes = [ShortCodebookDesc { code: 0, bits: 0 }; NUM_LITERALS];
                        lengths_to_codes(lit_lengths, &mut lit_codes)?;
                        let mut cr = ShortCodebookDescReader::new(&lit_codes);
                        let ret = Codebook::new(&mut cr, CodebookMode::LSB);
                        if ret.is_err() { return Err(DecompressError::InvalidHeader); }
                        self.dyn_lit_cb = Some(ret.unwrap());

                        let mut dist_codes = [ShortCodebookDesc { code: 0, bits: 0 }; NUM_DISTS];
                        lengths_to_codes(&dist_lengths[..self.hdist], &mut dist_codes)?;
                        let mut cr = ShortCodebookDescReader::new(&dist_codes);
                        let ret = Codebook::new(&mut cr, CodebookMode::LSB);
                        if ret.is_err() { return Err(DecompressError::InvalidHeader); }
                        self.dyn_dist_cb = Some(ret.unwrap());

                        self.state = InflateState::DynBlock;
                    } else {
                        unreachable!();
                    }
                },
                InflateState::DynCodeLengthsAdd(mode) => {
                    let base = REPEAT_BASE[mode] as usize;
                    let bits = REPEAT_BITS[mode];
                    let len = base + read_bits!(self, csrc, bits) as usize;
                    if self.cur_len_idx + len > self.hlit + self.hdist {
                        self.state = InflateState::End;
                        return Err(DecompressError::InvalidHeader);
                    }
                    let rpt = if mode == 0 {
                            if self.cur_len_idx == 0 {
                                self.state = InflateState::End;
                                return Err(DecompressError::InvalidHeader);
                            }
                            self.all_lengths[self.cur_len_idx - 1]
                        } else {
                            0
                        };
                    for _ in 0..len {
                        self.all_lengths[self.cur_len_idx] = rpt;
                        self.cur_len_idx += 1;
                    }
                    self.state = InflateState::DynCodeLengths;
                },
                InflateState::DynBlock => {
                    if let Some(ref lit_cb) = self.dyn_lit_cb {
                        let val = read_cb!(self, csrc, lit_cb);
                        if val < 256 {
                            self.put_literal(val as u8);
                            dst.push(val as u8);
                            self.output_idx += 1;
                        } else if val == 256 {
                            if self.final_block {
                                self.state = InflateState::End;
                                return Ok(self.output_idx);
                            } else {
                                self.state = InflateState::BlockStart;
                            }
                        } else {
                            let len_idx = (val - 257) as usize;
                            if len_idx >= LENGTH_BASE.len() {
                                self.state = InflateState::End;
                                return Err(DecompressError::InvalidData);
                            }
                            let len_bits = LENGTH_ADD_BITS[len_idx];
                            let add_base = LENGTH_BASE[len_idx] as usize;
                            if len_bits > 0 {
                                self.state = InflateState::DynBlockLengthExt(add_base, len_bits);
                            } else {
                                self.state = InflateState::DynBlockDist(add_base);
                            }
                        }
                    } else {
                        unreachable!();
                    }
                },
                InflateState::DynBlockLiteral(sym) => {
                    self.put_literal(sym);
                    dst.push(sym);
                    self.output_idx += 1;
                    self.state = InflateState::DynBlock;
                },
                InflateState::DynBlockLengthExt(base, bits) => {
                    let add = read_bits!(self, csrc, bits) as usize;
                    self.state = InflateState::DynBlockDist(base + add);
                },
                InflateState::DynBlockDist(length) => {
                    if let Some(ref dist_cb) = self.dyn_dist_cb {
                        let dist_idx = read_cb!(self, csrc, dist_cb) as usize;
                        if dist_idx >= DIST_BASE.len() {
                            self.state = InflateState::End;
                            return Err(DecompressError::InvalidData);
                        }
                        let dist_bits = DIST_ADD_BITS[dist_idx];
                        let dist_base = DIST_BASE[dist_idx] as usize;
                        if dist_bits == 0 {
                            self.state = InflateState::DynCopy(length, dist_base);
                        } else {
                            self.state = InflateState::DynBlockDistExt(length, dist_base, dist_bits);
                        }
                    } else {
                        unreachable!();
                    }
                },
                InflateState::DynBlockDistExt(length, base, bits) => {
                    let add = read_bits!(self, csrc, bits) as usize;
                    self.state = InflateState::DynCopy(length, base + add);
                },
                InflateState::DynCopy(length, dist) => {
                    dst.resize(self.output_idx + length, 0);
                    let ret = self.lz_copy(dist, length, &mut dst[self.output_idx..]);
                    if ret.is_err() {
                        self.state = InflateState::End;
                        return Err(DecompressError::InvalidData);
                    }
                    self.output_idx += length;
                    self.state = InflateState::DynBlock;
                }
                InflateState::End => {
                    return Ok(0);
                },
            }
        }
    }
    /// Resets decoder state.
    pub fn reset(&mut self) {
        self.bpos = 0;
        self.output_idx = 0;
        self.full_pos = 0;
        self.state = InflateState::Start;
    }

    #[allow(clippy::comparison_chain)]
    /// Decompresses input data into growable output.
    pub fn uncompress(src: &[u8], dst: &mut Vec<u8>) -> DecompressResult<()> {
        let mut csrc = CurrentSource::new(src, BitReaderState::default());
        if src.len() > 2 {
            let cm    = src[0] & 0xF;
            let cinfo = src[0] >> 4;
            let hdr   = (u16::from(src[0]) << 8) | u16::from(src[1]);
            if cm == 8 && cinfo <= 7 && (hdr % 31) == 0 {
                csrc.skip(16).unwrap();
            }
        }

        let mut fix_len_cb = None;

        let mut final_block = false;
        while !final_block {
            final_block = csrc.read_bool()?;

            let bmode = csrc.read(2)?;
            match bmode {
                0 => {
                                  csrc.align();
                    let len     = csrc.read(16)? as usize;
                    let inv_len = csrc.read(16)? as usize;
                    if (len ^ inv_len) != 0xFFFF {
                        return Err(DecompressError::InvalidHeader);
                    }
                    let src_pos = csrc.tell();
                    if src_pos + len > src.len() {
                        return Err(DecompressError::ShortData);
                    }
                    dst.extend_from_slice(&src[src_pos..][..len]);
                                  csrc.skip_bytes(len)?;
                },
                1 => {
                    if fix_len_cb.is_none() {
                        let mut cr = FixedLenCodeReader {};
                        fix_len_cb = Some(Codebook::new(&mut cr, CodebookMode::LSB).unwrap());
                    }
                    if let Some(ref len_cb) = &fix_len_cb {
                        loop {
                            let val = csrc.read_cb(len_cb)?;
                            if val < 256 {
                                dst.push(val as u8);
                            } else if val == 256 {
                                break;
                            } else {
                                let len_idx = (val - 257) as usize;
                                if len_idx >= LENGTH_BASE.len() {
                                    return Err(DecompressError::InvalidData);
                                }
                                let len_bits = LENGTH_ADD_BITS[len_idx];
                                let mut length = LENGTH_BASE[len_idx] as usize;
                                if len_bits > 0 {
                                    length += csrc.read(len_bits)? as usize;
                                }
                                let dist_idx = reverse_bits(csrc.read(5)?, 5) as usize;
                                if dist_idx >= DIST_BASE.len() {
                                    return Err(DecompressError::InvalidData);
                                }
                                let dist_bits = DIST_ADD_BITS[dist_idx];
                                let mut dist = DIST_BASE[dist_idx] as usize;
                                if dist_bits > 0 {
                                    dist += csrc.read(dist_bits)? as usize;
                                }

                                if dist > dst.len() {
                                    return Err(DecompressError::InvalidData);
                                }
                                let dst_idx = dst.len();
                                dst.resize(dst_idx + length, 0);
                                lz_copy(dst, dst_idx, dist, length);
                            }
                        }
                    } else {
                        unreachable!();
                    }
                },
                2 => {
                    let hlit = csrc.read(5)? as usize + 257;
                    if hlit >= 287 {
                        return Err(DecompressError::InvalidHeader);
                    }
                    let hdist = csrc.read(5)? as usize + 1;
                    let hclen = csrc.read(4)? as usize + 4;
                    let mut len_lengths = [0; 19];
                    let mut all_lengths = [0; NUM_LITERALS + NUM_DISTS];

                    for cur_len_idx in 0..hclen {
                        len_lengths[LEN_RECODE[cur_len_idx]] = csrc.read(3)? as u8;
                    }
                    let mut len_codes = [ShortCodebookDesc { code: 0, bits: 0 }; 19];
                    lengths_to_codes(&len_lengths, &mut len_codes)?;
                    let mut cr = ShortCodebookDescReader::new(&len_codes);
                    let ret = Codebook::new(&mut cr, CodebookMode::LSB);
                    if ret.is_err() {
                        return Err(DecompressError::InvalidHeader);
                    }
                    let dyn_len_cb = ret.unwrap();

                    let mut cur_len_idx = 0;
                    while cur_len_idx < hlit + hdist {
                        let val = csrc.read_cb(&dyn_len_cb)?;
                        if val < 16 {
                            all_lengths[cur_len_idx] = val as u8;
                            cur_len_idx += 1;
                        } else {
                            let mode = (val as usize) - 16;
                            if mode > 2 {
                                return Err(DecompressError::InvalidHeader);
                            }
                            let base = REPEAT_BASE[mode] as usize;
                            let bits = REPEAT_BITS[mode];
                            let len = base + (csrc.read(bits)? as usize);
                            if cur_len_idx + len > hlit + hdist {
                                return Err(DecompressError::InvalidHeader);
                            }
                            let rpt = if mode == 0 {
                                    if cur_len_idx == 0 {
                                        return Err(DecompressError::InvalidHeader);
                                    }
                                    all_lengths[cur_len_idx - 1]
                                } else {
                                    0
                                };
                            for _ in 0..len {
                                all_lengths[cur_len_idx] = rpt;
                                cur_len_idx += 1;
                            }
                        }
                    }
                    let (lit_lengths, dist_lengths) = all_lengths.split_at(hlit);

                    let mut lit_codes = [ShortCodebookDesc { code: 0, bits: 0 }; NUM_LITERALS];
                    lengths_to_codes(lit_lengths, &mut lit_codes)?;
                    let mut cr = ShortCodebookDescReader::new(&lit_codes);
                    let ret = Codebook::new(&mut cr, CodebookMode::LSB);
                    if ret.is_err() { return Err(DecompressError::InvalidHeader); }
                    let dyn_lit_cb = ret.unwrap();

                    let mut dist_codes = [ShortCodebookDesc { code: 0, bits: 0 }; NUM_DISTS];
                    lengths_to_codes(&dist_lengths[..hdist], &mut dist_codes)?;
                    let mut cr = ShortCodebookDescReader::new(&dist_codes);
                    let ret = Codebook::new(&mut cr, CodebookMode::LSB);
                    if ret.is_err() { return Err(DecompressError::InvalidHeader); }
                    let dyn_dist_cb = ret.unwrap();

                    loop {
                        let val = csrc.read_cb(&dyn_lit_cb)?;
                        if val < 256 {
                            dst.push(val as u8);
                        } else if val == 256 {
                            break;
                        } else {
                            let len_idx = (val - 257) as usize;
                            if len_idx >= LENGTH_BASE.len() {
                                return Err(DecompressError::InvalidData);
                            }
                            let len_bits = LENGTH_ADD_BITS[len_idx];
                            let mut length = LENGTH_BASE[len_idx] as usize;
                            if len_bits > 0 {
                                length += csrc.read(len_bits)? as usize;
                            }

                            let dist_idx = csrc.read_cb(&dyn_dist_cb)? as usize;
                            if dist_idx >= DIST_BASE.len() {
                                return Err(DecompressError::InvalidData);
                            }
                            let dist_bits = DIST_ADD_BITS[dist_idx];
                            let mut dist = DIST_BASE[dist_idx] as usize;
                            if dist_bits > 0 {
                                dist += csrc.read(dist_bits)? as usize;
                            }

                            if dist > dst.len() {
                                return Err(DecompressError::InvalidData);
                            }
                            let dst_idx = dst.len();
                            dst.resize(dst_idx + length, 0);
                            lz_copy(dst, dst_idx, dist, length);
                        }
                    }
                },
                _ => return Err(DecompressError::InvalidHeader),
            };
        }
        Ok(())
    }
}

impl Default for Inflate {
    fn default() -> Self {
        Self::new()
    }
}

fn lengths_to_codes(lens: &[u8], codes: &mut [ShortCodebookDesc]) -> DecompressResult<()> {
    let mut bits = [0u32; 32];
    let mut pfx  = [0u32; 33];
    for len in lens.iter() {
        let len = *len as usize;
        if len >= bits.len() {
            return Err(DecompressError::InvalidHeader);
        }
        bits[len] += 1;
    }
    bits[0] = 0;
    let mut code = 0;
    for i in 0..bits.len() {
        code = (code + bits[i]) << 1;
        pfx[i + 1] = code;
    }

    for (len, codes) in lens.iter().zip(codes.iter_mut()) {
        let len = *len as usize;
        if len != 0 {
            let bits = len as u8;
            *codes = ShortCodebookDesc { code: reverse_bits(pfx[len], bits), bits };
            pfx[len] += 1;
        } else {
            *codes = ShortCodebookDesc { code: 0, bits: 0 };
        }
    }

    Ok(())
}
