//! Output writers and their interface definition

use crate::io::byteio::ByteIOError;
use crate::input::metadata::*;

mod aviwrite;
mod imgseq;
mod null;
mod wavwrite;

/// A list of common errors appearing during input data handling
#[derive(Clone,Debug,PartialEq)]
pub enum EncoderError {
    /// Provided format was not found in the list
    FormatNotFound(String),
    /// Output file could not be created
    InvalidFilename(String),
    /// Input stream is not supported (e.g. audio for image sequence writer)
    Ignored,
    /// Write error
    WriteError,
    /// Format is not supported by the writer
    Unsupported,
    /// Input frame is in an unexpected format
    InvalidData,
    /// Trying to write more data than the output writer supports
    ContainerFull,
    /// No data was sent to output
    EmptyOutput
}

impl std::fmt::Display for EncoderError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> Result<(), std::fmt::Error> {
        match self {
            EncoderError::FormatNotFound(ref fmt) => write!(f, "no plugin found to handle the output format '{fmt}'"),
            EncoderError::InvalidFilename(ref name) => write!(f, "cannot create output file '{name}'"),
            EncoderError::Ignored => write!(f, "ignored stream"),
            EncoderError::WriteError => write!(f, "write error"),
            EncoderError::Unsupported => write!(f, "unsupported format"),
            EncoderError::InvalidData => write!(f, "invalid input format"),
            EncoderError::ContainerFull => write!(f, "writer does not accept more data"),
            EncoderError::EmptyOutput => write!(f, "nothing was sent to output"),
        }
    }
}

/// A specialised `Result` type for encoding operations.
pub type EncoderResult<T> = Result<T, EncoderError>;

impl From<ByteIOError> for EncoderError {
    fn from(_: ByteIOError) -> Self { EncoderError::WriteError }
}

/// Common interface for output writer
pub trait OutputWriter {
    /// Adds another stream to the output
    fn add_stream(&mut self, stream_no: usize, stream_info: StreamInfo) -> EncoderResult<()>;
    /// Finalises the header and prepares for writing data
    fn finish_header(&mut self) -> EncoderResult<()>;
    /// Writes a frame to output
    fn write(&mut self, stream_no: usize, frame: Frame) -> EncoderResult<()>;
    /// Finalises the writing
    fn finish(&mut self) -> EncoderResult<()>;
}

/// Registry entry for output plugin
pub struct OutputPlugin {
    /// Short format name e.g. 'imgseq'
    pub name:       &'static str,
    /// Long format name e.g. 'Still image sequence'
    pub long_name:  &'static str,
    /// Function used to attempt and create input plugin for the provided name
    pub create:     fn (name: &str) -> EncoderResult<Box<dyn OutputWriter>>,
}

/// The list of all compiled output plugins
pub const OUTPUT_PLUGINS: &[OutputPlugin] = &[
    OutputPlugin { name: "avi", long_name: "output to AVI format", create: aviwrite::create },
    OutputPlugin { name: "imgseq", long_name: "image sequence writer", create: imgseq::create },
    OutputPlugin { name: "null", long_name: "null output", create: null::create },
    OutputPlugin { name: "wav", long_name: "RIFF Waveform output", create: wavwrite::create },
];

/// Attempts to create an output writer for the provided format and name
pub fn open_output(format: &str, name: &str) -> EncoderResult<Box<dyn OutputWriter>> {
    for plugin in OUTPUT_PLUGINS.iter() {
        if plugin.name == format {
            return (plugin.create)(name);
        }
    }
    Err(EncoderError::FormatNotFound(format.to_owned()))
}

/// Attempts to create an output writer for the provided name
pub fn open_output_auto(name: &str) -> EncoderResult<Box<dyn OutputWriter>> {
    if name == "null" || name == "/dev/null" {
        return open_output("null", "");
    }
    const EXTENSION_MAP: &[(&str, &str)] = &[
        ("avi", ".avi"),
        ("avi", ".AVI"),
        ("imgseq", ".bmp"),
        ("imgseq", ".BMP"),
        ("imgseq", ".ppm"),
        ("wav", ".wav"),
        ("wav", ".WAV"),
    ];
    for (format, extension) in EXTENSION_MAP.iter() {
        if name.ends_with(extension) {
            return open_output(format, name);
        }
    }
    Err(EncoderError::FormatNotFound("autodetect".to_owned()))
}
