use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;

struct APArchive {
    fr:         FileReader<BufReader<File>>,
    offsets:    Vec<u32>,
    fileno:     usize,
}

impl ArchiveSource for APArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.fileno + 2 > self.offsets.len() {
            return Err(DecoderError::EOF);
        }
        let mut name = format!("{:06}", self.fileno);
        let mut hdr = [0; 16];
        // quick and dirty content detection
        let ext = if self.fr.peek_buf(&mut hdr).is_ok() {
                match &hdr[..4] {
                    b"RIFF" if &hdr[8..12] == b"WAVE" => ".wav",
                    b"ACC\x01" => ".acc",
                    b"DBE\x01" => ".dbe",
                    b"H2O\x01" => ".h2o",
                    _ if &hdr[4..8] == b".PTF" => ".ptf",
                    _ if self.offsets[self.fileno + 1] - self.offsets[self.fileno] == 768 => ".pal",
                    _ => "",
                }
            } else { "" };
        name += ext;
        Ok(name)
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let size = (self.offsets[self.fileno + 1] - self.offsets[self.fileno]) as usize;
        self.fileno += 1;
        copy_data(&mut self.fr, dst, size)
    }
    fn set_no_convert(&mut self) {}
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let num_files = usize::from(fr.read_u16le()?);
    validate!(num_files > 1);
    let mut offsets = Vec::with_capacity(num_files);
    let mut prev_offset = num_files as u32 * 4 + 1;
    for _ in 0..num_files {
        let offset = fr.read_u32le()?;
        validate!(offset > prev_offset);
        offsets.push(offset);
        prev_offset = offset;
    }
    fr.seek(std::io::SeekFrom::Start(offsets[0].into()))?;

    Ok(Box::new(APArchive {
        fr, offsets,
        fileno: 0,
    }))
}
