use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;

struct GOBArchive {
    fr:         FileReader<BufReader<File>>,
    offsets:    Vec<u32>,
    sizes:      Vec<u32>,
    names:      Vec<String>,
    fileno:     usize,
}

impl ArchiveSource for GOBArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.fileno >= self.offsets.len() {
            return Err(DecoderError::EOF);
        }
        Ok(self.names[self.fileno].clone())
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let offset = self.offsets[self.fileno];
        let size = self.sizes[self.fileno];
        self.fileno += 1;
        self.fr.seek(SeekFrom::Start(offset.into()))?;
        copy_data(&mut self.fr, dst, size as usize)
    }
    fn set_no_convert(&mut self) {}
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    fr.seek(SeekFrom::End(-4))?;
    let num_files = fr.read_u32le()? as usize;
    validate!(num_files > 0);
    fr.seek(SeekFrom::End(-4 - (num_files as i64) * 20))?;

    let mut name = [0; 12];

    let mut names = Vec::with_capacity(num_files);
    let mut offsets = Vec::with_capacity(num_files);
    let mut sizes = Vec::with_capacity(num_files);

    for _ in 0..num_files {
        fr.read_buf(&mut name)?;
        let offset = fr.read_u32le()?;
        let size = fr.read_u32le()?;
        let mut sname = String::with_capacity(12);
        for &c in name.iter() {
            match c {
                0 => break,
                0x20..=0x7E => {
                    sname.push(c as char);
                }
                _ => return Err(DecoderError::InvalidData),
            }
        }
        validate!(!sname.is_empty());
        offsets.push(offset);
        sizes.push(size);
        names.push(sname);
    }

    Ok(Box::new(GOBArchive {
        fr, offsets, sizes, names,
        fileno: 0,
    }))
}
