use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;
use crate::input::util::rnc::*;

struct Entry {
    name:   String,
    offset: u32,
    size:   u32,
    dsize:  u32,
}

struct CEArchive {
    fr:         FileReader<BufReader<File>>,
    entries:    Vec<Entry>,
    entry:      usize,
    unpack:     bool,
    no_convert: bool,
}

impl ArchiveSource for CEArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.entry < self.entries.len() {
            Ok(self.entries[self.entry].name.clone())
        } else {
            Err(DecoderError::EOF)
        }
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let entry = &self.entries[self.entry];
        self.entry += 1;

        self.fr.seek(SeekFrom::Start(entry.offset.into()))?;
        if self.no_convert || entry.dsize == entry.size {
            copy_data(&mut self.fr, dst, entry.size as usize)
        } else {
            let mut src = vec![0; entry.size as usize];
            self.fr.read_buf(&mut src)?;
            let mut dbuf = Vec::with_capacity(entry.dsize as usize);
            rnc_uncompress(&src, &mut dbuf)
                .map_err(|err| if err == DecoderError::ShortData {
                        DecoderError::InvalidData
                    } else { err })?;
            dst.write_buf(&dbuf)?;
            Ok(())
        }
    }
    fn set_no_convert(&mut self) {
        self.no_convert = true;
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    fr.seek(SeekFrom::End(-16))?;
    let cat_end = fr.tell() - 4;
    let cat_start = fr.read_u32le()?;
    let cat_size = fr.read_u32le()?;
    let mut tag = [0; 8];
    fr.read_buf(&mut tag)?;
    validate!(&tag == b"XCELZLIB");
    validate!(cat_size > 0x100);
    validate!(u64::from(cat_start + cat_size) == cat_end);
    fr.seek(SeekFrom::Start(u64::from(cat_start) + 0xD))?;

    let mut entries = Vec::new();
    let mut path = Vec::new();
    path.push(".".to_owned());
    while fr.tell() < cat_end {
        let offset = fr.read_u32le()?;
        let _flags = fr.read_u32le()?;
        let size   = fr.read_u32le()?;
        let dsize  = fr.read_u32le()?;
        let _smth4 = fr.read_u32le()?;
        let _smth5 = fr.read_byte()?;
        let namelen = fr.read_u32le()? as usize;
        if namelen > 0 {
            validate!((1..=16).contains(&namelen));
            let mut name = vec![0; namelen];
            fr.read_buf(&mut name)?;
            fr.read_byte()?;
            let is_dir = offset == 0 && size == 0xFFFFFFFF;
            if is_dir && &name != b".." {
                fr.read_u32le()?;
                fr.read_u32le()?;
                loop {
                    let b = fr.read_byte()?;
                    if b == 0 {
                        break;
                    }
                }
            }
            if &name == b".." {
                validate!(!path.is_empty());
                let ndirs = path.len();
                if ndirs > 1 {
                    path.remove(ndirs - 2);
                }
                continue;
            }
            let cur_name = String::from_utf8(name).map_err(|_| DecoderError::InvalidData)?;
            if is_dir {
                path.push(cur_name);
                continue;
            }
            let mut name = String::new();
            if !path.is_empty() {
                for dir in path.iter() {
                    name += dir;
                    name.push(std::path::MAIN_SEPARATOR);
                }
            }
            name += &cur_name;
            if size != 0xFFFFFFFF {
                entries.push(Entry{ name, size, dsize, offset });
            }
        } else {
            fr.read_skip(10)?;
        }
    }

    Ok(Box::new(CEArchive {
        fr, entries,
        entry: 0,
        no_convert: false,
        unpack: false,
    }))
}
