use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;
use super::{GenericEntry, GenericArchive};

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let mut entries = Vec::new();
    let mut raw_name = [0; 256];
    if fr.peek_u16le()? == 0x0A3D {
        // old archive format
        fr.read_u16le()?;
        loop {
            fr.read_buf(&mut raw_name[..16])?;
            let size = fr.read_u32le()?;
            let offset = fr.read_u32le()?;
            let mut name = String::with_capacity(12);
            fr.read_byte()?;
            for &c in raw_name.iter() {
                if c == 0 {
                    break;
                }
                validate!((0x20..=0x7F).contains(&c));
                if c != b'\\' {
                    name.push(c as char);
                } else {
                    name.push(std::path::MAIN_SEPARATOR);
                }
            }
            if name.is_empty() {
                break;
            }
            entries.push(GenericEntry{ name, size, offset });
        }
    } else {
        let tag = fr.peek_tag()?;
        match &tag {
            b"BigF" => { // BigFile
                let mut tag = [0; 12];
                fr.read_buf(&mut tag)?;
                validate!(&tag == b"BigFile 1.00");
                fr.read_u32le()?;
                let nfiles = fr.read_u32le()? as usize;
                validate!((1..=50000).contains(&nfiles));
                entries.reserve(nfiles);
                let _data_size = fr.read_u32le()?;
                let index_size = fr.read_u32le()?;
                fr.read_u32le()?;
                let mut offset = index_size;
                for _ in 0..nfiles {
                    let name_len = fr.read_u32le()? as usize;
                    validate!(name_len > 0 && name_len <= raw_name.len());
                    let _smth = fr.read_u32le()?;
                    validate!(offset >= index_size);
                    let size = fr.read_u32le()?;
                    let _size2 = fr.read_u32le()?;
                    let _size3 = fr.read_u32le()?;
                    fr.read_u32le()?;
                    fr.read_u32le()?;
                    let _smth2 = fr.read_u32le()?;
                    fr.read_buf(&mut raw_name[..name_len])?;
                    let mut name = String::with_capacity(name_len);
                    for &c in raw_name.iter().take(name_len) {
                        if c == 0 {
                            break;
                        }
                        validate!((0x20..=0x7F).contains(&c));
                        if c != b'\\' {
                            name.push(c as char);
                        } else {
                            name.push(std::path::MAIN_SEPARATOR);
                        }
                    }
                    validate!(!name.is_empty());
                    entries.push(GenericEntry{ name, size, offset });
                    offset += size;
                }
            },
            b"HNM4" => { // concatenated HNM4 files
                let mut count = 0;
                while let Ok(tag) = fr.read_tag() {
                    let offset = fr.tell() - 4;
                    validate!(&tag == b"HNM4");
                    fr.read_skip(8)?;
                    let size = fr.read_u32le()?;
                    validate!(size > 64);
                    entries.push(GenericEntry{
                        name: format!("{count:05}.hnm"),
                        offset: offset as u32,
                        size,
                    });
                    count += 1;
                    fr.read_skip((size - 16) as usize)?;
                }
            },
            _ => return Err(DecoderError::InvalidData),
        }
    }

    Ok(Box::new(GenericArchive {
        fr, entries,
        entry: 0,
    }))
}
