use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;

struct EggArchive {
    fr:         FileReader<BufReader<File>>,
    dimensions: Vec<(u16, u16)>,
    fileno:     usize,
    data:       Vec<u8>,
    no_convert: bool,
}

impl ArchiveSource for EggArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.fileno >= self.dimensions.len() {
            return Err(DecoderError::EOF);
        }
        if !self.no_convert {
            Ok(format!("{:06}.ppm", self.fileno))
        } else {
            Ok(format!("{:06}", self.fileno))
        }
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let (w, h) = self.dimensions[self.fileno];
        self.fileno += 1;
        let (width, height) = (usize::from(w), usize::from(h));
        if !self.no_convert {
            dst.write_buf(format!("P6\n{width} {height}\n255\n").as_bytes())?;
        }
        self.data.resize(width * height * 3, 0);
        self.fr.read_buf(&mut self.data)?;
        if !self.no_convert {
            for pix in self.data.chunks_exact_mut(3) {
                pix.swap(0, 2);
            }
        }
        dst.write_buf(&self.data)?;
        Ok(())
    }
    fn set_no_convert(&mut self) {
        self.no_convert = true;
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    validate!(&tag == b"EGG\x00");
    let toc_len = fr.read_u32le()? as usize;
    validate!(toc_len > 0 && (toc_len & 3) == 0);
    let toc2_len = fr.read_u32le()? as usize;
    let _some_offset = fr.read_u32le()?;

    let mut dimensions = Vec::with_capacity(toc_len / 4 + toc2_len / 4);
    for _ in 0..toc_len/4 {
        let width = fr.read_u16le()?;
        let height = fr.read_u16le()?;
        validate!((1..=1024).contains(&width) && (1..=768).contains(&height));
        dimensions.push((width, height));
    }
    fr.read_skip(toc2_len)?;

    Ok(Box::new(EggArchive {
        fr, dimensions,
        fileno: 0,
        data: Vec::new(),
        no_convert: false,
    }))
}
