use std::fs::File;
use std::io::{BufReader, Read, Seek};
use crate::input::*;
use crate::input::util::deflate::*;
use crate::input::util::lzo::*;
use crate::io::byteio::*;

trait ReadLimitedString {
    fn read_strn(&mut self, size: usize) -> DecoderResult<String>;
}

impl<T: Read+Seek> ReadLimitedString for FileReader<T> {
    fn read_strn(&mut self, mut size: usize) -> DecoderResult<String> {
        let mut ss = String::with_capacity(size);
        while size > 0 {
            let c = self.read_byte()?;
            size -= 1;
            if c == 0 {
                break;
            }
            validate!((0x20..=0x7F).contains(&c));
            ss.push(c as char);
        }
        self.read_skip(size)?;
        Ok(ss)
    }
}

struct BarnEntry {
    name:   String,
    offset: u32,
    size:   u32,
    compr:  u8,
}

struct GK3BarnArchive {
    fr:         FileReader<BufReader<File>>,
    entries:    Vec<BarnEntry>,
    entry:      usize,
    no_convert: bool,
    cdata:      Vec<u8>,
    udata:      Vec<u8>,
}

fn write_ppm(dst: &mut dyn ByteIO, data: &[u8]) -> DecoderResult<()> {
    if data.len() > 8 && &data[..4] == b"61nM" {
        let height = usize::from(read_u16le(&data[4..]).unwrap_or_default());
        let width = usize::from(read_u16le(&data[6..]).unwrap_or_default());
        let awidth = (width + 1) & !1;
        if (1..=2048).contains(&width) && (1..=2048).contains(&height)
            && data.len() == awidth * height * 2 + 8 {
            dst.write_buf(format!("P6\n{width} {height}\n255\n").as_bytes())?;
            let mut dline = [0; 2048 * 3];
            for sline in data[8..].chunks_exact(awidth * 2) {
                for (pix, pair) in dline.chunks_exact_mut(3).zip(sline.chunks_exact(2)) {
                    let clr = read_u16le(pair).unwrap_or_default();
                    let r = (clr >> 11) as u8;
                    let g = ((clr >> 5) & 0x3F) as u8;
                    let b = (clr & 0x1F) as u8;
                    pix[0] = (r << 3) | (r >> 2);
                    pix[1] = (g << 2) | (g >> 4);
                    pix[2] = (b << 3) | (b >> 2);
                }
                dst.write_buf(&dline[..width * 3])?;
            }
        }
    }
    dst.write_buf(data)?;
    Ok(())
}

impl ArchiveSource for GK3BarnArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.entry < self.entries.len() {
            Ok(self.entries[self.entry].name.clone())
        } else {
            Err(DecoderError::EOF)
        }
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let entry = &self.entries[self.entry];
        self.entry += 1;

        self.fr.seek(SeekFrom::Start(entry.offset.into()))?;
        let is_bmp = !self.no_convert && entry.name.ends_with(".BMP");
        if self.no_convert || (entry.compr == 0 || entry.compr == 3) {
            if !is_bmp {
                copy_data(&mut self.fr, dst, entry.size as usize)
            } else {
                self.udata.resize(entry.size as usize, 0);
                self.fr.read_buf(&mut self.udata)?;
                write_ppm(dst, &self.udata)
            }
        } else {
            let dsize = self.fr.read_u32le()? as usize;
            let csize = self.fr.read_u32le()?;
            validate!(entry.size > 8 && csize == entry.size - 8);
            self.cdata.resize(csize as usize, 0);
            self.fr.read_buf(&mut self.cdata)?;
            self.udata.clear();
            match entry.compr {
                0 => unreachable!(),
                1 => {
                    Inflate::uncompress(&self.cdata, &mut self.udata)
                        .map_err(|_| DecoderError::InvalidData)?;
                    validate!(self.udata.len() == dsize);
                },
                2 => {
                    let mut br = MemoryReader::new_read(&self.cdata);
                    lzo_unpack(&mut br, csize as usize, &mut self.udata)
                        .map_err(|_| DecoderError::InvalidData)?;
                    validate!(self.udata.len() == dsize);
                },
                _ => return Err(DecoderError::InvalidData),
            }
            if !is_bmp {
                dst.write_buf(&self.udata)?;
            } else {
                write_ppm(dst, &self.udata)?;
            }
            Ok(())
        }
    }
    fn set_no_convert(&mut self) {
        self.no_convert = true;
    }
}

struct BarnRecord {
    is_data:        bool,
    hdr_offset:     u32,
    data_offset:    u32,
    tot_size:       u32,
}

impl BarnRecord {
    fn read(br: &mut dyn ByteIO) -> DecoderResult<Self> {
        let tag = br.read_tag()?;
        let is_data = match &tag {
                b"riDD" => false,
                b"ataD" => true,
                _ => return Err(DecoderError::InvalidData),
            };
        let _ver_min = br.read_u16le()?;
        let _ver_maj = br.read_u16le()?;
        br.read_u32le()?;
        br.read_u32le()?;
        let tot_size = br.read_u32le()?;
        let hdr_offset = br.read_u32le()?;
        let data_offset = br.read_u32le()?;

        Ok(Self{ is_data, hdr_offset, data_offset, tot_size })
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    const POSSIBLE_START: u32 = 0x18;

    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let mut magic = [0; 8];
    fr.read_buf(&mut magic)?;
    validate!(&magic == b"GK3!Barn");
    let version1min = fr.read_u16le()?;
    let version1maj = fr.read_u16le()?;
    let version2min = fr.read_u16le()?;
    let version2maj = fr.read_u16le()?;
    validate!(version1maj == 1 && version1min == 0 && version2maj == 1 && version2min == 0);
    let tot_size = fr.read_u32le()?;
    let toc_offset = fr.read_u32le()?;
    validate!(toc_offset >= POSSIBLE_START && toc_offset < tot_size);
    fr.seek(SeekFrom::Start(toc_offset.into()))?;

    let nentries = fr.read_u32le()? as usize;
    validate!((1..=10000).contains(&nentries));
    let mut dirs = Vec::with_capacity(nentries - 1);
    let mut data_entry = None;
    for _ in 0..nentries {
        let entry = BarnRecord::read(&mut fr)?;
        validate!(entry.hdr_offset >= POSSIBLE_START && entry.hdr_offset < tot_size);
        validate!(entry.data_offset >= POSSIBLE_START && entry.data_offset < tot_size);
        if !entry.is_data {
            dirs.push(entry);
        } else {
            validate!(data_entry.is_none());
            data_entry = Some(entry);
        }
    }
    validate!(!dirs.is_empty());
    validate!(data_entry.is_some());
    let data_entry = data_entry.unwrap();

    let mut entries = Vec::new();
    for dir in dirs.iter() {
        fr.seek(SeekFrom::Start(dir.hdr_offset.into()))?;
        let bname = fr.read_strn(0x20)?;
        if !bname.is_empty() { // ignore external references
            continue;
        }
        fr.read_tag()?;
        let _name = fr.read_strn(0x28)?;
        fr.read_u32le()?;
        let nfiles = fr.read_u32le()? as usize;
        validate!(!name.is_empty());
        validate!((1..=100000).contains(&nfiles));

        fr.seek(SeekFrom::Start(dir.data_offset.into()))?;
        entries.reserve(nfiles);
        for _ in 0..nfiles {
            let size = fr.read_u32le()?;
            let offset = fr.read_u32le()?;
            let _csum = fr.read_u32le()?;
            let _ftype = fr.read_byte()?;
            let compr = fr.read_byte()?;
            let nlen = usize::from(fr.read_byte()?);
            let name = fr.read_strn(nlen)?;
            validate!(!name.is_empty());
            validate!(offset <= data_entry.tot_size);
            fr.read_byte()?;
            entries.push(BarnEntry{ name, size, offset: offset + data_entry.data_offset, compr });
        }
    }

    Ok(Box::new(GK3BarnArchive {
        fr, entries,
        entry: 0,
        no_convert: false,
        cdata: Vec::new(),
        udata: Vec::new(),
    }))
}
