use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct VDXArchive {
    fr:         FileReader<BufReader<File>>,
    fileno:     usize,
    nfiles:     usize,
}

impl ArchiveSource for VDXArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.fileno < self.nfiles {
            Ok(format!("{:03}.vdo", self.fileno))
        } else {
            Err(DecoderError::EOF)
        }
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let mut tag = [0; 6];
        self.fr.read_buf(&mut tag)?;
        validate!(&tag == b"RATVID");
        dst.write_buf(&tag)?;
        let ver = self.fr.read_u16le()?;
        validate!(ver == 1);
        dst.write_u16le(ver)?;

        let mut hdr = [0; 16];
        self.fr.read_buf(&mut hdr)?;
        dst.write_buf(&hdr)?;

        let amode = hdr[1];
        validate!(amode < 3);
        let ablk_size = usize::from(read_u16le(&hdr[4..]).unwrap_or_default());
        let nframes = usize::from(read_u16le(&hdr[8..]).unwrap_or_default());
        let nclrs = usize::from(read_u16le(&hdr[14..]).unwrap_or_default());
        validate!((1..=256).contains(&nclrs));
        let mut pal = [0; 768];
        self.fr.read_buf(&mut pal[..nclrs * 3])?;
        dst.write_buf(&pal[..nclrs * 3])?;

        for _ in 0..nframes {
            let mut ftype = self.fr.read_u16le()?;
            dst.write_u16le(ftype)?;
            if ftype == 2 {
                ftype = self.fr.read_u16le()?;
                dst.write_u16le(ftype)?;
            }
            match ftype {
                0 => {},
                1 => {
                    let size = self.fr.read_u16le()?;
                    dst.write_u16le(size)?;
                    copy_data(&mut self.fr, &mut *dst, usize::from(size))?;
                },
                _ => {
                    let size = usize::from(ftype);
                    copy_data(&mut self.fr, &mut *dst, size & 0x7FFF)?;
                    let len2 = self.fr.read_u16le()?;
                    dst.write_u16le(len2)?;
                    match len2 {
                        0 => {},
                        0xFFFF => {
                            let size = self.fr.read_u16le()?;
                            dst.write_u16le(size)?;
                            copy_data(&mut self.fr, &mut *dst, usize::from(size))?;
                        },
                        _ => {
                            copy_data(&mut self.fr, &mut *dst, usize::from(len2))?;
                        },
                    }
                },
            }
            match amode {
                0 => {},
                1 => {
                    copy_data(&mut self.fr, &mut *dst, ablk_size)?;
                },
                _ => {
                    let size = self.fr.read_u16le()?;
                    dst.write_u16le(size)?;
                    copy_data(&mut self.fr, &mut *dst, usize::from(size))?;
                }
            }
        }

        self.fileno += 1;
        Ok(())
    }
    fn set_no_convert(&mut self) {}
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let nfiles = usize::from(fr.read_u16le()?);
    validate!((1..=100).contains(&nfiles));

    Ok(Box::new(VDXArchive {
        fr,
        nfiles,
        fileno: 0,
    }))
}
