use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::io::byteio::*;

struct Entry {
    name:   String,
    offset: u32,
}

struct MCOArchive {
    fr:         FileReader<BufReader<File>>,
    entries:    Vec<Entry>,
    entry:      usize,
    no_convert: bool,
    frm:        Vec<u16>,
}

fn unpack_bitmap(br: &mut dyn ByteIO, frm: &mut [u16], width: usize) -> DecoderResult<()> {
    let mut pos = 0;
    while pos < frm.len() {
        let op = br.read_byte()?;
        if op < 0x80 {
            frm[pos] = (u16::from(op) << 8) | u16::from(br.read_byte()?);
            pos += 1;
        } else {
            let offs = &OFFSETS[usize::from(op & 0x7F)];
            let xoff = isize::from(offs.0);
            let yoff = isize::from(offs.1);
            let src_pos = (pos as isize) + xoff + yoff * (width as isize);
            validate!(src_pos >= 0);
            let mut src_pos = src_pos as usize;
            let len = usize::from(offs.2);
            validate!(pos + len <= frm.len());
            for _ in 0..len {
                frm[pos] = frm[src_pos];
                pos += 1;
                src_pos += 1;
            }
        }
    }
    Ok(())
}

fn output_bitmap(frm: &[u16], width: usize, height: usize, dst: &mut dyn ByteIO) -> DecoderResult<()> {
    dst.write_buf(format!("P6\n{width} {height}\n255\n").as_bytes())?;
    let mut line = vec![0; width * 3];
    for sline in frm.chunks_exact(width) {
        for (dpix, &pix) in line.chunks_exact_mut(3).zip(sline.iter()) {
            let r = ((pix >> 10) & 0x1F) as u8;
            let g = ((pix >>  5) & 0x1F) as u8;
            let b = ( pix        & 0x1F) as u8;
            dpix[0] = (r << 3) | (r >> 2);
            dpix[1] = (g << 3) | (g >> 2);
            dpix[2] = (b << 3) | (b >> 2);
        }
        dst.write_buf(&line)?;
    }
    Ok(())
}

impl ArchiveSource for MCOArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.entry < self.entries.len() {
            Ok(self.entries[self.entry].name.clone() + if self.no_convert { ".bin" } else { ".ppm" })
        } else {
            Err(DecoderError::EOF)
        }
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let entry = &self.entries[self.entry];
        self.entry += 1;

        self.fr.seek(SeekFrom::Start(entry.offset.into()))?;
        let size = self.fr.read_u32le()? as usize;
        if self.no_convert {
            copy_data(&mut self.fr, dst, size)
        } else {
            let end = self.fr.tell() + (size as u64);
            validate!(size > 8);
            let tag = self.fr.read_tag()?;
            if &tag != b"r16c" {
                return Err(DecoderError::NotImplemented);
            }
            let width = usize::from(self.fr.read_u16le()?);
            let height = usize::from(self.fr.read_u16le()?);
            self.frm.resize(width * height, 0);
            unpack_bitmap(&mut self.fr, &mut self.frm, width).map_err(|_| DecoderError::InvalidData)?;
            validate!(self.fr.tell() <= end);
            output_bitmap(&self.frm, width, height, dst)
        }
    }
    fn set_no_convert(&mut self) {
        self.no_convert = true;
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    fr.seek(SeekFrom::Start(0xFC))?;
    let version = fr.read_u32le()?;
    validate!((1..=3).contains(&version));
    let hdr_size = fr.read_u32le()?;
    validate!(hdr_size == 0x18);
    let _smth = fr.read_u32le()?;
    let _part1_size = fr.read_u32le()?;
    let toc_offset = fr.read_u32le()?;
    let toc_size = fr.read_u32le()?;
    validate!((16..=1048576).contains(&toc_size));
    let _toc1_off = fr.read_u32le()?;
    let _toc2_off = fr.read_u32le()?;
    fr.seek(SeekFrom::Start(toc_offset.into()))?;
    let toc_end = fr.tell() + u64::from(toc_size);

    let mut entries = Vec::new();
    let mut category = 0;
    let num_entries = fr.read_u32le()? as usize;
    if num_entries > 0 {
        println!("Unsupported archive structure");
        return Err(DecoderError::NotImplemented);
    }
    let _name_offset = fr.read_u32le()?;
    let num_entries = fr.read_u32le()? as usize;
    let _name_offset = fr.read_u32le()?;
    fr.read_skip(num_entries * 4)?;
    while fr.tell() < toc_end {
        let num_entries = fr.read_u32le()? as usize;
        let _name_offset = fr.read_u32le()?;
        validate!(num_entries <= 10000);
        for i in 0..num_entries {
            let offset = fr.read_u32le()?;
            let _zero = fr.read_u32le()?;
            validate!(u64::from(offset) >= toc_end);
            entries.push(Entry{name: format!("{category:03}{}{i:05}", std::path::MAIN_SEPARATOR), offset});
        }
        category += 1;
    }
    validate!(fr.tell() == toc_end);
    Ok(Box::new(MCOArchive {
        fr, entries,
        entry: 0,
        no_convert: false,
        frm: Vec::new(),
    }))
}

static OFFSETS: [(i8, i8, u8); 128] = [
    (-2, 0, 2), (-3, 0, 3), ( 0,-2, 2), ( 0,-2, 4), ( 0,-2, 3), (-1, 0, 1), ( 0,-1, 1), ( 0,-1, 2),
    ( 0,-2, 5), (-2, 0, 1), (-4, 0, 4), ( 0,-2, 6), (-1,-1, 2), ( 0,-1,16), ( 0,-2, 1), ( 0,-1, 3),
    (-3, 0, 2), ( 0,-2, 7), ( 1,-1, 2), ( 1,-1, 1), (-1,-2,16), ( 1,-2,16), ( 1,-1, 3), (-2,-1, 3),
    (-1,-1, 1), ( 0,-3, 2), ( 0,-4,16), ( 0,-4, 3), (-2,-1, 2), (-6, 0, 6), ( 2,-1, 2), ( 2,-1, 3),
    ( 0,-4, 4), ( 0,-3, 3), (-1,-1, 3), ( 1,-2, 3), (-1,-2, 2), ( 0,-2, 8), (-4, 0, 3), (-16, 0,16),
    ( 0,-4, 5), ( 0,-4, 2), ( 1,-2, 2), ( 2,-1, 4), (-1,-2, 3), ( 0,-4, 6), ( 0,-2,16), ( 0,-1, 4),
    (-4, 0, 2), (-1,-2, 4), ( 2,-1, 1), ( 0,-2, 9), (-1,-2, 5), ( 0,-3, 4), (-2,-1, 4), ( 3,-1, 2),
    (-2,-1, 1), (-3,-1, 2), ( 1,-1, 4), (-3,-1, 3), (-3, 0, 1), (-6, 0, 3), (-6, 0, 4), (-2,-2, 3),
    (-1,-1, 4), ( 0,-3, 1), (-2,-2,16), ( 0,-4, 7), (-6, 0, 5), ( 0,-2,10), (-1,-4,16), ( 2,-2,16),
    (-5, 0, 5), ( 3,-1, 3), ( 0,-5, 3), ( 2,-2, 2), ( 1,-2, 4), ( 0,-2,11), ( 1,-2, 6), ( 0,-6, 4),
    ( 1,-2, 5), (-1,-2, 6), ( 0,-5, 2), (-2,-2, 2), ( 0,-6, 5), ( 2,-2, 3), (-1,-2, 1), ( 0,-6, 3),
    ( 0,-4, 8), (-1,-3, 2), (-5, 0, 3), (-5, 0, 2), (-4,-1, 3), ( 1,-2, 1), (-2,-1, 5), (-6, 0, 2),
    (-4,-1, 2), ( 1,-1, 5), (-2,-2, 4), ( 1,-3, 2), ( 4,-1, 3), ( 3,-1, 1), ( 1,-3, 3), ( 2,-1, 5),
    (-3,-1, 4), (-1,-2, 7), ( 0,-5, 4), (-1,-2, 8), ( 2,-2, 4), ( 3,-1, 4), (-5, 0, 4), ( 5,-1, 3),
    (-4,-2, 3), (-4,-2, 4), (-4,-1, 4), (-1,-1, 5), ( 4,-1, 2), ( 1,-2,10), ( 0,-8, 4), ( 0,-6, 2),
    (-3,-2, 2), (-3,-1, 1), (-3,-3, 3), ( 0,-6, 6), (-5,-1, 3), (-8, 0, 3), ( 3,-2, 4), ( 3,-2, 3)
];
