use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::input::util::deflate::*;
use crate::io::byteio::*;

struct Entry {
    name:   String,
    offset: u32,
    size:   u32,
    dsize:  u32,
}

struct MNGArchive {
    fr:         FileReader<BufReader<File>>,
    entries:    Vec<Entry>,
    entry:      usize,
    cbuf:       Vec<u8>,
    dbuf:       Vec<u8>,
}

impl ArchiveSource for MNGArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.entry < self.entries.len() {
            Ok(self.entries[self.entry].name.clone())
        } else {
            Err(DecoderError::EOF)
        }
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let entry = &self.entries[self.entry];
        self.entry += 1;

        self.fr.seek(SeekFrom::Start(entry.offset.into()))?;
        if entry.size == entry.dsize {
            copy_data(&mut self.fr, dst, entry.size as usize)
        } else {
            validate!(entry.size > 0);
            self.cbuf.resize(entry.size as usize, 0);
            self.dbuf.clear();
            self.fr.read_buf(&mut self.cbuf)?;
            Inflate::uncompress(&self.cbuf, &mut self.dbuf)
                .map_err(|_| DecoderError::InvalidData)?;
            dst.write_buf(&self.dbuf)?;
            Ok(())
        }
    }
    fn set_no_convert(&mut self) {}
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    validate!(&tag == b"ZGWH");
    let nfiles = fr.read_u32le()? as usize;
    validate!((1..=10000).contains(&nfiles));

    let mut entries = Vec::with_capacity(nfiles);
    for _ in 0..nfiles {
        let mut name = String::with_capacity(12);
        while let Ok(c) = fr.read_byte() {
            if c == 0 { break; }
            validate!(c >= 0x20);
            if c != b'\\' {
                name.push(c as char);
            } else {
                name.push(std::path::MAIN_SEPARATOR);
            }
        }
        validate!(!name.is_empty());
        let size = fr.read_u32le()?;
        let dsize = fr.read_u32le()?;
        let offset = fr.read_u32le()?;
        entries.push(Entry { name, offset, size, dsize });
    }
    Ok(Box::new(MNGArchive {
        fr, entries,
        entry: 0,
        cbuf: Vec::new(),
        dbuf: Vec::new(),
    }))
}
