use std::fs::File;
use std::io::BufReader;
use crate::input::*;
use crate::input::util::imaadpcm::*;
use crate::io::byteio::*;
use crate::io::bitreader::*;

const DICT_SIZE: usize = 4096;
const INVALID_POS: usize = 65536;

struct LZWState {
    dict_sym:   [u8; DICT_SIZE],
    dict_prev:  [u16; DICT_SIZE],
    dict_pos:   usize,
    dict_lim:   usize,
}

impl LZWState {
    fn new() -> Self {
        Self {
            dict_sym:   [0; DICT_SIZE],
            dict_prev:  [0; DICT_SIZE],
            dict_pos:   0,
            dict_lim:   0,
        }
    }
    fn add(&mut self, prev: usize, sym: u8) {
        if self.dict_pos < self.dict_lim {
            self.dict_sym [self.dict_pos] = sym;
            self.dict_prev[self.dict_pos] = prev as u16;
            self.dict_pos += 1;
        }
    }
    fn decode_idx(&self, dst: &mut Vec<u8>, pos: usize, idx: usize) -> DecoderResult<usize> {
        let mut tot_len = 1;
        let mut tidx = idx;
        while tidx >= 256 {
            tidx = self.dict_prev[tidx] as usize;
            tot_len += 1;
        }
        for _ in 0..tot_len {
            dst.push(0);
        }

        let mut end = pos + tot_len - 1;
        let mut tidx = idx;
        while tidx >= 256 {
            dst[end] = self.dict_sym[tidx];
            end -= 1;
            tidx = self.dict_prev[tidx] as usize;
        }
        dst[end] = tidx as u8;

        Ok(tot_len)
    }
    fn unpack(&mut self, src: &[u8], lzw_bits: u8, dst: &mut Vec<u8>) -> DecoderResult<()> {
        validate!(src.len() > 1);
        let mut br = BitReader::new(src, BitReaderMode::BE);

        self.dict_pos = 256;
        self.dict_lim = 1 << lzw_bits;

        let mut lastidx = br.read(lzw_bits).map_err(|_| DecoderError::InvalidData)? as usize;
        validate!(lastidx < 0x100);
        dst.push(lastidx as u8);
        loop {
            let ret         = br.read(lzw_bits);
            if ret.is_err() {
                return Err(DecoderError::InvalidData);
            }
            let idx = ret.unwrap() as usize;
            if idx == self.dict_lim - 1 {
                return Ok(());
            }
            validate!(idx <= self.dict_pos);
            let pos = dst.len();
            if idx != self.dict_pos {
                self.decode_idx(dst, pos, idx)?;
                self.add(lastidx, dst[pos]);
            } else {
                self.decode_idx(dst, pos, lastidx)?;
                let lastsym = dst[pos];
                dst.push(lastsym);
                self.add(lastidx, lastsym);
            }
            lastidx = idx;
        }
    }
}

#[derive(Clone,Copy,Debug,PartialEq)]
enum Version {
    AceVentura,
    HolyGrail,
    WasteOfTime,
}

#[derive(Clone,Copy,Debug,PartialEq)]
enum ResType {
    Unknown(u16),
    Picture,
    MovieData,
    MovieToc,
    Music,
    AudioHG,
    AudioWOT,
    Global(u8),
}

impl ResType {
    fn is_movie(self) -> bool { matches!(self, ResType::MovieData | ResType::MovieToc) }
}

impl std::fmt::Display for ResType {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match *self {
            ResType::Unknown(num) => write!(f, "{num:03}"),
            ResType::Picture => write!(f, "pic"),
            ResType::MovieData => write!(f, "mov_data"),
            ResType::MovieToc => write!(f, "mov_toc"),
            ResType::Music => write!(f, "music"),
            ResType::AudioHG | ResType::AudioWOT => write!(f, "audio"),
            ResType::Global(num) => write!(f, "res{num:04}"),
        }
    }
}

struct ResCatalogue {
    cat:        u8,
    offset:     u32,
    size:       u32,
}

struct Entry {
    ftype:      ResType,
    number:     u32,
    offset:     u32,
    size:       u32,
}

struct BinArchive {
    fr:         FileReader<BufReader<File>>,
    entries:    Vec<Entry>,
    fileno:     usize,
    no_convert: bool,
    data:       Vec<u8>,
    lzw:        LZWState,
    pal:        [u8; 768],
    image:      Vec<u8>,
}

impl BinArchive {
    fn convert_output(&mut self, dst: &mut dyn ByteIO, ftype: ResType) -> DecoderResult<()> {
        match ftype {
            ResType::Picture => {
                self.unpack_picture(dst).map_err(|err|
                    if err == DecoderError::NotImplemented { err } else { DecoderError::InvalidData })?;
            },
            ResType::AudioHG | ResType::AudioWOT => {
                self.unpack_audio(dst, ftype).map_err(|_| DecoderError::InvalidData)?;
            },
            _ => {
                dst.write_buf(&self.data)?;
            }
        }
        Ok(())
    }
    fn unpack_picture(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);
        let _hdrsize = br.read_u16le()?;
        let width = usize::from(br.read_u16le()?);
        let stride = usize::from(br.read_u16le()?);
        let height = usize::from(br.read_u16le()?);
        let compr = br.read_u16le()?;
        validate!(stride >= width);
        validate!((1..=2048).contains(&width) && (1..=768).contains(&height));
        br.read_skip(10)?;

        self.image.clear();
        self.image.reserve(stride * height);
        match compr {
            0 => {
                loop {
                    let chunk = br.read_u16le()?;
                    if chunk == 0xE000 {
                        break;
                    }
                    let len = usize::from(chunk & 0x1FFF);
                    let mode = chunk >> 13;
                    validate!(len > 0);
                    if mode == 0 {
                        br.read_extend(&mut self.image, len)?;
                    } else {
                        let lzw_bits = match mode {
                            2 => 10,
                            3 => 11,
                            4 => 12,
                            _ => return Err(DecoderError::NotImplemented),
                        };
                        let len2 = usize::from(br.read_u16le()?);
                        validate!(len2 + 2 == len);
                        let start = br.tell() as usize;
                        br.read_skip(len2)?;
                        self.lzw.unpack(&self.data[start..][..len], lzw_bits, &mut self.image)?;
                    }
                }
            },
            0x80 => {
                for _ in 0..height {
                    br.read_u16le()?;
                }
                for _ in 0..height {
                    let mut pos = 0;
                    loop {
                        let op = br.read_byte()?;
                        match op {
                            0 => break,
                            1..=0x7F => {
                                let len = usize::from(op);
                                let clr = br.read_byte()?;
                                validate!(pos + len <= width);
                                for _ in 0..len {
                                    self.image.push(clr);
                                }
                                pos += len;
                            },
                            0xFF => {
                                let len = usize::from(br.read_byte()?);
                                validate!(pos + len <= width);
                                if len > 0 {
                                    br.read_extend(&mut self.image, len)?;
                                }
                                pos += len;
                            },
                            _ => {
                                let len = usize::from(!op);
                                validate!(pos + len <= width);
                                br.read_extend(&mut self.image, len)?;
                                pos += len;
                            },
                        }
                    }
                }
            },
            _ => return Err(DecoderError::NotImplemented),
        }

        let owidth = if self.image.len() > width * height { stride } else { width };
        dst.write_buf(format!("P6\n{width} {height}\n255\n").as_bytes())?;
        for line in self.image.chunks_exact(owidth).take(height).rev() {
            for &b in line[..width].iter() {
                dst.write_buf(&self.pal[usize::from(b) * 3..][..3])?;
            }
        }

        Ok(())
    }
    fn unpack_audio(&mut self, dst: &mut dyn ByteIO, ftype: ResType) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);
        let mut size = 0;
        if ftype == ResType::AudioHG {
            size = br.read_u32le()?;
            br.read_u16le()?;
            let _one = br.read_u32le()?;
            let _smth = br.read_u16le()?;
        }
        let mut wavhdr = [0; 16];
        br.read_buf(&mut wavhdr)?;
        if ftype == ResType::AudioWOT {
            size = br.read_u32le()?;
            br.read_u32le()?;
        }
        validate!(br.left() == i64::from(size));

        let channels = wavhdr[2];
        validate!(channels == 1 || channels == 2);
        dst.write_buf(b"RIFF")?;
        dst.write_u32le(size * 4 + 36)?;
        dst.write_buf(b"WAVEfmt ")?;
        dst.write_u32le(16)?;
        dst.write_buf(&wavhdr)?;
        dst.write_buf(b"data")?;
        dst.write_u32le(size * 4)?;

        let mut state = [IMAState::new(), IMAState::new()];
        while let Ok(b) = br.read_byte() {
            let samp = state[0].expand_sample(b & 0xF);
            dst.write_u16le(samp as u16)?;
            let samp = state[if channels == 2 { 1 } else { 0 }].expand_sample(b >> 4);
            dst.write_u16le(samp as u16)?;
        }

        Ok(())
    }
}

impl ArchiveSource for BinArchive {
    fn get_file_name(&mut self) -> DecoderResult<String> {
        if self.fileno >= self.entries.len() {
            return Err(DecoderError::EOF);
        }
        let entry = &self.entries[self.fileno];

        let num = entry.number;
        let ftype = entry.ftype;
        const SEP: char = std::path::MAIN_SEPARATOR;

        let name = match ftype {
            ResType::Unknown(_) => format!("unknown{SEP}{num:04}.{ftype}"),
            ResType::Picture if !self.no_convert => format!("images{SEP}{num:04}.ppm"),
            ResType::Picture => format!("images{SEP}{num:04}.{ftype}"),
            ResType::AudioHG if !self.no_convert => format!("audio{SEP}{num:04}.wav"),
            ResType::AudioWOT if !self.no_convert => format!("audio{SEP}{num:04}.wav"),
            ResType::AudioHG => format!("audio{SEP}{num:04}.{ftype}"),
            ResType::AudioWOT => format!("audio{SEP}{num:04}.{ftype}"),
            ResType::Music => format!("music{SEP}{num:04}.{ftype}"),
            ResType::MovieToc | ResType::MovieData => format!("movies{SEP}{num:04}.{ftype}"),
            ResType::Global(resnum) => format!("res{resnum:04}"),
        };
        Ok(name)
    }
    fn extract_file(&mut self, dst: &mut dyn ByteIO) -> DecoderResult<()> {
        let entry = &self.entries[self.fileno];
        self.fileno += 1;
        self.fr.seek(SeekFrom::Start(entry.offset.into()))?;
        if self.no_convert || entry.size == 0 {
            copy_data(&mut self.fr, dst, entry.size as usize)
        } else {
            self.data.resize(entry.size as usize, 0);
            self.fr.read_buf(&mut self.data)?;
            self.convert_output(dst, entry.ftype)?;
            Ok(())
        }
    }
    fn set_no_convert(&mut self) {
        self.no_convert = true;
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn ArchiveSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    let version = match &tag {
        b"7Lb\x01" => Version::AceVentura,
        b"7LB\x01" => Version::HolyGrail,
        b"7L\x01\x00" => Version::WasteOfTime,
        &[b'7', b'L', _, _] => return Err(DecoderError::NotImplemented),
        _ => return Err(DecoderError::InvalidData),
    };

    let mut res_cat = Vec::new();

    let mut entries = Vec::new();
    let mut pal = std::array::from_fn(|i| (i / 3) as u8);
    let mut unk_no = 0;
    let mut aud_no = 0;
    let mut pic_no = 0;
    let mut mov_no = 0;
    let mut mov_dat = true;
    let mut mus_no = 0;

    match version {
        Version::AceVentura => {
            fr.seek(SeekFrom::Start(0xF2))?;
            for cat in 0..12 {
                let offset = fr.read_u32le()?;
                let size = fr.read_u32le()?;
                if size > 0 {
                    res_cat.push(ResCatalogue{ cat, offset, size });
                }
            }
            validate!(res_cat.len() > 1 && res_cat[0].cat == 0);
            let toc = res_cat.remove(0);
            fr.seek(SeekFrom::Start(toc.offset.into()))?;
            validate!(toc.size % 10 == 0);
            let nfiles = (toc.size / 10) as usize;
            entries.reserve(nfiles + res_cat.len());
            for _i in 0..nfiles {
                let type_id = fr.read_u16le()?;
                let offset = fr.read_u32le()?;
                let size = fr.read_u32le()?;
                let ftype = match type_id {
                        1 => ResType::Picture,
                        2 if mov_dat => ResType::MovieData,
                        2 => ResType::MovieToc,
                        4 => ResType::Music,
                        9 => ResType::AudioHG,
                        _ => ResType::Unknown(type_id),
                    };
                let number = match ftype {
                        ResType::Unknown(_) => {
                            let ret = unk_no;
                            unk_no += 1;
                            ret
                        },
                        ResType::Picture => {
                            let ret = pic_no;
                            pic_no += 1;
                            ret
                        },
                        ResType::AudioHG | ResType::AudioWOT => {
                            let ret = aud_no;
                            aud_no += 1;
                            ret
                        },
                        ResType::Music => {
                            let ret = mus_no;
                            mus_no += 1;
                            ret
                        },
                        ResType::MovieData => mov_no,
                        ResType::MovieToc => {
                            let ret = mov_no;
                            mov_no += 1;
                            ret
                        },
                        ResType::Global(_) => unreachable!(),
                    };
                if ftype.is_movie() {
                    mov_dat = !mov_dat;
                }
                //xxx: Unknown(11-17) types offsets seem to be relative to some global resource
                entries.push(Entry{ ftype, number, offset, size });
            }
            for rc in res_cat.iter() {
                entries.push(Entry{ number: 0, ftype: ResType::Global(rc.cat), offset: rc.offset, size: rc.size });
                if rc.cat == 6 {
                    let mut palbuf = [0; 1024];
                    if fr.seek(SeekFrom::Start(rc.offset.into())).is_ok() && rc.size <= 1024 &&
                        fr.read_buf(&mut palbuf[..rc.size as usize]).is_ok() {
                        for (dclr, sclr) in pal.chunks_exact_mut(3).skip(10)
                                .zip(palbuf.chunks_exact(4).take(rc.size as usize / 4)) {
                            dclr[0] = sclr[0];
                            dclr[1] = sclr[1];
                            dclr[2] = sclr[2];
                        }
                    }
                }
            }
        }
        Version::HolyGrail => {
            fr.seek(SeekFrom::Start(0xE2))?;
            for cat in 0..12 {
                let offset = fr.read_u32le()?;
                let size = fr.read_u32le()?;
                if size > 0 {
                    res_cat.push(ResCatalogue{ cat, offset, size });
                }
            }
            validate!(res_cat.len() > 1 && res_cat[0].cat == 0);
            let toc = res_cat.remove(0);
            fr.seek(SeekFrom::Start(toc.offset.into()))?;
            validate!(toc.size % 10 == 0);
            let nfiles = (toc.size / 10) as usize;
            entries.reserve(nfiles + res_cat.len());
            for _i in 0..nfiles {
                let type_id = fr.read_u16le()?;
                let offset = fr.read_u32le()?;
                let size = fr.read_u32le()?;
                let ftype = match type_id {
                        1 => ResType::Picture,
                        2 if mov_dat => ResType::MovieData,
                        2 => ResType::MovieToc,
                        4 => ResType::Music,
                        9 => ResType::AudioHG,
                        _ => ResType::Unknown(type_id),
                    };
                let number = match ftype {
                        ResType::Unknown(_) => {
                            let ret = unk_no;
                            unk_no += 1;
                            ret
                        },
                        ResType::Picture => {
                            let ret = pic_no;
                            pic_no += 1;
                            ret
                        },
                        ResType::AudioHG | ResType::AudioWOT => {
                            let ret = aud_no;
                            aud_no += 1;
                            ret
                        },
                        ResType::Music => {
                            let ret = mus_no;
                            mus_no += 1;
                            ret
                        },
                        ResType::MovieData => mov_no,
                        ResType::MovieToc => {
                            let ret = mov_no;
                            mov_no += 1;
                            ret
                        },
                        ResType::Global(_) => unreachable!(),
                    };
                if ftype.is_movie() {
                    mov_dat = !mov_dat;
                }
                //xxx: Unknown(11-17) types offsets seem to be relative to some global resource
                entries.push(Entry{ ftype, number, offset, size });
            }
            for rc in res_cat.iter() {
                entries.push(Entry{ number: 0, ftype: ResType::Global(rc.cat), offset: rc.offset, size: rc.size });
                if rc.cat == 6 {
                    let mut palbuf = [0; 1024];
                    if fr.seek(SeekFrom::Start(rc.offset.into())).is_ok() && rc.size <= 1024 &&
                        fr.read_buf(&mut palbuf[..rc.size as usize]).is_ok() {
                        for (dclr, sclr) in pal.chunks_exact_mut(3).skip(10)
                                .zip(palbuf.chunks_exact(4).take(rc.size as usize / 4)) {
                            dclr[0] = sclr[0];
                            dclr[1] = sclr[1];
                            dclr[2] = sclr[2];
                        }
                    }
                }
            }
        },
        Version::WasteOfTime => {
            fr.seek(SeekFrom::Start(0x54))?;
            let nfiles = usize::from(fr.read_u16le()?);
            validate!(nfiles > 0);
            let nclrs = usize::from(fr.read_u16le()?);
            validate!((1..=256).contains(&nclrs));
            let txt_size = u32::from(fr.read_u16le()?);
            fr.read_skip(10)?;
            let res0_size = u32::from(fr.read_u16le()?);
            fr.read_u32le()?;
            let part1_size = fr.read_u32le()?;
            fr.read_skip(8)?;

            entries.reserve(nfiles + 4);
            let base_offset = fr.tell() as u32 + (nfiles as u32) * 10;

            let mut res_offset = base_offset + part1_size;
            if res0_size > 0 {
                entries.push(Entry{ ftype: ResType::Global(0), number: 0, offset: res_offset, size: res0_size });
                res_offset += res0_size;
            } else {
                res_offset += 0xD9;
            }
            let pal_offset = res_offset;
            entries.push(Entry{ ftype: ResType::Global(1), number: 0, offset: res_offset, size: (nclrs as u32) * 4 });
            res_offset += (nclrs as u32) * 4;
            entries.push(Entry{ ftype: ResType::Global(2), number: 0, offset: res_offset, size: txt_size });

            for _i in 0..nfiles {
                let type_id = fr.read_u16le()?;
                let mut offset = fr.read_u32le()?;
                let size = fr.read_u32le()?;
                let ftype = match type_id {
                        1 => ResType::Picture,
                        3 => ResType::AudioWOT,
                        _ => ResType::Unknown(type_id),
                    };
                if type_id > 10 {
                    offset += base_offset;
                }
                let number = match ftype {
                        ResType::Unknown(_) => {
                            let ret = unk_no;
                            unk_no += 1;
                            ret
                        },
                        ResType::Picture => {
                            let ret = pic_no;
                            pic_no += 1;
                            ret
                        },
                        ResType::AudioHG | ResType::AudioWOT => {
                            let ret = aud_no;
                            aud_no += 1;
                            ret
                        },
                        ResType::Music => {
                            let ret = mus_no;
                            mus_no += 1;
                            ret
                        },
                        ResType::MovieData => mov_no,
                        ResType::MovieToc => {
                            let ret = mov_no;
                            mov_no += 1;
                            ret
                        },
                        ResType::Global(_) => unreachable!(),
                    };
                if ftype.is_movie() {
                    mov_dat = !mov_dat;
                }
                entries.push(Entry{ ftype, number, offset, size });
            }
            let mut palbuf = [0; 1024];
            if fr.seek(SeekFrom::Start(u64::from(pal_offset))).is_ok() &&
                fr.read_buf(&mut palbuf[..nclrs * 4]).is_ok() {
                for (dclr, sclr) in pal.chunks_exact_mut(3).skip(10)
                        .zip(palbuf.chunks_exact(4).take(nclrs)) {
                    dclr[0] = sclr[0];
                    dclr[1] = sclr[1];
                    dclr[2] = sclr[2];
                }
            }
        },
    }

    Ok(Box::new(BinArchive {
        fr, entries,
        fileno: 0,
        no_convert: false,
        pal,
        data: Vec::new(),
        lzw: LZWState::new(),
        image: Vec::new(),
    }))
}
