use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

const REF_WIDTH: usize = 328;
const REF_HEIGHT: usize = 208;
const WIDTH: usize = 320;
const HEIGHT: usize = 200;
const FPS: u32 = 12;
const ARATE: u32 = 11025;
const MASK_SIZE: usize = 0x86;

struct SpaceAceDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    frame2:     Vec<u8>,
    dframe:     Vec<u8>,
    ref_frame:  [u8; REF_WIDTH * REF_HEIGHT],
    xshift:     usize,
    yshift:     usize,
    data:       Vec<u8>,
    mask:       [u8; MASK_SIZE],
    abuf:       Vec<u8>,
    audio:      bool,
    arate:      u32,
    aframe_len: usize,
}

macro_rules! put_pixel {
    (reference; $dst:expr, $x:expr, $y:expr, $width:expr, $val:expr) => {
        let x4 = $x * 4;
        let pix = $val;
        if pix != 0 {
            $dst[(x4 % $width) + (x4 / $width) + $y * REF_WIDTH] = pix;
        }
    };
    (update; $dst:expr, $x:expr, $y:expr, $val:expr) => {
        let pix = $val;
        if pix != 0 {
            $dst[$x + $y * WIDTH] = pix;
        }
    }
}

impl SpaceAceDecoder {
    fn decode_ref_frame(&mut self, width: usize, height: usize, xoff: usize, yoff: usize, xshift: usize, yshift: usize) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

//println!(" {width}x{height} offset {xoff},{yoff} << {xshift},{yshift}");
        self.xshift = xshift;
        self.yshift = yshift;
        self.dframe.clear();
        if width > 0 || height > 0 {
            loop {
                let op = br.read_byte()?;
                match op {
                    0x00 => {},
                    0x01..=0x7F => {
                        let len = usize::from(op);
                        br.read_extend(&mut self.dframe, len)?;
                    },
                    0x80 => break,
                    _ => {
                        let len = 256 - usize::from(op);
                        let clr = br.read_byte()?;
                        for _ in 0..len {
                            self.dframe.push(clr);
                        }
                    }
                }
            }
            if width > 0 && height > 0 && xoff == 0 && yoff == 0 && xshift == 0 && yshift == 0 {
                for (x, col) in self.dframe.chunks_exact(height).enumerate() {
                    for (y, &pix) in col.iter().enumerate() {
                        put_pixel!(reference; self.ref_frame, x, y, width, pix);
                    }
                }
                return Ok(());
            }
            let (cur_w, cur_h) = match (width, height) {
                    (0, _) => (REF_WIDTH, height - yoff),
                    (_, 0) => (width - xoff, REF_HEIGHT),
                    _ => (width - xoff, height - yoff),
                };
            validate!(cur_w > 0 && cur_h > 0 && cur_w * cur_h <= self.dframe.len());

            for (x, col) in self.dframe.chunks_exact(cur_h).enumerate() {
                for (y, &pix) in col.iter().enumerate() {
                    put_pixel!(reference; self.ref_frame, (x + xoff + xshift) % REF_WIDTH, (y + yoff + yshift) % REF_HEIGHT, cur_w, pix);
                }
            }
        } else {
            for el in self.ref_frame.iter_mut() {
                *el = 0;
            }
        }

        Ok(())
    }
    fn masked_blit(&mut self) {
        let mut br = BitReader::new(&self.mask, BitReaderMode::BE);
        if self.xshift == 0 && self.yshift == 0 {
            for (dstrip, sstrip) in self.frame.chunks_exact_mut(WIDTH * 8)
                    .zip(self.ref_frame.chunks_exact(REF_WIDTH * 8)) {
                for x in (0..WIDTH).step_by(8) {
                    if br.read_bool().unwrap_or(true) {
                        for (dline, sline) in dstrip[x..].chunks_mut(WIDTH)
                                .zip(sstrip[x..].chunks(REF_WIDTH)) {
                            dline[..8].copy_from_slice(&sline[..8]);
                        }
                    }
                }
                br.read_bool().unwrap_or(false);
            }
        } else {
            for (row_y, dstrip) in self.frame.chunks_exact_mut(WIDTH * 8).enumerate() {
                for x in (0..WIDTH).step_by(8) {
                    if br.read_bool().unwrap_or(true) {
                        for (y1, row) in dstrip[x..].chunks_mut(WIDTH).enumerate() {
                            for (x1, el) in row[..8].iter_mut().enumerate() {
                                *el = self.ref_frame[((x + x1 + self.xshift) % REF_WIDTH) + ((row_y * 8 + y1 + self.yshift) % REF_HEIGHT) * REF_WIDTH];
                            }
                        }
                    }
                }
                br.read_bool().unwrap_or(false);
            }
        }
    }
    fn decode_update(&mut self, _xshift: usize, _yshift: usize) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        'part_loop: for off in 0..4 {
            let mut x = 0;
            while x < WIDTH {
                let mut y = 0;
                while y < 200 {
                    let op = br.read_byte()?;
                    match op {
                        0x00 => {},
                        0x01..=0x7F => {
                            let len = usize::from(op);
                            validate!(y + len <= 200);
                            for _ in 0..len {
                                put_pixel!(update; self.frame, x * 4 + off, y, br.read_byte()?);
                                y += 1;
                            }
                        },
                        0x80 => continue 'part_loop,
                        0x81 => break,
                        _ => {
                            let len = 256 - usize::from(op);
                            validate!(y + len <= 200);
                            let clr = br.read_byte()?;
                            if clr != 0 {
                                for _ in 0..len {
                                    put_pixel!(update; self.frame, x * 4 + off, y, clr);
                                    y += 1;
                                }
                            } else {
                                y += len;
                            }
                        },
                    }
                }
                x += 1;
            }
        }

        Ok(())
    }
}

impl InputSource for SpaceAceDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: FPS,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    2,
                    sample_type: AudioSample::U8,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = &mut self.fr;
        loop {
            if self.audio && self.abuf.len() >= self.aframe_len {
                let mut audio = vec![0; self.aframe_len];
                audio.copy_from_slice(&self.abuf[..self.aframe_len]);
                self.abuf.drain(..self.aframe_len);
                self.audio = false;
                return Ok((1, Frame::AudioU8(audio)));
            }

            let next_off = br.read_u32le()?;
            let chunk_id = br.read_u16le()?;
            let chunk_size = (next_off - (br.tell() as u32)) as usize;
//println!("chunk {chunk_id} @ {:X} next {next_off:X}", br.tell() - 6);
            validate!(next_off > br.tell() as u32);

            match chunk_id {
                0 => {
                    let xshift = br.read_u16le()? as usize;
                    let yshift = br.read_u16le()? as usize;
                    let xoff = br.read_u16le()? as usize;
                    let width = br.read_u16le()? as usize;
                    let yoff = br.read_u16le()? as usize;
                    let height = br.read_u16le()? as usize;
                    let size = chunk_size - 16;
                    let offset = br.read_u32le()?;
                    validate!(offset == br.tell() as u32);
                    self.data.resize(size, 0);
                    br.read_buf(&mut self.data)?;
                    self.decode_ref_frame(width, height, xoff, yoff, xshift, yshift)
                            .map_err(|_| DecoderError::InvalidData)?;
                    br = &mut self.fr;
                },
                1 => {
                    let xshift = br.read_u16le()? as usize;
                    let yshift = br.read_u16le()? as usize;
                    let mask_offset = br.read_u32le()?;
                    let data_offset = br.read_u32le()?;
                    validate!(mask_offset == br.tell() as u32);
                    validate!(data_offset == mask_offset + (MASK_SIZE as u32));
                    validate!(data_offset <= next_off);
                    br.read_buf(&mut self.mask)?;
                    self.masked_blit();
                    br = &mut self.fr;
                    if data_offset < next_off {
                        self.data.resize((next_off - data_offset) as usize, 0);
                        br.read_buf(&mut self.data)?;
                        self.decode_update(xshift, yshift)
                                .map_err(|_| DecoderError::InvalidData)?;
                        br = &mut self.fr;
                    }
                },
                2 => {
                    let asize = br.read_u32le()?;
                    let offset = br.read_u32le()?;
                    validate!(asize > 0);
                    validate!(offset == br.tell() as u32);
                    validate!(br.tell() as u32 + asize == next_off);

                    br.read_extend(&mut self.abuf, asize as usize)?;
                },
                5 => {
                    br.seek(SeekFrom::Start(u64::from(next_off)))?;
                    std::mem::swap(&mut self.frame, &mut self.frame2);
                    self.audio = true;
                    return Ok((0, Frame::VideoPal(self.frame2.clone(), self.pal)));
                },
                6 => {
                    let nclrs = br.read_u16le()? as usize;
                    let offset = br.read_u32le()?;
                    validate!(nclrs <= 256 && nclrs * 3 + 6 == chunk_size);
                    validate!(offset == (br.tell() as u32));
                    br.read_vga_pal_some(&mut self.pal[..nclrs * 3])?;
                },
                _ => {
                    if next_off == 0x80000000 {
                        return Err(DecoderError::EOF);
                    }
                    br.seek(SeekFrom::Start(u64::from(next_off)))?;
                },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let fr = FileReader::new_read(BufReader::new(file));

    Ok(Box::new(SpaceAceDecoder {
        fr,
        pal: [0; 768],
        data: Vec::new(),
        mask: [0; MASK_SIZE],
        dframe: Vec::with_capacity(REF_WIDTH * REF_HEIGHT),
        ref_frame: [0; REF_WIDTH * REF_HEIGHT],
        frame:  vec![0; WIDTH * HEIGHT],
        frame2: vec![0; WIDTH * HEIGHT],
        xshift: 0,
        yshift: 0,
        abuf: Vec::with_capacity(32000),
        audio: false,
        aframe_len: (ARATE / FPS) as usize * 2,
        arate: ARATE,
    }))
}
