use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

// The format operates on 70-75fps base with repeating the same frame 5-8 times
// so it makes sense to show not so many frames instead.
const TICKS_PER_FRAME: usize = 4;

fn lzss_unpack(src: &[u8], dst: &mut Vec<u8>, num_ops: usize) -> DecoderResult<()> {
    let mut br = MemoryReader::new_read(src);

    let mut window = [0; 4096];
    let mut wpos = 4096 - 18;
    let mut flags = 0;
    let mut fbits = 0;
    dst.clear();
    for _ in 0..num_ops {
        if fbits == 0 {
            flags = br.read_byte()?;
            fbits = 8;
        }
        if (flags & 1) != 0 {
            let b = br.read_byte()?;
            window[wpos] = b;
            wpos = (wpos + 1) & 0xFFF;
            dst.push(b);
        } else {
            let lo_byte = br.read_byte()?;
            let hi_byte = br.read_byte()?;

            let offset = (u16::from(hi_byte & 0xF0) * 16 + u16::from(lo_byte)) as usize;
            let len = ((hi_byte & 0xF) + 3) as usize;

            for i in 0..len {
                let b = window[(offset + i) & 0xFFF];
                window[(wpos + i) & 0xFFF] = b;
                dst.push(b);
            }
            wpos = (wpos + len) & 0xFFF;
        }
        flags >>= 1;
        fbits  -= 1;
    }
    Ok(())
}

#[derive(Default)]
struct FadeState {
    factors:    [u16; 3],
    steps:      [u16; 3],
    mode:       u8,
    is_active:  bool,
}

impl FadeState {
    fn new() -> Self { Self::default() }
    fn is_active(&self) -> bool { self.is_active }
    fn set_state(&mut self, steps: [u16; 3], mode: u8) {
        self.is_active = (mode & 2) == 0;
        self.mode = mode;
        self.steps = steps;
        self.factors = if (mode & 1) == 0 { [0x3F00; 3] } else { [0; 3] };
    }
    fn reset(&mut self) {
        self.is_active = false;
    }
    fn update(&mut self, pal: &[u8; 768]) -> [u8; 768] {
        if !self.is_active {
            return if (self.mode & 1) == 0 { *pal } else { [0; 768] };
        }
        if (self.mode & 1) == 0 {
            for (factor, &step) in self.factors.iter_mut().zip(self.steps.iter()) {
                *factor = factor.saturating_sub(step);
            }
            if self.factors == [0; 3] {
                self.is_active = false;
            }
        } else {
            for (factor, &step) in self.factors.iter_mut().zip(self.steps.iter()) {
                *factor = factor.saturating_add(step).min(0x3F00);
            }
            if self.factors == [0x3F00; 3] {
                self.is_active = false;
            }
        }
        let mut dpal = [0; 768];
        for (dclr, clr) in dpal.chunks_exact_mut(3).zip(pal.chunks_exact(3)) {
            for (dst, (&src, &factor)) in dclr.iter_mut().zip(clr.iter().zip(self.factors.iter())) {
                *dst = src.saturating_sub((factor >> 6) as u8);
            }
        }
        dpal
    }
}

struct CDA2Decoder {
    fr:         FileReader<BufReader<File>>,
    is_cda2:    bool,
    data:       Vec<u8>,
    udata:      Vec<u8>,
    frame:      Vec<u8>,
    cur_frm:    usize,
    pal:        [u8; 768],
    fade:       FadeState,
    fade_start: usize,
    width:      usize,
    height:     usize,
    rate:       u16,
    frame_sz:   Vec<u32>,
    arate:      u16,
    audio:      Vec<i16>,
    ablk_size:  usize,
    repeats:    u8,
    apts:       usize,
    vpts:       usize,
    vpts_end:   usize,
}

impl CDA2Decoder {
    fn do_tile_mode_0(dst: &mut [u8], stride: usize, br: &mut dyn ByteIO, masks: &[u8]) -> DecoderResult<()> {
        let mut mask = BitReader::new(masks, BitReaderMode::BE);
        for line in dst.chunks_mut(stride) {
            for el in line[..32].iter_mut() {
                if mask.read_bool()? {
                    *el = br.read_byte()?;
                }
            }
        }
        Ok(())
    }
    fn do_tile_mode_1(dst: &mut [u8], stride: usize, br: &mut dyn ByteIO, masks: &[u8]) -> DecoderResult<()> {
        let mut mask = BitReader::new(masks, BitReaderMode::BE);
        for lines in dst.chunks_mut(stride * 2) {
            for x in (0..32).step_by(2) {
                if mask.read_bool()? {
                    let (line0, line1) = lines[x..].split_at_mut(stride);
                    br.read_buf(&mut line0[..2])?;
                    br.read_buf(&mut line1[..2])?;
                }
            }
        }
        Ok(())
    }
    fn do_tile_mode_2(dst: &mut [u8], stride: usize, br: &mut dyn ByteIO, masks: &[u8]) -> DecoderResult<()> {
        let mut mask = BitReader::new(masks, BitReaderMode::BE);
        for lines in dst.chunks_mut(stride * 4) {
            for x in (0..32).step_by(4) {
                if mask.read_bool()? {
                    for line in lines[x..].chunks_mut(stride) {
                        br.read_buf(&mut line[..4])?;
                    }
                }
            }
        }
        Ok(())
    }
    fn do_tile_mode_3(dst: &mut [u8], stride: usize, br: &mut dyn ByteIO) -> DecoderResult<()> {
        for line in dst.chunks_mut(stride) {
            br.read_buf(&mut line[..32])?;
        }
        Ok(())
    }
    fn draw_frame(&mut self, has_pal: bool) -> DecoderResult<()> {
        let off = if has_pal { self.pal.len() } else { 0 };
        let (modes, data) = self.udata[off..].split_at((self.width / 64) * (self.height / 40));

        let mut br = MemoryReader::new_read(data);

        for (strip, modes) in self.frame.chunks_exact_mut(self.width * 40)
                .zip(modes.chunks_exact(self.width / 64)) {
            for x in (0..self.width).step_by(32) {
                let tmodes = modes[x / 64];
                let tile_mode = if (x & 32) == 0 { tmodes >> 4 } else { tmodes & 0xF };
                match tile_mode {
                    0 => {
                        let mask_size = 160;
                        let masks = &data[br.tell() as usize..][..mask_size];
                        br.read_skip(mask_size)?;
                        Self::do_tile_mode_0(&mut strip[x..], self.width, &mut br, masks)?;
                    },
                    1 => {
                        let mask_size = 40;
                        let masks = &data[br.tell() as usize..][..mask_size];
                        br.read_skip(mask_size)?;
                        Self::do_tile_mode_1(&mut strip[x..], self.width, &mut br, masks)?;
                    },
                    2 => {
                        let mask_size = 10;
                        let masks = &data[br.tell() as usize..][..mask_size];
                        br.read_skip(mask_size)?;
                        Self::do_tile_mode_2(&mut strip[x..], self.width, &mut br, masks)?;
                    },
                    3 => Self::do_tile_mode_3(&mut strip[x..], self.width, &mut br)?,
                    _ => {},
                }
            }
        }

        Ok(())
    }
    fn get_audio(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut ret = vec![0; self.ablk_size];
        ret.copy_from_slice(&self.audio[..self.ablk_size]);
        self.audio.drain(..self.ablk_size);
        Ok((1, Frame::AudioS16(ret)))
    }
}

impl InputSource for CDA2Decoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: TICKS_PER_FRAME as u32 * 100,
                    tb_den: u32::from(self.rate),
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: u32::from(self.arate),
                    channels:    1,
                    sample_type: AudioSample::S16,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.audio.len() >= self.ablk_size && self.apts < self.vpts {
            self.apts += TICKS_PER_FRAME;
            return self.get_audio();
        }

        if self.vpts + TICKS_PER_FRAME <= self.vpts_end {
            self.vpts += TICKS_PER_FRAME;
            if !self.fade.is_active() {
                return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
            } else {
                let mut fade_pal = self.fade.update(&self.pal);
                for _ in 0..(self.vpts - self.fade_start - 1).min(TICKS_PER_FRAME - 1) {
                    fade_pal = self.fade.update(&self.pal);
                }
                return Ok((0, Frame::VideoPal(self.frame.clone(), fade_pal)));
            }
        }

        if self.cur_frm >= self.frame_sz.len() {
            return Err(DecoderError::EOF);
        }

        let mut br = &mut self.fr;
        let frm_size = self.frame_sz[self.cur_frm];
        let end = br.tell() + u64::from(frm_size);

        let frm_flags = br.read_byte()?;

        if (frm_flags & 0x02) != 0 {
            let audio_size = br.read_u16le()? as usize;
            validate!(br.tell() + (audio_size as u64) <= end);
            validate!((audio_size & 1) == 0);
            for _ in 0..audio_size/2 {
                let sample = br.read_u16le()? as i16;
                self.audio.push(sample);
            }
        }

        if (frm_flags & 0x08) != 0 {
            validate!(br.tell() < end);
            self.repeats = br.read_byte()?.max(1);
        }

        if (frm_flags & 0x20) != 0 {
            let fade_r    = br.read_u16le()?;
            let fade_g    = br.read_u16le()?;
            let fade_b    = br.read_u16le()?;
            let fade_mode = br.read_byte()?;
                            br.read_skip(1)?;
            self.fade.set_state([fade_r, fade_g, fade_b], fade_mode);
            self.fade_start = self.vpts_end;
        }

        if (frm_flags & 0x10) == 0 {
            let has_vframe = (frm_flags & 0x04) != 0;
            let has_pal    = (frm_flags & 0x01) != 0;
            if has_vframe {
                let nops = (if self.is_cda2 { br.read_u32le()? } else { u32::from(br.read_u16le()?) }) as usize;
                validate!(br.tell() < end);
                let raw_size = end - br.tell();
                self.data.resize(raw_size as usize, 0);
                br.read_buf(&mut self.data)?;
                lzss_unpack(&self.data, &mut self.udata, nops)
                        .map_err(|_| DecoderError::InvalidData)?;
                if has_pal {
                    validate!(self.udata.len() > self.pal.len());
                }
                self.draw_frame(has_pal).map_err(|_| DecoderError::InvalidData)?;
                br = &mut self.fr;
            }

            if has_pal {
                if has_vframe {
                    let mut br = MemoryReader::new_read(&self.udata[..768]);
                    br.read_vga_pal(&mut self.pal)?;
                } else {
                    validate!(br.tell() + 768 <= end);
                    br.read_vga_pal(&mut self.pal)?;
                }
                self.fade.reset();
            }
        } else {
            for el in self.frame.iter_mut() {
                *el = 0;
            }
        }

        br.seek(SeekFrom::Start(end))?;
        self.cur_frm += 1;

        self.vpts += TICKS_PER_FRAME;
        self.vpts_end += usize::from(self.repeats);

        if !self.fade.is_active() || self.fade_start >= self.vpts {
            Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
        } else {
            let mut fade_pal = self.fade.update(&self.pal);
            for _ in 0..(self.vpts - self.fade_start - 1).min(TICKS_PER_FRAME - 1) {
                fade_pal = self.fade.update(&self.pal);
            }
            Ok((0, Frame::VideoPal(self.frame.clone(), fade_pal)))
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag = br.read_tag()?;
    validate!(&tag == b"CDA2" || &tag == b"CDAH");
    let is_cda2 = &tag == b"CDA2";

    let nframes = br.read_u32le()? as usize;
    validate!((1..=65535).contains(&nframes));

    let _smth = br.read_u16le()?;
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 320 && (width % 32) == 0);
    validate!(height > 0 && height <= 200 && (height % 40) == 0);

    let mut rate = br.read_u16le()?;
    validate!(rate >= 200);

    let mut arate = br.read_u16le()?;
    validate!(arate == 0 || (8000..=22050).contains(&arate));
    let mut flags = br.read_u16le()?;

    if !is_cda2 {
        rate   = 7000;
        arate  = 22050;
        flags = 0;
    }

    // skip subtitles block if present
    if width == 320 && height == 200 && (flags & 1) != 0 {
        let text_size = br.read_u32le()? as usize;
        br.read_skip(text_size)?;
    }

    // frame CRCs
    if is_cda2 {
        br.read_skip(nframes * 4)?;
    }

    let mut frame_sz = Vec::with_capacity(nframes);
    for _ in 0..nframes {
        let size = br.read_u32le()?;
        validate!(size > 0);
        frame_sz.push(size);
    }

    let ablk_size = arate as usize * 100 * TICKS_PER_FRAME / (rate as usize);

    Ok(Box::new(CDA2Decoder {
        fr: br,
        is_cda2,
        frame: vec![0; width * height],
        data: Vec::new(),
        udata: Vec::new(),
        width, height, rate,
        pal: [0; 768],
        cur_frm: 0,
        frame_sz,
        arate, ablk_size,
        audio: Vec::new(),
        fade: FadeState::new(),
        fade_start: 0,
        repeats: 1,
        apts: 0,
        vpts: 0,
        vpts_end: 0,
    }))
}
