use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

trait ReadEscapedLength {
    fn read_esc_len(&mut self) -> DecoderResult<usize>;
}

impl<T: ?Sized + ByteIO> ReadEscapedLength for T {
    fn read_esc_len(&mut self) -> DecoderResult<usize> {
        let mut len = 32;
        loop {
            let b = usize::from(self.read_byte()?);
            len += b;
            if b != 0xFF {
                break;
            }
        }
        Ok(len)
    }
}

struct ManicKartsDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    width:      usize,
    height:     usize,
    nframes:    u16,
    cur_frm:    u16,
    has_audio:  bool,
    abuf:       Vec<u8>,
    audio:      bool,
}

impl ManicKartsDecoder {
    fn decode_rle(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        let mut pos = 0;
        while br.left() > 0 {
            let op = br.read_byte()?;
            match op {
                0x00..=0x1F => {
                    let len = usize::from(op);
                    validate!(pos + len <= self.frame.len());
                    pos += len;
                },
                0x20 => {
                    let len = br.read_esc_len()?;
                    validate!(pos + len <= self.frame.len());
                    pos += len;
                },
                0x21..=0x3E => {
                    let len = usize::from(op - 0x1F);
                    validate!(pos + len <= self.frame.len());
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                },
                0x3F => {
                    let len = br.read_esc_len()?;
                    validate!(pos + len <= self.frame.len());
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                },
                _ => {
                    validate!(pos < self.frame.len());
                    self.frame[pos] = op;
                    pos += 1;
                },
            }
        }
        Ok(())
    }
}

impl InputSource for ManicKartsDecoder {
    fn get_num_streams(&self) -> usize { if self.has_audio { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 2,
                tb_den: 25,
            }),
            1 if self.has_audio => StreamInfo::Audio(AudioInfo{
                sample_rate: 11025,
                sample_type: AudioSample::U8,
                channels:    1,
            }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        const ABUF_LEN: usize = 882;
        if self.abuf.len() >= ABUF_LEN && self.audio {
            let mut ret = vec![0; ABUF_LEN];
            ret.copy_from_slice(&self.abuf[..ABUF_LEN]);
            self.abuf.drain(..ABUF_LEN);
            self.audio = false;
            return Ok((1, Frame::AudioU8(ret)));
        }
        if self.cur_frm >= self.nframes {
            return Err(DecoderError::EOF);
        }
        let br = &mut self.fr;
        loop {
            let tag = br.read_byte().map_err(|_| DecoderError::EOF)?;
            let size = br.read_u16le()? as usize;
            match tag {
                0x3D if self.has_audio => {
                    br.read_skip(size.min(6))?;
                    if size > 6 {
                        br.read_extend(&mut self.abuf, size - 6)?;
                    }
                },
                0x3D => {
                    br.read_skip(size)?;
                },
                0x3E => {
                    validate!(size >= 4);
                    let offset = usize::from(br.read_byte()?);
                    validate!(offset * 3 + size - 1 <= self.pal.len());
                    br.read_vga_pal_some(&mut self.pal[offset * 3..][..size - 1])?;
                },
                0x3F => {
                    self.data.resize(size, 0);
                    br.read_buf(&mut self.data)?;
                    self.decode_rle().map_err(|_| DecoderError::InvalidData)?;
                    self.cur_frm += 1;
                    self.audio = true;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                _ => return Err(DecoderError::InvalidData),
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let nframes = br.read_u16le()?;
    validate!(nframes > 0);
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!((1..=320).contains(&width) && (1..=200).contains(&height));
    let two = br.read_u16le()?;
    validate!(two == 2);

    let first_type = br.peek_byte()?;
    validate!(matches!(first_type, 0x3D..=0x3F));

    Ok(Box::new(ManicKartsDecoder {
        fr: br,
        pal: [0; 768],
        data: Vec::new(),
        frame: vec![0; width * height],
        width, height,
        nframes,
        cur_frm: 0,
        has_audio: first_type == 0x3D,
        abuf: Vec::new(),
        audio: false,
    }))
}
