use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

const PAL_LEN: usize = 0x180;
const AFRAME_LEN: usize = 1470;

#[derive(Clone,Copy,Default)]
struct Node {
    weight:     u16,
    child0:     u16,
    child1:     u16,
}

fn build_tree(weights: &[u8; 256], nodes: &mut [Node; 514]) -> usize {
    const MAX_NODE: usize = 0x201;
    for (node, &w) in nodes.iter_mut().zip(weights.iter()) {
        node.weight = u16::from(w);
    }
    nodes[0x100].weight = 1;
    nodes[MAX_NODE].weight = 0xFFFF;

    let mut end = 0x101;
    loop {
        let mut minidx1 = MAX_NODE;
        let mut minidx2 = MAX_NODE;
        for (idx, node) in nodes.iter().enumerate().take(end) {
            if node.weight != 0 {
                if node.weight < nodes[minidx1].weight {
                    minidx2 = minidx1;
                    minidx1 = idx;
                } else if node.weight < nodes[minidx2].weight {
                    minidx2 = idx;
                }
            }
        }
        if minidx2 == MAX_NODE {
            break;
        }

        nodes[end].weight = nodes[minidx1].weight + nodes[minidx2].weight;
        nodes[end].child0 = minidx1 as u16;
        nodes[end].child1 = minidx2 as u16;
        nodes[minidx1].weight = 0;
        nodes[minidx2].weight = 0;
        end += 1;
    }
    end - 1
}

struct ArcHufDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    frm_no:     u32,
    nframes:    u32,
    frame:      Vec<u8>,
    data:       Vec<u8>,
    unp_buf:    Vec<u8>,
    nodes:      [Node; 514],
    root:       usize,
    audio:      Vec<u8>,
    audio_out:  bool,
}

impl ArcHufDecoder {
    fn unpack_huf(src: &[u8], dst: &mut Vec<u8>, nodes: &[Node; 514], root: usize) -> DecoderResult<()> {
        dst.clear();
        let mut br = BitReader::new(src, BitReaderMode::BE);
        loop {
            let mut idx = root;
            while idx > 0x100 {
                idx = usize::from(if br.read_bool()? { nodes[idx].child1 } else { nodes[idx].child0 });
            }
            if idx == 0x100 {
                return Ok(());
            }
            dst.push(idx as u8);
        }
    }
    fn decode_intra(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.unp_buf);
        let mut pos = 0;
        loop {
            let op = br.read_byte()?;
            if op < 0x80 {
                validate!(pos < self.frame.len());
                self.frame[pos] = op;
                pos += 1;
            } else {
                let len = usize::from(br.read_byte()?);
                if len == 0 {
                    return Ok(());
                }
                validate!(pos + len <= self.frame.len());
                for el in self.frame[pos..][..len].iter_mut() {
                    *el = op & 0x7F;
                }
                pos += len;
            }
        }
    }
    fn decode_inter(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.unp_buf);
        let mut pos = 0;
        let num_ops = usize::from(br.read_u16le()?);
        for _i in 0..num_ops {
            let op = br.read_byte()?;
            match op {
                0x00 => {
                    let len = usize::from(br.read_u16le()?);
                    validate!(pos + len <= self.frame.len());
                    pos += len;
                },
                0x01..=0x7F => {
                    let copy_len = usize::from(op);
                    let skip_len = usize::from(br.read_byte()?);
                    validate!(pos + copy_len + skip_len <= self.frame.len());
                    br.read_buf(&mut self.frame[pos + skip_len..][..copy_len])?;
                    pos += skip_len + copy_len;
                },
                _ => {
                    let run_len = 256 - usize::from(op);
                    let skip_len = usize::from(br.read_byte()?);
                    validate!(pos + run_len + skip_len <= self.frame.len());
                    let clr = br.read_byte()?;
                    for el in self.frame[pos + skip_len..][..run_len].iter_mut() {
                        *el = clr;
                    }
                    pos += skip_len + run_len;
                },
            }
        }
        Ok(())
    }
}

impl InputSource for ArcHufDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 15,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    channels:    1,
                    sample_type: AudioSample::U8,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.audio_out {
            self.audio_out = false;
            let mut ret = vec![0; AFRAME_LEN];
            ret.copy_from_slice(&self.audio[..AFRAME_LEN]);
            self.audio.drain(..AFRAME_LEN);
            return Ok((1, Frame::AudioU8(ret)));
        }
        if self.frm_no >= self.nframes {
            return Err(DecoderError::EOF);
        }
        let csize = self.fr.read_u32le()? as usize;
        let ctype = self.fr.read_byte()?;
        validate!(csize > PAL_LEN + AFRAME_LEN + 5 && ctype < 2);
        self.data.resize(csize - PAL_LEN - AFRAME_LEN - 5, 0);
        self.fr.read_vga_pal_some(&mut self.pal[..PAL_LEN])?;
        self.fr.read_buf(&mut self.data)?;
        // while it should have EOF symbol coded, in reality this often does not happen
        let _ = Self::unpack_huf(&self.data, &mut self.unp_buf, &self.nodes, self.root);
        if ctype == 0 {
            self.decode_intra().map_err(|_| DecoderError::InvalidData)?;
        } else {
            self.decode_inter().map_err(|_| DecoderError::InvalidData)?;
        }
        self.fr.read_extend(&mut self.audio, AFRAME_LEN)?;
        self.frm_no += 1;
        self.audio_out = true;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let mut tag = [0; 8];
    fr.read_buf(&mut tag)?;
    validate!(&tag == b"[AVF-95]");
    let width = fr.read_u16le()? as usize;
    let height = fr.read_u16le()? as usize;
    validate!((1..=320).contains(&width) && (1..=240).contains(&height));
    let nframes = fr.read_u32le()?;
    validate!(nframes > 0);
    fr.read_skip(9)?;

    let mut weights = [0; 256];
    fr.read_buf(&mut weights)?;
    let mut nodes = [Node::default(); 514];
    let root = build_tree(&weights, &mut nodes);

    let mut audio = vec![0; 44100 * 3];
    fr.read_buf(&mut audio)?;

    Ok(Box::new(ArcHufDecoder {
        fr,
        pal: [0; 768],
        width, height, nframes,
        frm_no: 0,
        data: Vec::new(),
        frame: vec![0; width * height],
        unp_buf: Vec::new(),
        nodes, root,
        audio,
        audio_out: true,
    }))
}
