use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

const RGB_BUF_SIZE: usize = 128000;

struct VideoData {
    width:      usize,
    height:     usize,
    framerate:  u32,
    ydata:      Vec<u8>,
    tmp_y:      Vec<u8>,
    udata:      Vec<u8>,
    vdata:      Vec<u8>,
    data:       Vec<u8>,
    r_tab:      [i32; 256],
    g_tab1:     [i32; 256],
    g_tab2:     [i32; 256],
    b_tab:      [i32; 256],
    rgb_buf:    Vec<u8>,
    buf_pos:    usize,
    is_rgb:     bool,
}

impl VideoData {
    fn read_raw_luma(&mut self, br: &mut dyn ByteIO, size: usize, width: usize, height: usize, y_bits: u8) -> DecoderResult<()> {
        self.data.resize(size, 0);
        br.read_buf(&mut self.data)?;
        let mut ysrc = BitReader::new(&self.data, BitReaderMode::BE);
        for line in self.ydata.chunks_exact_mut(self.width).take(height) {
            for el in line[..width].iter_mut() {
                *el = (ysrc.read(y_bits)? as u8) << (8 - y_bits);
            }
        }
        Ok(())
    }
    fn read_inter_luma(&mut self, br: &mut dyn ByteIO, size: usize, width: usize, height: usize, y_bits: u8) -> DecoderResult<()> {
        self.data.resize(size, 0);
        br.read_buf(&mut self.data)?;

        let mut ysrc = BitReader::new(&self.data, BitReaderMode::BE);
        ysrc.skip(32)?;

        let mut x = 0;
        let mut yoff = 0;
        let stride = self.width;
        let tot_size = stride * height;
        while yoff < tot_size {
            let op = ysrc.read(y_bits)?;
            if op != 0 {
                self.ydata[x + yoff] = self.ydata[x + yoff].wrapping_add((op as u8) << (8 - y_bits));
                x += 1;
                if x == width {
                    x = 0;
                    yoff += stride;
                }
            } else {
                let skip = if ysrc.peek(1) == 0 {
                        ysrc.read(8)? as usize
                    } else {
                        (ysrc.read(16)? - 0x8000) as usize
                    };
                x += skip + 1;
                while x >= width {
                    x -= width;
                    yoff += stride;
                }
                validate!(x + yoff <= tot_size);
            }
        }
        Ok(())
    }
    fn read_chroma(&mut self, br: &mut dyn ByteIO, size: usize, width: usize, height: usize, use_u: bool) -> DecoderResult<()> {
        let dst = if use_u { &mut self.udata } else { &mut self.vdata };
        if size == width * height / 16 {
            for line in dst.chunks_exact_mut(self.width / 4).take(height / 4) {
                br.read_buf(&mut line[..width / 4])?;
            }
        } else {
            self.data.resize(size, 0);
            br.read_buf(&mut self.data)?;

            let mut br = MemoryReader::new_read(&self.data);
            let mut x = 0;
            let mut yoff = 0;
            let stride = self.width / 4;
            let tot_size = stride * (height / 4);
            while br.left() > 0 && yoff < tot_size {
                let op = br.read_byte()?;
                if op != 0 {
                    dst[x + yoff] = dst[x + yoff].wrapping_add(op);
                    x += 1;
                    if x == width / 4 {
                        x = 0;
                        yoff += stride;
                    }
                } else {
                    let skip = if br.peek_byte()? & 0x80 == 0 {
                            usize::from(br.read_byte()?)
                        } else {
                            (br.read_u16be()? as usize) & 0x7FFF
                        };
                    x += skip + 1;
                    while x >= width / 4 {
                        x -= width / 4;
                        yoff += stride;
                    }
                    validate!(x + yoff <= tot_size);
                }
            }
        }
        Ok(())
    }
    fn do_mv(&mut self, br: &mut dyn ByteIO, mask_size: usize, width: usize, height: usize) -> DecoderResult<()> {
        self.data.resize(mask_size, 0);
        br.read_buf(&mut self.data)?;

        std::mem::swap(&mut self.ydata, &mut self.tmp_y);
        let mut br = BitReader::new(&self.data, BitReaderMode::BE);
        for stripe in self.ydata.chunks_exact_mut(width * 8).take(height / 8) {
            for x in (0..width).step_by(8) {
                let yoff = br.read(9)? as usize;
                let xoff = br.read(10)? as usize;
                let src_pos = xoff + yoff * width;
                validate!(src_pos + 8 + 7 * width <= self.tmp_y.len());
                for (dst, src) in stripe[x..].chunks_mut(width)
                        .zip(self.tmp_y[src_pos..].chunks(width)) {
                    dst[..8].copy_from_slice(&src[..8]);
                }
            }
        }
        Ok(())
    }
    fn decode_mode16(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        let mut pos = self.buf_pos;
        let mut flags = 0;
        let mut fbits = 0;
        while br.left() > 0 {
            if fbits == 0 {
                flags = br.read_byte()?;
                fbits = 8;
            }
            let is_raw = (flags & 1) != 0;
            flags >>= 1;
            fbits  -= 1;
            if is_raw {
                br.read_buf(&mut self.rgb_buf[pos * 3..][..3])?;
                pos += 1;
            } else {
                let mut offset = br.read_u16le()? as usize;
                let len = usize::from(br.read_byte()?) + 3;
                if offset > 0x5555 {
                    offset += 0x5555;
                }
                if pos >= offset && pos + len < RGB_BUF_SIZE {
                    for i in 0..(len * 3) {
                        self.rgb_buf[pos * 3 + i] = self.rgb_buf[(pos - offset) * 3 + i];
                    }
                    pos += len;
                } else {
                    let mut src_pos = pos + RGB_BUF_SIZE - offset;
                    if src_pos >= RGB_BUF_SIZE {
                        src_pos -= RGB_BUF_SIZE;
                    }
                    for _ in 0..len {
                        for i in 0..3 {
                            self.rgb_buf[pos * 3 + i] = self.rgb_buf[src_pos * 3 + i];
                        }
                        pos += 1;
                        src_pos += 1;
                        if pos >= RGB_BUF_SIZE {
                            pos = 0;
                        }
                        if src_pos >= RGB_BUF_SIZE {
                            src_pos = 0;
                        }
                    }
                }
            }
        }
        if pos >= RGB_BUF_SIZE {
            pos = 0;
        }
        self.buf_pos = pos;

        Ok(())
    }
    fn make_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if !self.is_rgb {
            let mut frame = vec![0; self.width * self.height * 3];

            let mut uiter = self.udata.chunks_exact(self.width / 4);
            let mut viter = self.vdata.chunks_exact(self.width / 4);
            let mut uline = uiter.next().unwrap();
            let mut vline = viter.next().unwrap();
            for (lineno, (dline, sline)) in frame.chunks_exact_mut(self.width * 3)
                    .zip(self.ydata.chunks_exact(self.width)).enumerate() {
                if lineno != 0 && (lineno & 3) == 0 {
                    uline = uiter.next().unwrap();
                    vline = viter.next().unwrap();
                }
                for (x, (dst, &src)) in dline.chunks_exact_mut(3).zip(sline.iter()).enumerate() {
                    let yval = i32::from(src);
                    let uval = usize::from(uline[x / 4]);
                    let vval = usize::from(vline[x / 4]);
                    let r = yval + ((self.r_tab[uval] + (1 << 16)) >> 16);
                    let b = yval + ((self.b_tab[vval] + (1 << 16)) >> 16);
                    let g = yval + ((-self.g_tab1[uval] - self.g_tab2[vval] + (1 << 16)) >> 16);
                    dst[2] = r.max(0).min(255) as u8;
                    dst[1] = g.max(0).min(255) as u8;
                    dst[0] = b.max(0).min(255) as u8;
                }
            }

            Ok((0, Frame::VideoRGB24(frame)))
        } else {
            let img_size = self.width * self.height;
            let frame = if self.buf_pos >= img_size {
                    self.rgb_buf[(self.buf_pos - img_size) * 3..][..img_size * 3].to_vec()
                } else {
                    let mut frm = vec![0; img_size * 3];
                    let (head, tail) = frm.split_at_mut((img_size - self.buf_pos) * 3);
                    tail.copy_from_slice(&self.rgb_buf[..self.buf_pos * 3]);
                    head.copy_from_slice(&self.rgb_buf[(RGB_BUF_SIZE - img_size + self.buf_pos) * 3..]);
                    frm
                };
            Ok((0, Frame::VideoRGB24(frame)))
        }
    }
}

struct AsconDecoder {
    fr:         FileReader<BufReader<File>>,
    vdata:      VideoData,
    abuf:       Vec<u8>,
    audio:      bool,
    arate:      u32,
    channels:   u8,
    bits:       u8,
    aframe_len: usize,
}

impl InputSource for AsconDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.vdata.width,
                    height: self.vdata.height,
                    bpp:    24,
                    tb_num: 100,
                    tb_den: self.vdata.framerate,
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    self.channels,
                    sample_type: if self.bits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        loop {
            if self.audio && self.abuf.len() >= self.aframe_len {
                self.audio = false;
                if self.bits == 8 {
                    let mut audio = vec![0; self.aframe_len];
                    audio.copy_from_slice(&self.abuf[..self.aframe_len]);
                    self.abuf.drain(..self.aframe_len);
                    return Ok((1, Frame::AudioU8(audio)));
                } else {
                    let mut audio = vec![0; self.aframe_len / 2];
                    for (dst, src) in audio.iter_mut().zip(self.abuf.chunks_exact(2)) {
                        *dst = read_u16le(src).unwrap_or(0) as i16;
                    }
                    self.abuf.drain(..self.aframe_len);
                    return Ok((1, Frame::AudioS16(audio)));
                }
            }

            let tag = br.read_tag()?;
            let size = br.read_u32le()? as usize;
            match &tag {
                b"BILD" => {
                    let end = br.tell() + (size as u64);
                    validate!(size > 34);
                    let part0_size = br.read_u32le()?;
                    let hdr_size = br.read_u32le()? as usize;
                    validate!(part0_size == 0);
                    validate!(hdr_size < size);
                    let width = br.read_u16le()? as usize;
                    let height = br.read_u16le()? as usize;
                    validate!((width | height) & 3 == 0);
                    validate!(width <= self.vdata.width && height <= self.vdata.height);
                    let mode = br.read_u16le()?;
                    let y_size = br.read_u32le()? as usize;
                    let u_size = br.read_u32le()? as usize;
                    let v_size = br.read_u32le()? as usize;
                    br.read_u32le()?;
                    br.read_u32le()?;
                    let mut y_bits = 0;
                    if hdr_size < 0x2E {
                        br.read_skip(hdr_size - 0x22)?;
                    } else {
                        br.read_skip(11)?;
                        y_bits = br.read_byte()?;
                        validate!(matches!(y_bits, 0 | 5 | 6));
                        br.read_skip(hdr_size - 0x2E)?;
                    }
                    if y_bits == 0 {
                        y_bits = 5;
                    }
                    validate!(size >= hdr_size + y_size + u_size + v_size);
                    self.vdata.is_rgb = mode == 16;
                    match mode {
                        2 => {
                            validate!(y_size >= width * height * usize::from(y_bits) / 8);
                            validate!(u_size == v_size && u_size == width * height / 16);
                            self.vdata.read_raw_luma(&mut *br, y_size, width, height, y_bits)?;
                            self.vdata.read_chroma(&mut *br, u_size, width, height, true)?;
                            self.vdata.read_chroma(&mut *br, v_size, width, height, false)?;
                        },
                        3 => {
                            self.vdata.read_inter_luma(&mut *br, y_size, width, height, y_bits)
                                    .map_err(|_| DecoderError::InvalidData)?;
                            self.vdata.read_chroma(&mut *br, u_size, width, height, true)
                                    .map_err(|_| DecoderError::InvalidData)?;
                            self.vdata.read_chroma(&mut *br, v_size, width, height, false)
                                    .map_err(|_| DecoderError::InvalidData)?;
                        },
                        4 => {
                            validate!(((width | height) & 7) == 0);
                            let mask_size = (width * height / 64 * 19 + 7) / 8;
                            validate!(size >= hdr_size + y_size + u_size + v_size + mask_size);
                            self.vdata.do_mv(&mut *br, mask_size, width, height)?;
                            self.vdata.read_inter_luma(&mut *br, y_size, width, height, y_bits)
                                    .map_err(|_| DecoderError::InvalidData)?;
                            self.vdata.read_chroma(&mut *br, u_size, width, height, true)
                                    .map_err(|_| DecoderError::InvalidData)?;
                            self.vdata.read_chroma(&mut *br, v_size, width, height, false)
                                    .map_err(|_| DecoderError::InvalidData)?;
                        },
                        16 => {
                            validate!(width * height <= self.vdata.rgb_buf.len());
                            self.vdata.data.resize(y_size, 0);
                            br.read_buf(&mut self.vdata.data)?;
                            self.vdata.decode_mode16().map_err(|_| DecoderError::InvalidData)?;
                        },
                        _ => return Err(DecoderError::NotImplemented),
                    }
                    validate!(br.tell() <= end);
                    br.seek(SeekFrom::Start(end))?;

                    self.audio = true;

                    return self.vdata.make_frame();
                },
                b"ENDE" => return Err(DecoderError::EOF),
                b"SAMP" => {
                    validate!(size >= 15);
                    let _crc = br.read_u32le()?;
                    let hdr_size = br.read_u32le()? as usize;
                    validate!(hdr_size <= size);
                    let arate = br.read_u24le()?;
                    let channels = br.read_byte()?;
                    let bits = br.read_byte()?;
                    validate!(arate == self.arate && channels == self.channels && bits == self.bits);
                    br.read_skip(hdr_size - 13)?;
                    let audio_len = size - hdr_size;
                    if audio_len > 0 {
                        br.read_extend(&mut self.abuf, audio_len)?;
                    }
                },
                _ => br.read_skip(size)?,
            }
        }
    }
}

fn generate_tab(start: i32, step: i32) -> [i32; 256] {
    let mut val = start;
    let mut tab = [0; 256];
    for el in tab.iter_mut() {
        *el = val;
        val += step;
    }
    tab
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag = br.read_tag()?;
    validate!(&tag == b"AN15");
    let size = br.read_u32le()? as usize;
    validate!(size >= 64);
    let part0_size = br.read_u32le()?;
    let part1_size = br.read_u32le()? as usize;
    validate!(part0_size == 0 && part1_size == size);
    let _tag = br.read_tag()?; // only "V0.9" is known
    let _num_vframes = br.read_u32le()?;
    let _num_aframes = br.read_u32le()?;
    br.read_u32le()?;
    let _end_offset = br.read_u32le()?;
    let _offset = br.read_u32le()?;
    br.read_u32le()?;
    br.read_u32le()?;
    br.read_u32le()?;
    let _val_0xbba2 = br.read_u32le()?;
    br.read_u24le()?;
    let _smth = br.read_u32le()?;
    br.read_byte()?;
    let framerate = br.read_u32le()?;
    validate!((100..=30000).contains(&framerate));
    let _smth3 = br.read_u32le()?;
    br.read_skip(part1_size - 0x40)?;

    let r_tab  = generate_tab(-0xB37480, 0x166E9);
    let g_tab1 = generate_tab(-0x2C0D00, 0x581A);
    let g_tab2 = generate_tab(-0x5B6900, 0xB6D2);
    let b_tab  = generate_tab(-0xE2D100, 0x1C5A2);

    let mut width = 0;
    let mut height = 0;
    let mut arate = 0;
    let mut channels = 0;
    let mut bits = 0;

    let start_pos = br.tell();
    // scan for audio/video parameters
    let mut video_found = false;
    let mut audio_found = false;
    let mut vcounter = 0;
    while !video_found || !audio_found {
        let tag = br.read_tag()?;
        let size = br.read_u32le()? as usize;
        match &tag {
            b"BILD" if !video_found => {
                validate!(size > 34);
                let part0_size = br.read_u32le()?;
                let hdr_size = br.read_u32le()? as usize;
                validate!(part0_size == 0);
                validate!(hdr_size < size);
                width = br.read_u16le()? as usize;
                height = br.read_u16le()? as usize;
                validate!((width | height) & 3 == 0);
                validate!((4..=640).contains(&width) && (4..=480).contains(&height));
                br.read_skip(size - 12)?;
                video_found = true;
            },
            b"BILD" => {
                vcounter += 1;
                if vcounter > 4 {
                    break;
                }
                br.read_skip(size)?;
            },
            b"SAMP" if !audio_found => {
                validate!(size >= 15);
                let _crc = br.read_u32le()?;
                let hdr_size = br.read_u32le()? as usize;
                validate!(hdr_size <= size);
                arate = br.read_u24le()?;
                channels = br.read_byte()?;
                bits = br.read_byte()?;
                validate!((8000..=32000).contains(&arate));
                validate!(channels == 1 || channels == 2);
                validate!(bits == 8 || bits == 16);
                br.read_skip(size - 13)?;
                audio_found = true;
            },
            b"ENDE" => break,
            _ => br.read_skip(size)?,
        }
    }
    validate!(video_found);
    br.seek(SeekFrom::Start(start_pos))?;

    let bsize = usize::from(channels) * usize::from(bits / 8);
    Ok(Box::new(AsconDecoder {
        fr: br,
        vdata: VideoData {
            data: Vec::new(),
            width, height, framerate,
            ydata: vec![0; width * height],
            tmp_y: vec![0; width * height],
            udata: vec![0; width * height / 16],
            vdata: vec![0; width * height / 16],
            r_tab, g_tab1, g_tab2, b_tab,
            rgb_buf: vec![0; RGB_BUF_SIZE * 3],
            buf_pos: 0,
            is_rgb: false,
        },
        abuf: Vec::with_capacity(32000),
        audio: false,
        aframe_len: (arate * 100 / framerate) as usize * bsize,
        arate, channels, bits,
    }))
}
