use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

const BPP: usize = 4;

const HEADER_SIZE: usize = 0x2F;

const MV_2BIT: [(i8, i8); 4] = [(-1, 0), (-1, -1), (1, -1), (0, -2)];
const MV_4BIT: [(i8, i8); 16] = [
    (-2, -3), ( 2, -3), (-1, -4), ( 1, -4),
    (-1, -2), ( 1, -2), ( 0, -3), ( 0, -4),
    (-2,  0), (-2, -1), ( 2, -1), (-2, -2),
    ( 2, -2), (-1, -3), ( 1, -3), ( 0, -5)
];

struct CNMDecoder {
    fr:         FileReader<BufReader<File>>,
    nframes:    usize,
    cur_vfrm:   usize,
    cur_afrm:   usize,
    width:      usize,
    height:     usize,
    fps:        u32,
    interlaced: bool,
    frame:      Vec<usize>,
    vdata:      Vec<u8>,
    tiles:      Vec<u8>,
    tile_size:  usize,
    ntiles:     usize,
    arate:      u32,
    channels:   u8,
    bits:       u8,
    audio8:     Vec<u8>,
    audio16:    Vec<i16>,
}

impl CNMDecoder {
    fn unpack_frame(&mut self) -> DecoderResult<Vec<u8>> {
        let src = &self.vdata;
        let mut br = MemoryReader::new_read(src);

        let size                            = br.read_u32le()? as usize;
        validate!(src.len() >= size + HEADER_SIZE);
        let part2_off                       = br.read_u32le()? as u64;
        validate!(part2_off > 0 && part2_off < (size as u64));
        let num_tiles                       = br.read_u16le()? as usize;
        validate!(num_tiles > 0 && num_tiles < 4096);
        let tile_size                       = br.read_u16le()? as usize;
        let width                           = br.read_u32le()? as usize;
        let height                          = br.read_u32le()? as usize;

        validate!(width == self.width);
        validate!(height == self.height);
        self.tiles.resize(tile_size * num_tiles * BPP, 0);

                                              br.seek(SeekFrom::Start(part2_off + (HEADER_SIZE as u64)))?;
        let tile_w = if tile_size == 2 { 2 } else { 4 };
        let tsize = tile_w * BPP;

        match tile_size {
            2 | 4 => {
                                              br.read_buf(&mut self.tiles[..tsize])?;
                let off = br.tell() as usize;
                let mut bir = BitReader::new(&src[off..], BitReaderMode::BE);
                for tile in 1..num_tiles {
                    for i in 0..tsize {
                        self.tiles[tile * tsize + i] = self.tiles[tile * tsize + i - tsize];
                    }
                    let bits                = bir.read(3)? as u8 + 1;
                    validate!(bits < 8);
                    for el in self.tiles[tile * tsize..][..tsize].iter_mut() {
                        let mut delta       = bir.read(bits)? as i16;
                        if delta != 0 && bir.read_bool()? {
                            delta = -delta;
                        }
                        *el = (i16::from(*el) + delta) as u8;
                    }
                }
            },
            _ => {
                validate!(tile_size == num_tiles * tsize);
                                              br.read_buf(&mut self.tiles[..tile_size])?;
            },
        };

        let stride = width / tile_w;
        self.frame.resize(stride * (height / 2), 0);

        let mut br = BitReader::new(&src[HEADER_SIZE..], BitReaderMode::BE);
        let idx_bits = if num_tiles < 0x400 { 10 } else if num_tiles < 0x800 { 11 } else { 12 };
        for y in 0..(height / 2) {
            for x in 0..stride {
                let dst_pos = x + y * stride;
                if !br.read_bool()? {
                    let idx             = br.read(idx_bits)? as usize;
                    validate!(idx < num_tiles);
                    self.frame[dst_pos] = idx;
                } else {
                    let (mv_x, mv_y) = if br.read_bool()? {
                            (0, -1)
                        } else if br.read_bool()? {
                            MV_2BIT[br.read(2)? as usize]
                        } else {
                            MV_4BIT[br.read(4)? as usize]
                        };

                    let isrc = (dst_pos as isize) + isize::from(mv_x) + isize::from(mv_y) * (stride as isize);
                    validate!(isrc >= 0);
                    let src_pos = isrc as usize;
                    validate!(src_pos < self.frame.len());
                    self.frame[dst_pos] = self.frame[src_pos];
                }
            }
        }

        let dstride = self.width * 3;
        let mut dframe = vec![0; dstride * self.height];
        for (drows, srow) in dframe.chunks_exact_mut(dstride * 2).rev().zip(self.frame.chunks_exact(stride)) {
            for (dst, &idx) in drows.chunks_exact_mut(tile_w * 3).zip(srow.iter()) {
                for (dpix, tpix) in dst.chunks_exact_mut(3).zip(self.tiles[idx * tsize..].chunks_exact(BPP)) {
                    dpix[0] = tpix[2];
                    dpix[1] = tpix[1];
                    dpix[2] = tpix[0];
                }
            }
            // double lines
            let (line0, line1) = drows.split_at_mut(dstride);
            line1.copy_from_slice(line0);
        }

        Ok(dframe)
    }
}

impl InputSource for CNMDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    24,
                tb_num: 100,
                tb_den: self.fps,
             })
        } else if stream_no == 1 && self.arate > 0 {
            StreamInfo::Audio(AudioInfo{
                sample_rate: self.arate,
                sample_type: if self.bits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                channels:    self.channels,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        loop {
            if self.cur_vfrm >= self.nframes && (self.arate == 0 || self.cur_afrm >= self.nframes) {
                return Err(DecoderError::EOF);
            }

            let ftype = br.read_byte()?;
            match ftype {
                0x41 | 0x42 | 0x5A => {
                    validate!(self.arate > 0);
                    let asize = br.read_u32le()? as usize;
                    let align = usize::from(self.channels * self.bits / 8);
                    let tail = asize % align;

                    self.cur_afrm += 1;

                    if asize <= tail {
                        br.read_skip(tail)?;
                        continue;
                    }

                    if self.bits == 8 {
                        let mut ret = vec![0; asize - tail];
                        br.read_buf(&mut ret)?;
                        br.read_skip(tail)?;
                        return Ok((1, Frame::AudioU8(ret)));
                    } else {
                        let mut ret = vec![0; (asize - tail) / 2];
                        for el in ret.iter_mut() {
                            *el = br.read_u16le()? as i16;
                        }
                        br.read_skip(tail)?;
                        return Ok((1, Frame::AudioS16(ret)));
                    }
                },
                0x53 => {
                    let size = br.peek_u32le()? as usize;
//println!("frame type {ftype:02X} size {size:X} @ {:X}", br.tell() - 5);
                    self.vdata.resize(size + HEADER_SIZE, 0);
                    br.read_buf(&mut self.vdata)?;

                    let frame = self.unpack_frame().map_err(|_| DecoderError::InvalidData)?;

                    self.cur_vfrm += 1;

                    return Ok((0, Frame::VideoRGB24(frame)));
                },
                _ => unimplemented!(),
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let mut magic = [0; 8];
    br.read_buf(&mut magic)?;
    validate!(&magic == b"CNM UNR\0");

    let nframes = br.read_u32le()? as usize;
    validate!(nframes > 0);
    let fps = br.read_u32le()?;
    validate!(fps > 100 && fps < 100 * 100);
    br.read_byte()?;
    let width = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!(width > 0 && width <= 1280 && height > 0 && height <= 960);
    br.read_skip(1)?;
    let interlaced = br.read_byte()? != 0;
    let naudio = br.read_byte()?;
    if naudio > 1 {
        return Err(DecoderError::NotImplemented);
    }
    let vframes = br.read_u32le()? as usize;
    validate!(vframes == nframes);
    let aframes = br.read_u32le()? as usize;
    validate!((naudio == 0 && aframes == 0) || (aframes == nframes));

    br.seek(SeekFrom::Start(0xC0))?;

    let (arate, channels, bits) = if naudio > 0 {
            let channels = br.read_byte()? + 1;
            validate!(channels == 1 || channels == 2);
            let bits = br.read_byte()?;
            validate!(bits == 8 || bits == 16);
            let arate = u32::from(br.read_u16le()?);
            validate!(arate > 2000 && arate <= 48000);
            br.read_skip(12)?;
            (arate, channels, bits)
        } else {
            (0, 0, 0)
        };
    br.read_skip(vframes * 4)?;
    br.read_skip(aframes * 4)?;

    Ok(Box::new(CNMDecoder {
        fr: br,
        width, height, fps, interlaced, nframes,
        cur_vfrm:   0,
        frame:      Vec::new(),
        vdata:      Vec::new(),
        tiles:      Vec::new(),
        tile_size:  0,
        ntiles:     0,
        cur_afrm:   0,
        audio8:     Vec::new(),
        audio16:    Vec::new(),
        arate, channels, bits,
    }))
}
