use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use crate::input::util::lzss::lz_copy;
use super::super::*;

const DEF_WIDTH:  usize = 640;
const DEF_HEIGHT: usize = 320;
const AFRAME_LEN: usize = 1471;

struct PacoDecoder {
    fr:         FileReader<BufReader<File>>,
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    is_16bit:   bool,
    vdata:      Vec<u8>,
    tmp:        Vec<u8>,
    has_audio:  bool,
    arate:      u32,
    channels:   u8,
    audio:      Vec<i16>,
}

fn decode_lz_code(br: &mut BitReader, dst: &mut [u8], prev: &[u8], pos: usize, allow_prev: bool) -> DecoderResult<usize> {
    let mut tok = br.read(7)?;
    if tok < 0x18 {
        tok += 0x100;
    } else {
        tok = (tok << 1) | br.read(1)?;
        if tok < 0xC0 {
            tok -= 0x30;
        } else if tok < 0xC8 {
            tok += 0x58;
        } else {
            tok = tok * 2 + br.read(1)? - 0x100;
        }
    }

    match tok {
        0x00..=0xFF => {
            dst[pos] = tok as u8;
            Ok(1)
        },
        0x100..=0x11D => {
            let copy_len = (match tok {
                    0x100..=0x108 => tok - 0xFE,
                    0x109..=0x11C => {
                        let bits = ((tok - 0x105) >> 2) as u8;
                        let base = ((tok - 0x105) & 3) + 4;
                        (base << bits) + br.read(bits)? + 3
                    },
                    _ => 0x102,
                }) as usize;
            let mut offset = br.read(5)?;
            if offset >= 4 {
                let bits = ((offset - 2) >> 1) as u8;
                offset = (((offset & 1) + 2) << bits) + br.read(bits)?;
            }
            let offset = (offset + 1) as usize;
            validate!(pos >= offset && pos + copy_len <= dst.len());
            lz_copy(dst, pos, offset, copy_len);
            Ok(copy_len)
        },
        0x11E if allow_prev => {
            let len = br.read(6)? as usize + 3;
            validate!(pos + len <= dst.len());
            dst[pos..][..len].copy_from_slice(&prev[pos..][..len]);
            Ok(len)
        },
        _ => Err(DecoderError::InvalidData),
    }
}

fn decode_mode2(src: &[u8], dst: &mut [u8], prev: &[u8], tmp: &mut [u8], is_16bit: bool) -> DecoderResult<()> {
    let mut br = BitReader::new(src, BitReaderMode::BE);
    let mut read_pos = 0;
    let mut write_pos = 0;

    const MAX_OFFSET: usize = (DEF_WIDTH - 8) * (DEF_HEIGHT - 8);
    let (stride, blk_w) = if is_16bit { (DEF_WIDTH * 2, 8 * 2) } else { (DEF_WIDTH, 8) };
    for stripe in dst.chunks_exact_mut(stride * 8) {
        for x in (0..stride).step_by(blk_w) {
            while read_pos + 3 > write_pos {
                let len = decode_lz_code(&mut br, tmp, prev, write_pos, false)?;
                write_pos += len;
            }
            let blk_offs = read_u24le(&tmp[read_pos..])? as usize;
            read_pos += 3;
            if blk_offs < MAX_OFFSET {
                let xoff = blk_offs % (DEF_WIDTH - 8);
                let yoff = blk_offs / (DEF_WIDTH - 8);
                let xoff2 = if is_16bit { xoff * 2 } else { xoff };
                let prev_src = &prev[xoff2 + yoff * stride..];
                for (dline, sline) in stripe[x..].chunks_mut(stride)
                        .zip(prev_src.chunks(stride)).take(8) {
                    dline[..blk_w].copy_from_slice(&sline[..blk_w]);
                }
            } else {
                while read_pos + blk_w * 8 > write_pos {
                    let len = decode_lz_code(&mut br, tmp, prev, write_pos, false)?;
                    write_pos += len;
                }
                for (dline, sline) in stripe[x..].chunks_mut(stride)
                        .zip(tmp[read_pos..].chunks_exact(blk_w)).take(8) {
                    dline[..blk_w].copy_from_slice(sline);
                }
                read_pos += blk_w * 8;
            }
        }
    }
    Ok(())
}

fn decode_mode3(src: &[u8], dst: &mut [u8], prev: &[u8]) -> DecoderResult<()> {
    let mut br = BitReader::new(src, BitReaderMode::BE);
    let mut pos = 0;

    while pos < dst.len() {
        let len = decode_lz_code(&mut br, dst, prev, pos, true)?;
        pos += len;
    }
    Ok(())
}

impl InputSource for PacoDecoder {
    fn get_num_streams(&self) -> usize { if self.has_audio { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    if self.is_16bit { 16 } else { 8 },
                    tb_num: 1471,
                    tb_den: 22050,
                 }),
            1 if self.has_audio => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    sample_type: AudioSample::S16,
                    channels:    self.channels,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        let aframe_len = AFRAME_LEN * usize::from(self.channels);
        loop {
            if self.has_audio && self.audio.len() >= aframe_len {
                let mut ret = vec![0; aframe_len];
                ret.copy_from_slice(&self.audio[..aframe_len]);
                self.audio.drain(..aframe_len);
                return Ok((1, Frame::AudioS16(ret)));
            }

            let ctype = br.read_byte().map_err(|_| DecoderError::EOF)?;
            let size = br.read_u32le()? as usize;

            match ctype {
                1 => return Err(DecoderError::InvalidData),
                2 | 3 => {
                    self.vdata.resize(size, 0);
                    br.read_buf(&mut self.vdata)?;
                    std::mem::swap(&mut self.frame, &mut self.pframe);
                    let dst_frm = if self.is_16bit {
                            &mut self.frame
                        } else {
                            &mut self.frame[..self.width * self.height]
                        };
                    if ctype == 2 {
                        decode_mode2(&self.vdata, dst_frm, &self.pframe, &mut self.tmp, self.is_16bit)
                                .map_err(|_| DecoderError::InvalidData)?;
                    } else {
                        decode_mode3(&self.vdata, dst_frm, &self.pframe)
                                .map_err(|_| DecoderError::InvalidData)?;
                    }

                    if self.is_16bit {
                        let mut frm16 = vec![0; self.width * self.height];
                        for (dst, src) in frm16.iter_mut().zip(self.frame.chunks_exact(2)) {
                            *dst = read_u16le(src).unwrap_or(0);
                        }
                        return Ok((0, Frame::VideoRGB16(frm16)));
                    } else {
                        return Ok((0, Frame::VideoPal(dst_frm.to_vec(), self.pal)));
                    }
                },
                4 => {
                    validate!(!self.is_16bit);
                    validate!(size <= self.pal.len() && (size % 3) == 0);
                    br.read_buf(&mut self.pal[..size])?;
                },
                5 => { // audio
                    validate!((size & 1) == 0);
                    self.audio.reserve(size / 2);
                    for _ in 0..size / 2 {
                        self.audio.push(br.read_u16le()? as i16);
                    }
                },
                20 => { // silence
                    validate!(size == 4);
                    let silence_len = br.read_u32le()? as usize;
                    validate!((silence_len & 1) == 0);
                    self.audio.reserve(silence_len / 2);
                    for _ in 0..silence_len / 2 {
                        self.audio.push(0);
                    }
                },
                _ => br.read_skip(size)?,
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag = br.read_tag()?;
    validate!(&tag == b"Paco");
    let chunk = br.read_byte()?;
    validate!(chunk == 1);
    let len = br.read_u32le()?;
    validate!(len == 7);

    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 1024 && height > 0 && height <= 768);
    if width != DEF_WIDTH || height != DEF_HEIGHT {
        return Err(DecoderError::NotImplemented);
    }
    let nframes = br.read_u16le()?;
    validate!(nframes > 0);
    let has_audio = br.read_byte()? != 0;

    let mut is_16bit = true;
    let mut channels = if has_audio { 2 } else { 1 };
    let mut arate    = if has_audio { 22050 } else { 0 };

    let start = br.tell();
    // Scan first chunks for palette and audio quirks
    let mut first_audio = true;
    loop {
        let ret = br.read_byte();
        if ret.is_err() { break; }
        let ctype = ret.unwrap();
        let csize = br.read_u32le()?;
        match ctype {
            2 | 3 => break,
            4 => is_16bit = false,
            5 if first_audio => {
                match csize {
                    22050 => {
                        channels = 1;
                        arate = 11025;
                    },
                    44100 => channels = 1,
                    _ => {},
                }
                first_audio = false;
            },
            _ => {},
        }
        br.read_skip(csize as usize)?;
    }
    br.seek(SeekFrom::Start(start))?;

    Ok(Box::new(PacoDecoder {
        fr: br,
        vdata: Vec::new(),
        frame: vec![0; width * height * 2],
        pframe: vec![0; width * height * 2],
        tmp: vec![0; width * height * 2 + (width / 8) * (height / 8) * 3],
        width, height, is_16bit,
        pal: [0; 768],
        has_audio, arate, channels,
        audio: Vec::with_capacity(22050 * 2),
    }))
}
