use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;
const BLOCK_SIZE: usize = 0xAB15;
const AUDIO_OFFSET: usize = 0x9D4D;

enum State {
    Block,
    Audio,
    Frame2,
}

struct DMVDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    tiles:      [u8; WIDTH * HEIGHT / 4],
    buf:        [u8; BLOCK_SIZE],
    frame2:     Option<Vec<u8>>,
    state:      State,
}

enum LZOp {
    Raw(usize),
    Copy(usize, usize),
}

impl DMVDecoder {
    fn unpack(src: &[u8], dst: &mut [u8; WIDTH * HEIGHT / 4]) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(src);
        let mut pos = 0;
        while pos < dst.len() {
            let b = br.read_byte()?;
            let op = b >> 5;
            let count = usize::from(b & 0x1F);
            let op = match op {
                    0 => {
                        LZOp::Raw(count)
                    },
                    2 => {
                        LZOp::Copy(usize::from(br.read_byte()?), count)
                    },
                    3 => {
                        let copy_hi = usize::from(br.read_byte()?);
                        let offset = usize::from(br.read_byte()?);
                        LZOp::Copy(offset, copy_hi * 0x20 + count)
                    },
                    4 => {
                        let offset = usize::from(br.read_u16le()?);
                        LZOp::Copy(offset, count)
                    },
                    5 => {
                        let copy_hi = usize::from(br.read_byte()?);
                        let offset = usize::from(br.read_u16le()?);
                        LZOp::Copy(offset, copy_hi * 0x20 + count)
                    },
                    _ => {
                        let copy_hi = usize::from(br.read_byte()?);
                        LZOp::Raw(copy_hi * 0x20 + count)
                    }
                };
            match op {
                LZOp::Raw(len) => {
                    let raw_len = len + 1;
                    validate!(pos + raw_len <= dst.len());
                    br.read_buf(&mut dst[pos..][..raw_len])?;
                    pos += raw_len;
                },
                LZOp::Copy(offset, len) => {
                    let copy_len = len + 4;
                    validate!(pos >= offset + copy_len);
                    validate!(pos + copy_len <= dst.len());
                    for _ in 0..copy_len {
                        dst[pos] = dst[pos - offset - copy_len];
                        pos += 1;
                    }
                },
            }
        }

        Ok(())
    }
    fn draw_frame(&mut self, clr_size: usize) -> DecoderResult<Vec<u8>> {
        let mut frame0 = vec![0; WIDTH * HEIGHT];
        let mut frame1 = vec![0; WIDTH * HEIGHT];

        let clrs = &self.buf[0x301..][..clr_size];

        let mut clr_pos = 0;
        for ((dstrip0, dstrip1), src) in frame0.chunks_exact_mut(WIDTH * 2)
                .zip(frame1.chunks_exact_mut(WIDTH * 2)).zip(self.tiles.chunks_exact(WIDTH / 2)) {
            let (dline0, dline1) = dstrip0.split_at_mut(WIDTH);
            let (dline2, dline3) = dstrip1.split_at_mut(WIDTH);
            for (((d0, d1), (d2, d3)), &b) in dline0.chunks_exact_mut(2)
                    .zip(dline1.chunks_exact_mut(2))
                        .zip(dline2.chunks_exact_mut(2)
                            .zip(dline3.chunks_exact_mut(2))).zip(src.iter()) {
                let idx = usize::from(b.wrapping_sub(1)); // indices should be in range 1..=154
                validate!(idx < TILE_CLRS.len());
                let oct = &TILE_IDC[idx];
                let tile = &clrs[clr_pos..];
                clr_pos += usize::from(TILE_CLRS[idx]);
                validate!(clr_pos <= clrs.len());

                d0[0] = tile[usize::from(oct[0])]; d0[1] = tile[usize::from(oct[1])];
                d1[0] = tile[usize::from(oct[2])]; d1[1] = tile[usize::from(oct[3])];
                d2[0] = tile[usize::from(oct[4])]; d2[1] = tile[usize::from(oct[5])];
                d3[0] = tile[usize::from(oct[6])]; d3[1] = tile[usize::from(oct[7])];
            }
        }

        self.frame2 = Some(frame1);

        Ok(frame0)
    }
}

impl InputSource for DMVDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 2,
                    tb_den: 25,
                }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    sample_type: AudioSample::U8,
                    channels:    1,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        match self.state {
            State::Block => {
                self.state = State::Audio;

                self.fr.read_buf(&mut self.buf).map_err(|_| DecoderError::EOF)?;

                for (dst, &src) in self.pal[3..].iter_mut().zip(self.buf[4..0x301].iter()) {
                    *dst = (src << 2) | (src >> 4);
                }
                let clr_size = usize::from(read_u16le(&self.buf)?);
                let idc_size = usize::from(read_u16le(&self.buf[2..])?);
                validate!(clr_size + idc_size + 0x301 <= AUDIO_OFFSET);
                Self::unpack(&self.buf[0x301 + clr_size..][..idc_size], &mut self.tiles)
                    .map_err(|_| DecoderError::InvalidData)?;
                let frame = self.draw_frame(clr_size).map_err(|_| DecoderError::InvalidData)?;
                Ok((0, Frame::VideoPal(frame, self.pal)))
            },
            State::Audio => {
                self.state = State::Frame2;

                const BLK_SIZE: usize = BLOCK_SIZE - AUDIO_OFFSET;
                let audio = self.buf[AUDIO_OFFSET..].to_vec();
                Ok((1, Frame::AudioU8(audio)))
            },
            State::Frame2 => {
                self.state = State::Block;

                let mut frm = None;
                std::mem::swap(&mut frm, &mut self.frame2);
                Ok((0, Frame::VideoPal(frm.unwrap(), self.pal)))
            },
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let fr = FileReader::new_read(BufReader::new(file));

    Ok(Box::new(DMVDecoder {
        fr,
        pal: std::array::from_fn(|i| (i / 3) as u8),//[0; 768],
        tiles: [0; WIDTH * HEIGHT / 4],
        frame2: None,
        buf: [0; BLOCK_SIZE],
        state: State::Block,
    }))
}

static TILE_IDC: [[u8; 8]; 154] = [
   [0, 0, 0, 0, 0, 0, 0, 0], [0, 0, 0, 0, 1, 1, 1, 1],
   [0, 0, 1, 1, 0, 0, 1, 1], [0, 1, 0, 1, 0, 1, 0, 1],
   [0, 0, 0, 0, 1, 1, 2, 2], [0, 0, 0, 0, 1, 2, 1, 2],
   [0, 0, 1, 1, 0, 0, 2, 2], [0, 0, 1, 1, 2, 2, 1, 1],
   [0, 0, 1, 1, 2, 2, 2, 2], [0, 0, 1, 2, 0, 0, 1, 2],
   [0, 1, 0, 1, 0, 2, 0, 2], [0, 1, 0, 1, 2, 1, 2, 1],
   [0, 1, 0, 1, 2, 2, 2, 2], [0, 1, 0, 2, 0, 1, 0, 2],
   [0, 1, 2, 1, 0, 1, 2, 1], [0, 1, 2, 2, 0, 1, 2, 2],
   [0, 0, 0, 0, 1, 1, 2, 3], [0, 0, 0, 0, 1, 2, 1, 3],
   [0, 0, 0, 0, 1, 2, 3, 2], [0, 0, 0, 0, 1, 2, 3, 3],
   [0, 0, 1, 1, 0, 0, 2, 3], [0, 0, 1, 1, 2, 2, 3, 3],
   [0, 0, 1, 1, 2, 3, 1, 1], [0, 0, 1, 1, 2, 3, 2, 3],
   [0, 0, 1, 2, 0, 0, 1, 3], [0, 0, 1, 2, 0, 0, 3, 2],
   [0, 0, 1, 2, 0, 0, 3, 3], [0, 0, 1, 2, 3, 3, 1, 2],
   [0, 0, 1, 2, 3, 3, 3, 3], [0, 1, 0, 1, 0, 2, 0, 3],
   [0, 1, 0, 1, 2, 1, 3, 1], [0, 1, 0, 1, 2, 2, 3, 3],
   [0, 1, 0, 1, 2, 3, 2, 3], [0, 1, 0, 2, 0, 1, 0, 3],
   [0, 1, 0, 2, 0, 3, 0, 2], [0, 1, 0, 2, 0, 3, 0, 3],
   [0, 1, 0, 2, 3, 1, 3, 2], [0, 1, 0, 2, 3, 3, 3, 3],
   [0, 1, 2, 1, 0, 1, 3, 1], [0, 1, 2, 1, 0, 3, 2, 3],
   [0, 1, 2, 1, 3, 1, 2, 1], [0, 1, 2, 1, 3, 1, 3, 1],
   [0, 1, 2, 1, 3, 3, 3, 3], [0, 1, 2, 2, 0, 1, 3, 3],
   [0, 1, 2, 2, 0, 3, 2, 2], [0, 1, 2, 2, 3, 1, 2, 2],
   [0, 1, 2, 2, 3, 3, 2, 2], [0, 1, 2, 2, 3, 3, 3, 3],
   [0, 1, 2, 3, 0, 1, 2, 3], [0, 0, 0, 0, 1, 2, 3, 4],
   [0, 0, 1, 1, 2, 2, 3, 4], [0, 0, 1, 1, 2, 3, 2, 4],
   [0, 0, 1, 1, 2, 3, 4, 3], [0, 0, 1, 1, 2, 3, 4, 4],
   [0, 0, 1, 2, 0, 0, 3, 4], [0, 0, 1, 2, 3, 3, 1, 4],
   [0, 0, 1, 2, 3, 3, 4, 2], [0, 0, 1, 2, 3, 3, 4, 4],
   [0, 0, 1, 2, 3, 4, 1, 2], [0, 0, 1, 2, 3, 4, 1, 4],
   [0, 0, 1, 2, 3, 4, 3, 2], [0, 0, 1, 2, 3, 4, 3, 4],
   [0, 1, 0, 1, 2, 2, 3, 4], [0, 1, 0, 1, 2, 3, 2, 4],
   [0, 1, 0, 1, 2, 3, 4, 3], [0, 1, 0, 1, 2, 3, 4, 4],
   [0, 1, 0, 2, 0, 3, 0, 4], [0, 1, 0, 2, 3, 1, 3, 4],
   [0, 1, 0, 2, 3, 1, 4, 2], [0, 1, 0, 2, 3, 1, 4, 4],
   [0, 1, 0, 2, 3, 3, 4, 2], [0, 1, 0, 2, 3, 3, 4, 4],
   [0, 1, 0, 2, 3, 4, 3, 2], [0, 1, 0, 2, 3, 4, 3, 4],
   [0, 1, 2, 1, 0, 3, 2, 4], [0, 1, 2, 1, 0, 3, 4, 3],
   [0, 1, 2, 1, 0, 3, 4, 4], [0, 1, 2, 1, 3, 1, 4, 1],
   [0, 1, 2, 1, 3, 3, 2, 4], [0, 1, 2, 1, 3, 3, 4, 4],
   [0, 1, 2, 1, 3, 4, 2, 4], [0, 1, 2, 1, 3, 4, 3, 4],
   [0, 1, 2, 2, 0, 1, 3, 4], [0, 1, 2, 2, 0, 3, 4, 3],
   [0, 1, 2, 2, 0, 3, 4, 4], [0, 1, 2, 2, 3, 1, 3, 4],
   [0, 1, 2, 2, 3, 1, 4, 4], [0, 1, 2, 2, 3, 3, 4, 4],
   [0, 1, 2, 2, 3, 4, 2, 2], [0, 1, 2, 2, 3, 4, 3, 4],
   [0, 1, 2, 3, 0, 1, 2, 4], [0, 1, 2, 3, 0, 1, 4, 3],
   [0, 1, 2, 3, 0, 1, 4, 4], [0, 1, 2, 3, 0, 4, 2, 3],
   [0, 1, 2, 3, 0, 4, 2, 4], [0, 1, 2, 3, 4, 1, 2, 3],
   [0, 1, 2, 3, 4, 1, 4, 3], [0, 1, 2, 3, 4, 4, 2, 3],
   [0, 1, 2, 3, 4, 4, 4, 4], [0, 0, 1, 1, 2, 3, 4, 5],
   [0, 0, 1, 2, 3, 3, 4, 5], [0, 0, 1, 2, 3, 4, 1, 5],
   [0, 0, 1, 2, 3, 4, 3, 5], [0, 0, 1, 2, 3, 4, 5, 2],
   [0, 0, 1, 2, 3, 4, 5, 4], [0, 0, 1, 2, 3, 4, 5, 5],
   [0, 1, 0, 1, 2, 3, 4, 5], [0, 1, 0, 2, 3, 1, 4, 5],
   [0, 1, 0, 2, 3, 3, 4, 5], [0, 1, 0, 2, 3, 4, 3, 5],
   [0, 1, 0, 2, 3, 4, 5, 2], [0, 1, 0, 2, 3, 4, 5, 4],
   [0, 1, 0, 2, 3, 4, 5, 5], [0, 1, 2, 1, 0, 3, 4, 5],
   [0, 1, 2, 1, 3, 3, 4, 5], [0, 1, 2, 1, 3, 4, 2, 5],
   [0, 1, 2, 1, 3, 4, 3, 5], [0, 1, 2, 1, 3, 4, 5, 4],
   [0, 1, 2, 1, 3, 4, 5, 5], [0, 1, 2, 2, 0, 3, 4, 5],
   [0, 1, 2, 2, 3, 1, 4, 5], [0, 1, 2, 2, 3, 3, 4, 5],
   [0, 1, 2, 2, 3, 4, 3, 5], [0, 1, 2, 2, 3, 4, 5, 4],
   [0, 1, 2, 2, 3, 4, 5, 5], [0, 1, 2, 3, 0, 1, 4, 5],
   [0, 1, 2, 3, 0, 4, 2, 5], [0, 1, 2, 3, 0, 4, 5, 3],
   [0, 1, 2, 3, 0, 4, 5, 4], [0, 1, 2, 3, 0, 4, 5, 5],
   [0, 1, 2, 3, 4, 1, 2, 5], [0, 1, 2, 3, 4, 1, 4, 5],
   [0, 1, 2, 3, 4, 1, 5, 3], [0, 1, 2, 3, 4, 1, 5, 5],
   [0, 1, 2, 3, 4, 4, 2, 5], [0, 1, 2, 3, 4, 4, 5, 3],
   [0, 1, 2, 3, 4, 4, 5, 5], [0, 1, 2, 3, 4, 5, 2, 3],
   [0, 1, 2, 3, 4, 5, 2, 5], [0, 1, 2, 3, 4, 5, 4, 3],
   [0, 1, 2, 3, 4, 5, 4, 5], [0, 0, 1, 2, 3, 4, 5, 6],
   [0, 1, 0, 2, 3, 4, 5, 6], [0, 1, 2, 1, 3, 4, 5, 6],
   [0, 1, 2, 2, 3, 4, 5, 6], [0, 1, 2, 3, 0, 4, 5, 6],
   [0, 1, 2, 3, 4, 1, 5, 6], [0, 1, 2, 3, 4, 4, 5, 6],
   [0, 1, 2, 3, 4, 5, 2, 6], [0, 1, 2, 3, 4, 5, 4, 6],
   [0, 1, 2, 3, 4, 5, 6, 3], [0, 1, 2, 3, 4, 5, 6, 5],
   [0, 1, 2, 3, 4, 5, 6, 6], [0, 1, 2, 3, 4, 5, 6, 7],
];

const TILE_CLRS: [u8; 154] = [
    1, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 8
];
