use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;

trait AlignToSector {
    fn align_to_sector(&mut self) -> ByteIOResult<()>;
}

impl<T: ?Sized + ByteIO> AlignToSector for T {
    fn align_to_sector(&mut self) -> ByteIOResult<()> {
        let next_pos = (self.tell() + 0x7FF) & !0x7FF;
        self.seek(SeekFrom::Start(next_pos))?;
        Ok(())
    }
}

struct ThunderDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    palchange:  bool,
    frame:      Vec<u8>,
    data:       Vec<u8>,
    fps:        u32,
    abuf:       Vec<u8>,
    audio:      bool,
    aframe_len: usize,
    cur_frame:  usize,
    nframes:    usize,
}

#[derive(Debug,PartialEq)]
enum RLEMode {
    None,
    Skip(u16),
    Run(u8, u8),
}

impl ThunderDecoder {
    fn unpack_rle(&mut self, xoff: usize, yoff: usize, width: usize, height: usize, scale2x: bool) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        br.read_skip(1)?; // intra/inter flag?
        let mut mode = RLEMode::None;
        for line in self.frame.chunks_exact_mut(WIDTH).skip(yoff).take(height) {
            if !scale2x {
                for el in line[xoff..][..width].iter_mut() {
                    if mode == RLEMode::None {
                        let op = br.read_byte()?;
                        mode = match op {
                                0x00..=0xDF => {
                                    *el = op;
                                    continue;
                                },
                                0xE0..=0xEF => RLEMode::Skip(u16::from(op & 0xF) + 2),
                                0xF0..=0xFE => {
                                    let clr = br.read_byte()?;
                                    RLEMode::Run((op & 0xF) + 3, clr)
                                },
                                0xFF => RLEMode::Skip(u16::from(br.read_byte()?) + 18),
                            };
                    }
                    mode = match mode {
                        RLEMode::Skip(1) => RLEMode::None,
                        RLEMode::Skip(count) => RLEMode::Skip(count - 1),
                        RLEMode::Run(1, clr) => {
                            *el = clr;
                            RLEMode::None
                        },
                        RLEMode::Run(count, clr) => {
                            *el = clr;
                            RLEMode::Run(count - 1, clr)
                        },
                        RLEMode::None => unreachable!(),
                    };
                }
            } else {
                for pair in line[xoff..][..width * 2].chunks_exact_mut(2) {
                    if mode == RLEMode::None {
                        let op = br.read_byte()?;
                        mode = match op {
                                0x00..=0xDF => {
                                    pair[0] = op;
                                    pair[1] = op;
                                    continue;
                                },
                                0xE0..=0xEF => RLEMode::Skip(u16::from(op & 0xF) + 2),
                                0xF0..=0xFE => {
                                    let clr = br.read_byte()?;
                                    RLEMode::Run((op & 0xF) + 3, clr)
                                },
                                0xFF => RLEMode::Skip(u16::from(br.read_byte()?) + 18),
                            };
                    }
                    mode = match mode {
                        RLEMode::Skip(1) => RLEMode::None,
                        RLEMode::Skip(count) => RLEMode::Skip(count - 1),
                        RLEMode::Run(1, clr) => {
                            pair[0] = clr;
                            pair[1] = clr;
                            RLEMode::None
                        },
                        RLEMode::Run(count, clr) => {
                            pair[0] = clr;
                            pair[1] = clr;
                            RLEMode::Run(count - 1, clr)
                        },
                        RLEMode::None => unreachable!(),
                    };
                }
            }
        }
        Ok(())
    }
}

impl InputSource for ThunderDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1000,
                    tb_den: self.fps,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 11025,
                    channels:    1,
                    sample_type: AudioSample::U8,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        if self.audio && self.aframe_len > 0 && self.abuf.len() >= self.aframe_len {
            let mut audio = vec![0; self.aframe_len];
            audio.copy_from_slice(&self.abuf[..self.aframe_len]);
            self.abuf.drain(..self.aframe_len);
            self.audio = false;
            for el in audio.iter_mut() {
                *el ^= 0x80;
            }
            return Ok((1, Frame::AudioU8(audio)));
        }

        if self.cur_frame >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.cur_frame += 1;
        let mut size = br.read_u16le()? as usize;
        validate!(size > 13);
        let asize  = br.read_u16le()? as usize;
        let xoff   = br.read_u16le()? as usize;
        let yoff   = br.read_u16le()? as usize;
        let width  = br.read_u16le()? as usize;
        let height = br.read_u16le()? as usize;
        let _mode  = br.read_byte()?;
        validate!(xoff + width <= WIDTH && yoff + height <= HEIGHT);
        size -= 13;
        if self.palchange {
            validate!(size > 15);
            let start_clr = usize::from(br.read_byte()?);
            let nclrs  = usize::from(br.read_byte()?);
            if nclrs > 0 {
                validate!(size > nclrs * 3);
                validate!(start_clr + nclrs <= 256);
                br.read_vga_pal_some(&mut self.pal[start_clr * 3..][..nclrs * 3])?;
            }
            size -= nclrs * 3 + 2;
        }
        validate!(size > asize);
        self.data.resize(size - asize, 0);
        br.read_buf(&mut self.data)?;

        br.read_extend(&mut self.abuf, asize)?;

        let scale2x = (xoff == 0) && (width == WIDTH / 2);
        self.unpack_rle(xoff, yoff, width, height, scale2x)
            .map_err(|_| DecoderError::InvalidData)?;
        self.audio = true;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let _version = br.read_u16le()?;
    let nframes = br.read_u16le()? as usize;
    validate!(nframes > 0);
    br.read_u16le()?;
    br.read_u32le()?;
    let _vsize_in_sectors = br.read_u16le()? as usize;
    br.read_u16le()?;
    let num_preload = br.read_u16le()? as usize;
    let nclrs = br.read_u16le()? as usize;
    validate!((1..=256).contains(&nclrs));
    let palchange = br.read_u16le()? != 0;
    let aframe_len = br.read_u16le()? as usize;
    br.read_u16le()?;
    for _ in 0..18 {
        br.read_u16le()?;
    }
    br.read_u16le()?;
    let _audio_mode = br.read_u16le()?;

    let mut pal = [0; 768];
    br.read_byte()?;
    br.read_byte()?;
    let start_clr = br.read_u16le()? as usize;
    let end_clr = br.read_u16le()? as usize;
    validate!(start_clr <= end_clr && end_clr < 256);
    br.read_vga_pal_some(&mut pal[..nclrs * 3])?;

    br.read_skip(nframes * 4)?; // relative offsets to frame starts

    // determine total audio duration
    let mut alen_hi = 0;
    let mut alen_lo = 0;
    for _ in 0..nframes {
        let aduration = br.read_u16le()? as usize;
        if aduration < alen_lo {
            alen_hi += 0x10000;
        }
        alen_lo = aduration;
    }

    br.align_to_sector()?;

    let mut abuf = vec![0; aframe_len * num_preload];
    br.read_buf(&mut abuf)?;
    br.align_to_sector()?;

    let aframe_len = (alen_lo + alen_hi) / nframes;
    let fps = if aframe_len == 0 { 15 } else { 11025000 / (aframe_len as u32) };

    Ok(Box::new(ThunderDecoder {
        fr: br,
        pal, palchange,
        data: Vec::new(),
        fps,
        frame:  vec![0; WIDTH * HEIGHT],
        abuf,
        audio: false,
        aframe_len,
        cur_frame: 0,
        nframes,
    }))
}
