use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;
use super::super::util::path::*;

type Codebook = [[u8; 16]; 256];

struct Header {
    width:      usize,
    height:     usize,
    y_deltas:   [i16; 8],
    c_deltas:   [i16; 8],
}

impl Header {
    fn load(br: &mut dyn ByteIO) -> DecoderResult<Self> {
        let version = br.read_u32be()?;
        validate!(version == 1 || version == 3);

        let _x_offset = br.read_u32be()?;
        let _y_offset = br.read_u32be()?;
        let width    = (br.read_u16be()? as usize) * 8;
        let height   = (br.read_u16be()? as usize) * 8;
        validate!(width > 0 && width <= 1024);
        validate!(height > 0 && height <= 768);

        let mut y_deltas = [0; 8];
        for delta in y_deltas.iter_mut() {
            *delta = br.read_u16be()? as i16;
        }
        let mut c_deltas = [0; 8];
        for delta in c_deltas.iter_mut() {
            *delta = br.read_u16be()? as i16;
        }

        Ok(Header { width, height, y_deltas, c_deltas })
    }
}

#[derive(Default)]
struct IMAState {
    pred:   i32,
    step:   u8,
}

impl IMAState {
    fn expand(&mut self, nibble: u8) -> i16 {
        let sign  = (nibble & 8) != 0;
        let delta =  nibble & 7;
        let diff = ((i32::from(delta * 2) + 1) * i32::from(DK_ADPCM_STEPS[self.step as usize])) >> 3;
        if !sign {
            self.pred += diff;
        } else {
            self.pred -= diff;
        }
        let new_step = self.step as i8 + STEP_ADJ[delta as usize];
        self.step = new_step.max(0).min((DK_ADPCM_STEPS.len() as i8) - 1) as u8;

        self.pred as i16
    }
}

const NO_VEC: usize = 256;

struct CodebookState<'a> {
    cb:         &'a Codebook,
    src:        &'a [u8],
    src_pos:    usize,
    cur_vec:    usize,
    vec_pos:    usize,
    vec_size:   usize,
}

impl<'a> CodebookState<'a> {
    fn new(cb: &'a Codebook, src: &'a [u8]) -> Self {
        Self {
            cb,
            src,
            src_pos:  16,
            cur_vec:  NO_VEC,
            vec_pos:  0,
            vec_size: 0,
        }
    }
    fn next(&mut self) -> DecoderResult<[u8; 2]> {
        if self.cur_vec == NO_VEC {
            if self.src_pos < self.src.len() {
                self.cur_vec  = usize::from(self.src[self.src_pos]);
                self.src_pos += 1;
            } else {
                return Err(DecoderError::ShortData);
            }
            self.vec_size = usize::from(self.cb[self.cur_vec][0]);
            self.vec_pos  = 0;
        }
        let entry = &self.cb[self.cur_vec];
        let ret = [entry[self.vec_pos + 1], entry[self.vec_pos + 2]];
        self.vec_pos += 2;
        if self.vec_pos >= self.vec_size {
            self.cur_vec = NO_VEC;
        }
        Ok(ret)
    }
}

fn chroma_delta(table: &[i16; 8], vals: [u8; 2]) -> u32 {
    let d0 = i32::from(table[usize::from(vals[0])]);
    let d1 = i32::from(table[usize::from(vals[1])]);
    ((d0 << 26) + (d1 << 16) + (d0 << 10) + d1) as u32
}

fn luma_delta(table: &[i16; 8], vals: [u8; 2]) -> u32 {
    let d0 = i32::from(table[usize::from(vals[0])]) >> 1;
    let d1 = i32::from(table[usize::from(vals[1])]) >> 1;
    (((d0 * 0x421) << 16) + d1 * 0x421) as u32
}

fn luma_corr(table: &[i16; 8], vals: [u8; 2]) -> u32 {
    let d0 = (table[usize::from(vals[0])] & 1) as u32;
    let d1 = (table[usize::from(vals[1])] & 1) as u32;
    (d0 << 31) | (d1 << 15)
}

struct DuckDecoder {
    fr:         FileReader<File>,
    header:     Header,
    offs:       Vec<u32>,
    codebook:   Codebook,
    vdata:      Vec<u8>,
    adata:      Vec<u8>,
    top:        Vec<u32>,
    line1:      Vec<u32>,
    line2:      Vec<u32>,
    frameno:    usize,
    audio:      bool,
    adpcm:      [IMAState; 2],
}

impl DuckDecoder {
    fn decode_audio(&mut self) -> DecoderResult<(usize, Frame)> {
        validate!(self.adata[0] == 0xF7 && self.adata[1] == 0x7F);
        let asamples = read_u16be(&self.adata[2..])? as usize;
        validate!(asamples == self.adata.len() - 10);
        self.adpcm[0].step = self.adata[7];
        self.adpcm[1].step = self.adata[9];
        let mut dst = Vec::with_capacity(asamples * 2);
        for &val in self.adata[10..].iter() {
            let samp_l = self.adpcm[0].expand(val >> 4);
            let samp_r = self.adpcm[1].expand(val & 0xF);
            dst.push(samp_l);
            dst.push(samp_r);
        }
        Ok((1, Frame::AudioS16(dst)))
    }
    fn decode_video(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut frm = Vec::with_capacity(self.header.width * self.header.height);

        let method = self.vdata[0];
        let width = read_u16le(&self.vdata[14..])? as usize;
        let height = read_u16le(&self.vdata[12..])? as usize;
        validate!(width * 2 == self.header.width && height >= self.header.height);

        let mut cstate = CodebookState::new(&self.codebook, &self.vdata);
        for el in self.top.iter_mut() {
            *el = 0;
        }

        match method {
            1 => {
                for _ in (0..self.header.height).step_by(8) {
                    let mut acc = [0u32; 2];
                    let mut corr = [0u32; 2];
                    for (top_quad, (quad0, quad1)) in self.top.chunks_exact(4)
                            .zip(self.line1.chunks_exact_mut(4)
                                .zip(self.line2.chunks_exact_mut(4))) {
                        let c_delta = chroma_delta(&self.header.c_deltas, cstate.next()?);
                        acc[0] = acc[0].wrapping_add(c_delta);

                        for (el0, &pred) in quad0.iter_mut().zip(top_quad.iter()) {
                            let state = cstate.next()?;
                            let y_delta = luma_delta(&self.header.y_deltas, state);
                            let y_corr  = luma_corr(&self.header.y_deltas, state);
                            acc[0] = acc[0].wrapping_add(y_delta);
                            corr[0] ^= y_corr;
                            *el0 = pred.wrapping_add(acc[0]) ^ corr[0];
                        }
                        for (el1, &pred) in quad1.iter_mut().zip(quad0.iter()) {
                            let state = cstate.next()?;
                            let y_delta = luma_delta(&self.header.y_deltas, state);
                            let y_corr  = luma_corr(&self.header.y_deltas, state);
                            acc[1] = acc[1].wrapping_add(y_delta);
                            corr[1] ^= y_corr;
                            *el1 = pred.wrapping_add(acc[1]) ^ corr[1];
                        }
                    }
                    for _ in 0..4 {
                        for &pix in self.line1.iter() {
                            frm.push((pix >> 16) as u16);
                            frm.push(pix as u16);
                        }
                    }
                    for _ in 0..4 {
                        for &pix in self.line2.iter() {
                            frm.push((pix >> 16) as u16);
                            frm.push(pix as u16);
                        }
                    }
                    std::mem::swap(&mut self.top, &mut self.line2);
                }
            },
            3 => {
                for _ in (0..self.header.height).step_by(4) {
                    let mut acc = 0u32;
                    for (x, (cur, &top)) in self.line1.iter_mut().zip(self.top.iter()).enumerate() {
                        if (x & 3) == 0 {
                            let c_delta = chroma_delta(&self.header.c_deltas, cstate.next()?);
                            acc = acc.wrapping_add(c_delta);
                        }
                        let y_delta = luma_delta(&self.header.y_deltas, cstate.next()?);
                        acc = acc.wrapping_add(y_delta);
                        *cur = acc.wrapping_add(top);
                    }
                    for _ in 0..4 {
                        for &pix in self.line1.iter() {
                            frm.push((pix >> 16) as u16);
                            frm.push(pix as u16);
                        }
                    }
                    std::mem::swap(&mut self.line1, &mut self.top);
                }
            },
            _ => unreachable!(),
        }

        Ok((0, Frame::VideoRGB16(frm)))
    }
}

impl InputSource for DuckDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.header.width,
                    height: self.header.height,
                    bpp:    15,
                    tb_num: 754, // ~14.622 fps
                    tb_den: 11025,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    channels:    2,
                    sample_type: AudioSample::S16,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frameno >= self.offs.len() {
            return Err(DecoderError::EOF);
        }
        let br = &mut self.fr;
        if self.audio {
            br.seek(SeekFrom::Start(u64::from(self.offs[self.frameno])))?;
            self.audio = false;
            let asize = br.read_u32be()? as usize;
            validate!(asize > 10);
            let vsize = br.read_u32be()? as usize;
            validate!(vsize > 16);
            self.adata.resize(asize, 0);
            br.read_buf(&mut self.adata)?;
            self.vdata.resize(vsize, 0);
            br.read_buf(&mut self.vdata)?;

            self.decode_audio()
        } else {
            self.frameno += 1;
            self.audio = true;

            self.decode_video().map_err(|_| DecoderError::InvalidData)
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let mut bname = name.to_owned();
    match (bname.rfind('.'), bname.rfind(std::path::MAIN_SEPARATOR)) {
        (Some(epos), Some(spos)) => {
            if epos > spos {
                bname.truncate(epos);
            }
        },
        (Some(epos), None) => {
            bname.truncate(epos);
        },
        _ => {}
    }

    //println!("basename: '{bname}'");

    let hdr_name = bname.clone() + ".hdr";
    let file = open_file_igncase(&hdr_name).map_err(|_| DecoderError::AuxInputNotFound(hdr_name))?;
    let mut br = FileReader::new_read(file);
    let header = Header::load(&mut br)?;

    let frm_name = bname.clone() + ".frm";
    let file = open_file_igncase(&frm_name).map_err(|_| DecoderError::AuxInputNotFound(frm_name))?;
    let mut br = FileReader::new_read(BufReader::new(file));
    let mut offs = Vec::new();
    while let Ok(val) = br.read_u32be() {
        offs.push(val);
    }
    validate!(!offs.is_empty());

    let tbl_name = bname.clone() + ".tbl";
    let file = open_file_igncase(&tbl_name).map_err(|_| DecoderError::AuxInputNotFound(tbl_name))?;
    let mut br = FileReader::new_read(file);
    let mut codebook = [[0; 16]; 256];
    for entry in codebook.iter_mut() {
        br.read_buf(entry)?;
    }

    let duk_name = bname + ".duk";
    let file = open_file_igncase(&duk_name).map_err(|_| DecoderError::AuxInputNotFound(duk_name))?;
    let fr = FileReader::new_read(file);

    let width = header.width;
    Ok(Box::new(DuckDecoder {
        fr,
        header, offs, codebook,
        top:     vec![0; width / 2],
        line1:   vec![0; width / 2],
        line2:   vec![0; width / 2],
        vdata:   Vec::new(),
        adata:   Vec::with_capacity(0x5EE),
        audio:   true,
        adpcm:   [IMAState::default(), IMAState::default()],
        frameno: 0,
    }))
}

const STEP_ADJ: [i8; 16] = [
    -1, -1, -1, -1, 2, 4, 6, 8,
    -1, -1, -1, -1, 2, 4, 6, 8
];

const DK_ADPCM_STEPS: [i16; 89] = [
       0x7,    0x8,    0x9,    0xA,    0xB,    0xC,    0xD,    0xF,
      0x10,   0x12,   0x13,   0x15,   0x17,   0x1A,   0x1C,   0x1F,
      0x22,   0x26,   0x29,   0x2E,   0x32,   0x37,   0x3D,   0x43,
      0x4A,   0x51,   0x59,   0x62,   0x6C,   0x76,   0x82,   0x8F,
      0x9E,   0xAD,   0xBF,   0xD2,   0xE7,   0xFE,  0x117,  0x133,
     0x152,  0x174,  0x199,  0x1C2,  0x1EF,  0x220,  0x256,  0x292,
     0x2D4,  0x31D,  0x36C,  0x3C4,  0x424,  0x48E,  0x503,  0x583,
     0x610,  0x6AC,  0x756,  0x812,  0x8E1,  0x9C4,  0xABE,  0xBD1,
     0xCFF,  0xE4C,  0xFBA, 0x114D, 0x1308, 0x14EF, 0x1707, 0x1954,
    0x1BDD, 0x1EA6, 0x21B7, 0x2516, 0x28CB, 0x2CDF, 0x315C, 0x364C,
    0x3BBA, 0x41B2, 0x4844, 0x4F7E, 0x5771, 0x6030, 0x69CE, 0x7463,
    0x7FFF
];
