use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 400;
const HEIGHT: usize = 300;

struct DXVDecoder {
    fr:         FileReader<File>,
    frameno:    u32,
    nframes:    u32,
    frame:      Vec<u8>,
    vdata:      Vec<u8>,
}

impl DXVDecoder {
    fn decode_rle(src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(src);
        let mut pos = 0;
        while pos < dst.len() {
            let op = usize::from(br.read_byte()?);
            if op < 0x80 {
                let len = op;
                validate!(pos + len <= dst.len());
                br.read_buf(&mut dst[pos..][..len])?;
                pos += len;
            } else {
                let len = op - 0x80;
                let clr = br.read_byte()?;
                validate!(pos + len <= dst.len());
                for el in dst[pos..][..len].iter_mut() {
                    *el = clr;
                }
                pos += len;
            }
        }
        Ok(())
    }
}

impl InputSource for DXVDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 10,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frameno >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.frameno += 1;

        let fsize = self.fr.read_u32le()? as usize;
        validate!(fsize > 0 && fsize <= self.frame.len() * 2);

        self.vdata.resize(fsize, 0);
        self.fr.read_buf(&mut self.vdata)?;
        Self::decode_rle(&self.vdata, &mut self.frame)
            .map_err(|_| DecoderError::InvalidData)?;
        let mut pal = [0; 768];
        let (fpal, ffrm) = self.frame.split_at(768);
        pal.copy_from_slice(fpal);
        Ok((0, Frame::VideoPal(ffrm[..WIDTH * HEIGHT].to_vec(), pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);

    let nframes = fr.read_u32le()?;
    validate!((1..=10000).contains(&nframes));

    Ok(Box::new(DXVDecoder {
        fr,
        frame: vec![0; 141172], // all frames unpack to this size
        vdata: Vec::new(),
        frameno: 0,
        nframes,
    }))
}
