use std::fs::File;
use std::io::BufReader;
use crate::input::util::rnc::*;
use crate::io::byteio::*;
use super::super::*;

enum State {
    PacketStart,
    VideoFrames(u32),
}

struct ETVDecoder {
    fr:         FileReader<BufReader<File>>,
    data:       Vec<u8>,
    udata:      Vec<u8>,
    frame:      Vec<u8>,
    pal:        [u8; 768],
    fps:        u32,
    width:      usize,
    height:     usize,
    nframes:    usize,
    cur_frm:    usize,
    pkt_ends:   Vec<u64>,
    pkt_no:     usize,
    state:      State,
    arate:      u32,
    abits:      u8,
    channels:   u8,
}

impl ETVDecoder {
    fn decode_video(&mut self) -> DecoderResult<(usize, Frame)> {
        let tsize = self.width / 4 * self.height / 4 * 2 / 8;
        validate!(self.udata.len() >= tsize);
        let (modes, pixels) = self.udata.split_at(tsize);
        let mut br = MemoryReader::new_read(pixels);

        let mut modes = modes.iter();
        let mut tm = 0;
        let mut tl = 0;
        for stripe in self.frame.chunks_exact_mut(self.width * 4) {
            for x in (0..self.width).step_by(4) {
                if tl == 0 {
                    tm = *modes.next().unwrap();
                    tl = 4;
                }
                let tmode = tm & 3;
                tm >>= 2;
                tl -= 1;
                match tmode {
                    0 => {
                        for line in stripe[x..].chunks_mut(self.width) {
                            br.read_buf(&mut line[..4])?;
                        }
                    },
                    2 => {
                        let clr = br.read_byte()?;
                        for line in stripe[x..].chunks_mut(self.width) {
                            for el in line[..4].iter_mut() {
                                *el = clr;
                            }
                        }
                    },
                    _ => {},
                }
            }
        }

        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

impl InputSource for ETVDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: self.fps,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    self.channels,
                    sample_type: if self.abits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        match self.state {
            State::PacketStart => {
                if self.pkt_no >= self.pkt_ends.len() {
                    return Err(DecoderError::EOF);
                }
                self.state = State::VideoFrames(self.fps);
                let nsamples = (self.arate as usize) * usize::from(self.channels);
                if self.abits == 8 {
                    let mut audio = vec![0; nsamples];
                    br.read_buf(&mut audio)?;
                    Ok((1, Frame::AudioU8(audio)))
                } else {
                    let mut audio = Vec::with_capacity(nsamples);
                    for _ in 0..nsamples {
                        let samp = br.read_u16le()? as i16;
                        audio.push(samp);
                    }
                    Ok((1, Frame::AudioS16(audio)))
                }
            },
            State::VideoFrames(left) => {
                if self.cur_frm >= self.nframes {
                    return Err(DecoderError::EOF);
                }
                let size = br.read_u32le()? as usize;
                validate!(size >= 5);
                validate!(br.tell() + (size as u64) - 5 <= self.pkt_ends[self.pkt_no]);
                if br.read_byte()? != 0 {
                    br.read_buf(&mut self.pal)?;
                    br.read_skip(256 * 256 / 8)?; // noise data, ignored
                }
                let csize = br.read_u32le()? as usize;
                validate!(csize >= 4);
                validate!(br.tell() + (csize as u64) - 4 <= self.pkt_ends[self.pkt_no]);
                self.data.resize(csize - 4, 0);
                br.read_buf(&mut self.data)?;
                rnc_uncompress(&self.data, &mut self.udata).map_err(|_| DecoderError::InvalidData)?;
                self.cur_frm += 1;
                self.state = if left > 1 { State::VideoFrames(left - 1) } else {
                        br.seek(SeekFrom::Start(self.pkt_ends[self.pkt_no]))?;
                        self.pkt_no += 1;
                        State::PacketStart
                    };
                self.decode_video().map_err(|err|
                        if err == DecoderError::ShortData { DecoderError::InvalidData } else { err })
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag = br.read_tag()?;
    validate!(&tag == b"ETV\x0A");
    let width  = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!(width <= 1024 && height <= 768);
    validate!((width | height) & 3 == 0);
    let nframes = br.read_u32le()? as usize;
    validate!(nframes > 0);
    let fps = br.read_u32le()?;
    validate!(fps > 0 && fps <= 50);
    let channels = br.read_u32le()?;
    validate!(channels == 1 || channels == 2);
    let arate = br.read_u32le()?;
    validate!(arate > 0 && arate <= 48000);
    let abits = br.read_u32le()?;
    validate!(abits == 8 || abits == 16);
    br.read_u32le()?; // unknown
    br.read_u32le()?; // unknown
    let mut pkt_ends = Vec::with_capacity(200);
    let mut pkt_end = 0x28 + 200 * 4;
    for _ in 0..200 {
        let size = br.read_u32le()?;
        if size == 0 {
            break;
        }
        pkt_end += u64::from(size);
        pkt_ends.push(pkt_end);
    }
    br.seek(SeekFrom::Start(0x28 + 200 * 4))?;

    Ok(Box::new(ETVDecoder {
        fr: br,
        frame: vec![0; width * height],
        pal: [0; 768],
        data: Vec::new(),
        udata: Vec::new(),
        width, height, fps, nframes,
        pkt_ends,
        state: State::PacketStart,
        pkt_no: 0,
        cur_frm: 0,
        arate,
        channels: channels as u8,
        abits: abits as u8,
    }))
}
