use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct VideoData {
    frame:      Vec<u8>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    vdata:      Vec<u8>,
}

impl VideoData {
    fn decode_key_frame(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        for line in self.frame.chunks_exact_mut(self.width) {
            let nops = usize::from(br.read_byte()?);
            let mut pos = 0;
            for _ in 0..nops {
                let op = usize::from(br.read_byte()?);
                if (op & 0x80) == 0 {
                    let len = op;
                    validate!(pos + len <= line.len());
                    let clr = br.read_byte()?;
                    for el in line[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                } else {
                    let len = 256 - op;
                    validate!(pos + len <= line.len());
                    br.read_buf(&mut line[pos..][..len])?;
                    pos += len;
                }
            }
            validate!(pos == line.len());
        }

        Ok(())
    }
    fn decode_delta_frame(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        let skip_lines = br.read_u16le()? as usize;
        validate!(skip_lines <= self.height);
        let nlines = br.read_u16le()? as usize;
        validate!(skip_lines + nlines <= self.height);

        for line in self.frame.chunks_exact_mut(self.width).skip(skip_lines).take(nlines) {
            let nops = usize::from(br.read_byte()?);
            let mut pos = 0;
            for _ in 0..nops {
                let skip = usize::from(br.read_byte()?);
                validate!(pos + skip <= line.len());
                pos += skip;
                let op = usize::from(br.read_byte()?);
                if (op & 0x80) == 0 {
                    let len = op;
                    validate!(pos + len <= line.len());
                    br.read_buf(&mut line[pos..][..len])?;
                    pos += len;
                } else {
                    let len = 256 - op;
                    validate!(pos + len <= line.len());
                    let clr = br.read_byte()?;
                    for el in line[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                }
            }
        }

        Ok(())
    }
}

struct FLADecoder {
    fr:         FileReader<BufReader<File>>,
    frm_no:     u32,
    nframes:    u32,
    fps:        u8,
    video:      VideoData,
}

impl InputSource for FLADecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.video.width,
                height: self.video.height,
                bpp:    8,
                tb_num: 1,
                tb_den: u32::from(self.fps),
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frm_no >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.frm_no += 1;

        let br = &mut self.fr;
        let nchunks = br.read_u16le()? as usize;
        let size = br.read_u32le()? as usize;
        validate!(size >= nchunks * 4);
        let end = br.tell() + (size as u64);

        for _ in 0..nchunks {
            let op = br.read_u16le()?;
            let size = br.read_u16le()? as usize;
            validate!(br.tell() + (size as u64) <= end);
            match op {
                1 => {
                    let nclrs = br.read_u16le()? as usize;
                    let start = br.read_u16le()? as usize;
                    validate!(start + nclrs <= 256);
                    validate!(size == nclrs * 3 + 4);
                    //br.read_vga_pal_some(&mut self.video.pal[start * 3..][..nclrs * 3])?;
                    br.read_buf(&mut self.video.pal[start * 3..][..nclrs * 3])?;
                },
                2 => { // special messages
                    br.read_skip(size)?;
                },
                3 => { // play (external) sample
                    validate!(size >= 9);
                    br.read_skip(size)?;
                },
                4 => { // sample balance
                    validate!(size >= 8);
                    br.read_skip(size)?;
                },
                5 => { // stop sample
                    validate!(size >= 2);
                    br.read_skip(size)?;
                },
                6 => { // FLI_LC
                    self.video.vdata.resize(size, 0);
                    br.read_buf(&mut self.video.vdata)?;
                    self.video.decode_delta_frame().map_err(|_| DecoderError::InvalidData)?;
                },
                7 => { // black frame
                    validate!(size == 0);
                    for el in self.video.frame.iter_mut() {
                        *el = 0;
                    }
                },
                8 => { // FLI_BRUN
                    self.video.vdata.resize(size, 0);
                    br.read_buf(&mut self.video.vdata)?;
                    self.video.decode_key_frame().map_err(|_| DecoderError::InvalidData)?;
                },
                9 | 16 => { // copy, FLI_COPY
                    validate!(size == self.video.frame.len());
                    br.read_buf(&mut self.video.frame)?;
                },
                _ => {
                    println!("Unknown opcode {op:02X} @ {:X}", br.tell());
                    br.read_skip(size)?;
                },
            }
        }
        validate!(br.tell() == end);

        Ok((0, Frame::VideoPal(self.video.frame.clone(), self.video.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag         = br.read_tag()?;
    validate!(&tag[..3] == b"V1." && tag[3].is_ascii_digit());
    if tag[3] != b'3' {
        return Err(DecoderError::NotImplemented);
    }
    validate!(br.read_byte()? == 0);
    br.read_skip(1)?;

    let nframes = br.read_u32le()?;
    validate!(nframes > 0 && nframes < 4096);
    let fps = br.read_byte()?;
    validate!(fps > 0 && fps <= 30);
    br.read_skip(1)?;
    let width  = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width <= 320 && height <= 240);

    let num_smth = br.read_u16le()? as usize;
    br.read_u16le()?;
    br.read_skip(num_smth * 4)?;

    Ok(Box::new(FLADecoder {
        fr: br,
        frm_no: 0,
        nframes,
        fps,
        video: VideoData {
            width, height,
            pal: [0; 768],
            frame: vec![0; width * height],
            vdata: Vec::new(),
        },
    }))
}
