use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::input::util::imaadpcm::*;
use super::super::*;

struct Bits8<'a> {
    src:    &'a [u8],
    pos:    usize,
    buf:    u8,
    bit:    u8,
}

impl<'a> Bits8<'a> {
    fn new(src: &'a [u8]) -> Self { Bits8 { src, pos: 0, buf: 0, bit: 0 } }
    fn read_bit(&mut self) -> ByteIOResult<bool> {
        if self.bit == 0 {
            if self.pos < self.src.len() {
                self.buf = self.src[self.pos];
                self.pos += 1;
                self.bit = 8;
            } else {
                return Err(ByteIOError::ReadError);
            }
        }
        let bit = (self.buf & 0x80) != 0;
        self.buf <<= 1;
        self.bit -= 1;
        Ok(bit)
    }
}

struct FVVideoDecoder {
    fr:         FileReader<BufReader<File>>,
    frame:      Vec<u8>,
    pal:        [u8; 768],
    data:       Vec<u8>,
    width:      usize,
    height:     usize,
    fps:        u32,
    asizes:     Vec<usize>,
    vsizes:     Vec<usize>,
    audio:      bool,
    arate:      u32,
    abits:      u32,
    abuf8:      Vec<u8>,
    abuf16:     Vec<i16>,
    aframe_len: usize,
    cur_frame:  usize,
}

impl FVVideoDecoder {
    fn decode_video(&mut self) -> DecoderResult<()> {
        let bitsize = read_u16le(&self.data)? as usize;
        let bsize = (bitsize + 8) >> 3;
        validate!(bsize + 2 <= self.data.len());

        let mut flags = Bits8::new(&self.data[2..][..bsize]);
        let mut br = MemoryReader::new_read(&self.data[2 + bsize..]);

        if (bsize + 2 != self.data.len()) && flags.read_bit()? {
            for dst in self.pal.iter_mut() {
                let b                   = br.read_byte()?;
                *dst = (b << 2) | (b >> 4);
            }
        }

        let stride = self.width;
        // for some reason last row should not be decoded
        for row4 in self.frame.chunks_mut(stride * 4).take(self.height / 4 - 1) {
            for x in (0..self.width).step_by(4) {
                if flags.read_bit()? {
                    if flags.read_bit()? {
                        let c0          = br.read_byte()?;
                        let c1          = br.read_byte()?;
                        let mut mask    = br.read_u16le()?;
                        for dst in row4[x..].chunks_mut(stride) {
                            for pix in dst.iter_mut().take(4) {
                                *pix = if (mask & 0x8000) != 0 { c1 } else { c0 };
                                mask <<= 1;
                            }
                        }
                    } else {
                        for dst in row4[x..].chunks_mut(stride) {
                                          br.read_buf(&mut dst[..4])?;
                        }
                    }
                }
            }
        }
        Ok(())
    }
}

impl InputSource for FVVideoDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: self.fps,
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    1,
                    sample_type: if self.abits == 16 { AudioSample::S16 } else { AudioSample::U8 },
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.audio && self.abuf8.len() >= self.aframe_len {
            let mut audio = vec![0; self.aframe_len];
            audio.copy_from_slice(&self.abuf8[..self.aframe_len]);
            self.abuf8.drain(..self.aframe_len);
            self.audio = false;
            return Ok((1, Frame::AudioU8(audio)));
        }
        if self.audio && self.abuf16.len() >= self.aframe_len {
            let mut audio = vec![0; self.aframe_len];
            audio.copy_from_slice(&self.abuf16[..self.aframe_len]);
            self.abuf16.drain(..self.aframe_len);
            self.audio = false;
            return Ok((1, Frame::AudioS16(audio)));
        }
        if self.cur_frame >= self.vsizes.len() {
            return Err(DecoderError::EOF);
        }

        let mut br = &mut self.fr;
        self.data.resize(self.vsizes[self.cur_frame], 0);
        br.read_buf(&mut self.data)?;
        self.decode_video().map_err(|_| DecoderError::InvalidData)?;
        br = &mut self.fr;

        if self.arate > 0 && self.asizes[self.cur_frame] > 0 {
            let asize = self.asizes[self.cur_frame];
            if self.abits == 8 {
                br.read_extend(&mut self.abuf8, asize)?;
            } else {
                validate!((asize & 1) == 0);
                for _ in 0..asize/2 {
                    let samp = br.read_u16le()? as i16;
                    self.abuf16.push(samp);
                }
            }
        }
        self.audio = true;
        self.cur_frame += 1;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open_video(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut src = FileReader::new_read(BufReader::new(file));

    let magic                       = src.read_tag()?;
    validate!(&magic == b"2TSF");
    let width                       = src.read_u32le()? as usize;
    let height                      = src.read_u32le()? as usize;
    validate!((1..=640).contains(&width) && (1..=480).contains(&height));
    let _flags                      = src.read_u32le()?;
    let nframes                     = src.read_u32le()? as usize;
    let fps                         = src.read_u32le()?;
    validate!((1..=30).contains(&fps));
    let arate                       = src.read_u32le()?;
    let abits                       = src.read_u32le()?;

    if arate != 0 {
        validate!(abits == 8 || abits == 16);
    }

    let mut vsizes = Vec::with_capacity(nframes);
    let mut asizes = Vec::with_capacity(nframes);
    for _ in 0..nframes {
        let vsize                   = src.read_u32le()? as usize;
        let asize                   = src.read_u16le()? as usize;
        vsizes.push(vsize);
        asizes.push(asize);
    }

    Ok(Box::new(FVVideoDecoder {
        fr: src,
        width, height, fps,
        data: Vec::new(),
        frame:  vec![0; width * height],
        pal: [0; 768],
        asizes, vsizes,
        arate, abits,
        abuf8: Vec::new(),
        abuf16: Vec::new(),
        aframe_len: (arate / fps) as usize,
        cur_frame: 0,
        audio: false,
    }))
}

struct FVAudioDecoder {
    fr:         FileReader<BufReader<File>>,
    size:       u64,
    arate:      u32,
    abits:      u16,
    state:      IMAState,
    count:      usize,
}

impl InputSource for FVAudioDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Audio(AudioInfo{
                sample_rate: self.arate,
                channels:    1,
                sample_type: if self.abits == 16 { AudioSample::S16 } else { AudioSample::U8 },
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        const BLOCK_SIZE: usize = 0x2000;

        let br = &mut self.fr;
        if br.tell() >= self.size {
            return Err(DecoderError::EOF);
        }

        let bsize = BLOCK_SIZE.min((self.size - br.tell()) as usize);

        let mut audio = Vec::with_capacity(bsize * 2);
        for _ in 0..bsize {
            let val = br.read_byte()?;
            let mut samp0 = self.state.expand_sample(val & 0xF);
            let mut samp1 = self.state.expand_sample(val >> 4);
            if self.count < 50 {
                samp0 = 0;
                samp1 = 0;
            }
            audio.push(samp0);
            audio.push(samp1);
            self.count += 2;
        }

        Ok((0, Frame::AudioS16(audio)))
    }
}

pub fn open_audio(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let magic                       = br.read_tag()?;
    validate!(&magic == b"FCMP");
    let size                        = u64::from(br.read_u32le()?);
    let arate                       = br.read_u32le()?;
    validate!(arate != 0);
    let abits                       = br.read_u16le()?;
    validate!(abits == 8 || abits == 16);

    Ok(Box::new(FVAudioDecoder {
        fr: br,
        size, arate, abits,
        state: IMAState::new(),
        count: 0,
    }))
}
