use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

struct GAVDecoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    global_pal: bool,
    width:      usize,
    height:     usize,
    fps:        u16,
    vdata:      Vec<u8>,
    nfrm:       u32,
    frm_no:     u32,
}

impl GAVDecoder {
    fn unpack_rle(&mut self) -> DecoderResult<Vec<u8>> {
        let mut br = MemoryReader::new_read(&self.vdata);
        let frame_size = self.width * self.height;
        let mut frame = Vec::with_capacity(frame_size);
        while frame.len() < frame_size {
            let op = br.read_byte()?;
            match op {
                0 => break,
                1..=0x7F => {
                    let len = usize::from(op);
                    validate!(frame.len() + len <= frame_size);
                    br.read_extend(&mut frame, len)?;
                },
                _ => {
                    let len = usize::from(op & 0x7F);
                    let clr = br.read_byte()?;
                    validate!(frame.len() + len <= frame_size);
                    for _ in 0..len {
                        frame.push(clr);
                    }
                }
            }
        }
        Ok(frame)
    }
}

impl InputSource for GAVDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 12,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frm_no >= self.nfrm {
            return Err(DecoderError::EOF);
        }
        self.frm_no += 1;
        let size = self.fr.read_u32le()? as usize;
        validate!((1..=1048576).contains(&size));
        if !self.global_pal {
            validate!(size > self.pal.len());
            self.fr.read_vga_pal(&mut self.pal)?;
            self.vdata.resize(size - self.pal.len(), 0);
        } else {
            self.vdata.resize(size, 0);
        }
        self.fr.read_buf(&mut self.vdata)?;
        let frame = self.unpack_rle().map_err(|_| DecoderError::InvalidData)?;
        Ok((0, Frame::VideoPal(frame, self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);

    let tag = fr.read_tag()?;
    validate!(&tag == b"GAV ");
    let nfrm = fr.read_u32le()?;
    validate!(nfrm > 0);
    let fps = fr.read_u16le()?;
    validate!((1..=30).contains(&fps));
    let width = usize::from(fr.read_u16le()?);
    let height = usize::from(fr.read_u16le()?);
    validate!((1..=320).contains(&width) && (1..=200).contains(&height));
    let global_pal = fr.read_byte()? == 0;
    fr.read_byte()?;

    let mut pal = [0; 768];
    fr.read_vga_pal(&mut pal)?;

    Ok(Box::new(GAVDecoder {
        fr,
        pal, global_pal,
        vdata: Vec::with_capacity(width * height),
        width, height, nfrm, fps,
        frm_no: 0,
    }))
}
