use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::input::util::lzss::*;
use super::super::*;

struct BaseReader {
    fr:         FileReader<BufReader<File>>,
    has_audio:  bool,
    abuf:       Vec<u8>,
    base:       u64,
}

const DATA_BLK: usize = 0x2000;
const AUDIO_BLK: usize = 0x800;
impl BaseReader {
    fn align(&mut self) -> DecoderResult<()> {
        if (self.fr.tell() & 1) != 0 {
            self.fr.read_byte()?;
        }
        Ok(())
    }
    fn read_buf(&mut self, dst: &mut [u8]) -> DecoderResult<()> {
        if !self.has_audio {
            self.fr.read_buf(dst)?;
        } else {
            let mut pos = 0;
            while pos < dst.len() {
                let next_audio = self.base + (DATA_BLK as u64);
                let to_read = ((next_audio - self.fr.tell()) as usize).min(dst.len() - pos);
                if to_read > 0 {
                    self.fr.read_buf(&mut dst[pos..][..to_read])?;
                    pos += to_read;
                } else {
                    self.fr.read_extend(&mut self.abuf, AUDIO_BLK)?;
                    self.base = self.fr.tell();
                }
            }
        }
        Ok(())
    }
    fn skip_null(&mut self) -> DecoderResult<()> {
        let mut skip_len = self.fr.read_u32le()? as usize;
        validate!(skip_len >= 8);
        skip_len -= 8;
        if !self.has_audio {
            self.fr.read_skip(skip_len)?;
        } else {
            let mut pos = 0;
            while pos < skip_len {
                let next_audio = self.base + (DATA_BLK as u64);
                let to_skip = ((next_audio - self.fr.tell()) as usize).min(skip_len - pos);
                if to_skip > 0 {
                    self.fr.read_skip(to_skip)?;
                    pos += to_skip;
                } else {
                    self.fr.read_extend(&mut self.abuf, AUDIO_BLK)?;
                    self.base = self.fr.tell();
                }
            }
        }
        Ok(())
    }
}

struct MicroidsDecoder {
    br:         BaseReader,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    udata:      Vec<u8>,
    width:      usize,
    height:     usize,
    delay:      u32,
    csizes:     Vec<u16>,
    chunk:      usize,
    out_audio:  bool,
    ablk_len:   usize,
}

impl MicroidsDecoder {
    // almost FLI BRUN for pixel pairs
    fn decode_inter(&mut self) -> DecoderResult<()> {
        let src = &self.udata;
        let mut br = MemoryReader::new_read(src);

        let yskip = usize::from(br.read_u16le()?);
        let nlines = usize::from(br.read_u16le()?);
        if nlines == 0 {
            return Ok(());
        }
        validate!(yskip + nlines <= self.height);
        let ops_off = usize::from(br.read_u16le()?);
        validate!(ops_off >= 8 && ops_off <= src.len());
        let mut ops = MemoryReader::new_read(&src[ops_off..]);
        br.read_skip(2)?;
        for line in self.frame.chunks_exact_mut(self.width).skip(yskip).take(nlines) {
            let size = usize::from(ops.read_byte()?);
            if size == 0 { continue; }
            let mut pos = 0;
            for _ in 0..size / 2 {
                let skip = usize::from(ops.read_byte()?);
                pos += skip * 2;
                let count = usize::from(ops.read_byte()?);
                if count < 0x80 {
                    let len = count * 2;
                    validate!(pos + len <= line.len());
                    if len > 0 {
                        br.read_buf(&mut line[pos..][..len])?;
                    }
                    pos += len;
                } else {
                    let len = (256 - count) * 2;
                    validate!(pos + len <= line.len());
                    let mut clr2 = [0; 2];
                    br.read_buf(&mut clr2)?;
                    for dst in line[pos..][..len].chunks_exact_mut(2) {
                        dst.copy_from_slice(&clr2);
                    }
                    pos += len;
                }
            }
            if (size & 1) == 1 {
                let _skip = usize::from(ops.read_byte()?);
            }
        }
        Ok(())
    }
}

impl InputSource for MicroidsDecoder {
    fn get_num_streams(&self) -> usize { if self.br.has_audio { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: self.delay,
                    tb_den: 1000,
                }),
            1 if self.br.has_audio => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    sample_type: AudioSample::U8,
                    channels:    1,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut tag = [0; 4];
        while self.chunk < self.csizes.len() {
            if self.out_audio && self.br.abuf.len() >= self.ablk_len {
                let mut ret = vec![0; self.ablk_len];
                ret.copy_from_slice(&self.br.abuf[..self.ablk_len]);
                self.br.abuf.drain(..self.ablk_len);
                self.out_audio = false;
                return Ok((1, Frame::AudioU8(ret)));
            }
            self.br.align()?;

            let len = usize::from(self.csizes[self.chunk]);
            self.chunk += 1;
            self.br.read_buf(&mut tag)?;
            match &tag {
                b"PAL\x00" => {
                    validate!(len > 0 && len <= self.pal.len());
                    self.br.read_buf(&mut self.pal[..len])?;
                },
                b"IMAG" => {
                    validate!(len == self.frame.len());
                    self.br.read_buf(&mut self.frame)?;
                    self.out_audio = true;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                b"LZIM" => {
                    validate!(len > 5);
                    self.data.resize(len, 0);
                    self.br.read_buf(&mut self.data)?;
                    let unp_size = read_u32le(&self.data).unwrap_or_default() as usize;
                    validate!(unp_size == self.frame.len());
                    lzss_unpack_ext(&self.data[4..], &mut self.frame, 0x20)
                            .map_err(|_| DecoderError::InvalidData)?;
                    self.out_audio = true;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                b"FRM2" => {
                    validate!(len >= 4);
                    self.udata.resize(len, 0);
                    self.br.read_buf(&mut self.udata)?;
                    self.decode_inter().map_err(|_| DecoderError::InvalidData)?;
                    self.out_audio = true;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                b"LZF2" => {
                    validate!(len > 5);
                    self.data.resize(len, 0);
                    self.br.read_buf(&mut self.data)?;
                    let unp_size = read_u32le(&self.data).unwrap_or_default() as usize;
                    self.udata.resize(unp_size, 0);
                    lzss_unpack_ext(&self.data[4..], &mut self.udata, 0x20)
                            .map_err(|_| DecoderError::InvalidData)?;
                    self.decode_inter().map_err(|_| DecoderError::InvalidData)?;
                    self.out_audio = true;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                b"NULL" => {
                    self.chunk -= 1;
                    self.br.skip_null()?;
                },
                _ => return Err(DecoderError::InvalidData),
            }
        }
        Err(DecoderError::EOF)
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    validate!(&tag == b"ANIM");
    let data_start = fr.read_u32le()?;
    validate!(data_start > 0x18);
    let width  = fr.read_u16le()? as usize;
    let height = fr.read_u16le()? as usize;
    validate!((1..=320).contains(&width) && (1..=200).contains(&height));
    let depth = fr.read_u16le()?;
    validate!(depth == 8);
    let nframes = usize::from(fr.read_u16le()?);
    validate!(nframes > 0);
    let delay = fr.read_u32le()?;
    validate!((30..=1000).contains(&delay));
    fr.read_u32le()?;
    fr.read_u16le()?;
    fr.read_u16le()?;
    let nsectors = fr.read_u32le()? as usize;
    let mut csizes = Vec::with_capacity(nframes + 1);
    for _ in 0..=nframes {
        let size = fr.read_u16le()?;
        validate!(size >= 4);
        csizes.push(size - 4);
    }
    fr.seek(SeekFrom::Start(data_start.into()))?;
    let has_audio = nsectors > 0;
    let mut abuf = Vec::new();
    if has_audio {
        abuf.resize(nsectors * 0x800, 0);
        fr.read_buf(&mut abuf)?;
    }
    let base = fr.tell();

    Ok(Box::new(MicroidsDecoder {
        br: BaseReader { fr, abuf, has_audio, base },
        pal: [0; 768],
        data: Vec::new(),
        udata: Vec::new(),
        frame: vec![0; width * height],
        width, height, delay,
        csizes,
        chunk: 0,
        out_audio: false,
        ablk_len: (22050 * delay / 1000) as usize,
    }))
}
