use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct IcomDecoder {
    fr:         FileReader<BufReader<File>>,
    data:       Vec<u8>,
    frame:      Vec<u8>,
    width:      usize,
    height:     usize,
    fps:        u32,
    pal:        [u8; 768],
    arate:      u32,
    abuf:       Vec<u8>,
    audio:      bool,
    ablk_size:  usize,
    next_blk:   u32,
}

impl IcomDecoder {
    fn decode_delta(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        let mut pos = 0;
        while br.left() >= 2 && pos < self.frame.len() {
            let mask = br.read_byte()?;
            if mask == 0x05 {
                let esc = br.read_byte()?;
                if esc == 0x02 {
                    let offset = br.read_u16le()? as usize;
                    pos += offset * 8;
                    continue;
                }
            }

            validate!(pos + 8 <= self.frame.len());
            for i in 0..8 {
                if (mask << i) & 0x80 != 0 {
                    let clr = br.read_byte()?;
                    self.frame[pos + i] = clr;
                }
            }
            pos += 8;
        }
        Ok(())
    }
}

impl InputSource for IcomDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: self.fps,
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = &mut self.fr;
        loop {
            if self.arate > 0 && self.audio && self.abuf.len() >= self.ablk_size {
                let mut audio = vec![0; self.ablk_size];
                audio.copy_from_slice(&self.abuf[..self.ablk_size]);
                self.abuf.drain(..self.ablk_size);
                self.audio = false;
                return Ok((1, Frame::AudioU8(audio)));
            }

            let mut cur_size = br.read_u32le()? as usize;
            let last_frame = cur_size == 0;
            if last_frame { // xxx: maybe handle the very last chunk in a better way
                if self.next_blk == 0 {
                    return Err(DecoderError::EOF);
                }
                validate!(br.tell() + 6 < u64::from(self.next_blk));
                cur_size = (u64::from(self.next_blk) - br.tell() + 4) as usize;
            }
            let chunk_end = br.tell() - 4 + (cur_size as u64);
            let _prev_size = br.read_u32le()? as usize;
            let ctype      = br.read_u16le()?;
            validate!((10..=1048576).contains(&cur_size));

            match ctype {
                0x02 => {
                    validate!(cur_size == 40);
                    let nblk = u32::from(br.read_u16le()?) * 0x800;
                    validate!(nblk == 0 || u64::from(nblk) > br.tell());
                    self.next_blk = nblk;
                    br.read_skip(cur_size - 12)?;
                },
                0x04 => {
                    validate!(cur_size > 32);
                    let _frameno = br.read_u32le()?;
                    let asize = usize::from(br.read_u16le()?);
                    validate!(asize == 0 || asize > 18);
                    if asize > 0 {
                        validate!(asize < cur_size);
                        br.read_extend(&mut self.abuf, asize - 18)?;
                        br.read_skip(2)?;
                    }
                    let pos = br.tell();
                    if pos < chunk_end {
                        self.data.resize((chunk_end - pos) as usize, 0);
                        br.read_buf(&mut self.data)?;
                        self.decode_delta().map_err(|_| DecoderError::InvalidData)?;
                        br = &mut self.fr;
                    }
                    br.seek(SeekFrom::Start(chunk_end))?;

                    if last_frame { // end of block
                        br.seek(SeekFrom::Start(u64::from(self.next_blk)))?;
                        self.next_blk = 0;
                    }

                    self.audio = true;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                0x08 => {
                    validate!(cur_size > 32);
                    let _frameno = br.read_u32le()?;
                    let asize = usize::from(br.read_u16le()?);
                    validate!(asize == 0 || asize > 18);
                    if asize > 0 {
                        validate!(asize < cur_size);
                        br.read_extend(&mut self.abuf, asize - 18)?;
                        br.read_skip(2)?;
                    }
                    let pos = br.tell();
                    validate!(pos + (self.frame.len() as u64) <= chunk_end);
                    br.read_buf(&mut self.frame)?;
                    validate!(br.tell() <= chunk_end);
                    br.seek(SeekFrom::Start(chunk_end))?;

                    if last_frame { // end of block
                        br.seek(SeekFrom::Start(u64::from(self.next_blk)))?;
                        self.next_blk = 0;
                    }

                    self.audio = true;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                0x20 => {
                    validate!(cur_size > 32);
                    let _frameno = br.read_u32le()?;
                    let asize = usize::from(br.read_u16le()?);
                    validate!(asize == 0 || asize > 18);
                    if asize > 0 {
                        validate!(asize < cur_size);
                        br.read_extend(&mut self.abuf, asize - 18)?;
                        br.read_skip(2)?;
                    }
                    let img_size = br.read_u16le()? as usize;
                    let pos = br.tell();
                    validate!(img_size == self.frame.len() + 2);
                    validate!(pos + (img_size as u64) - 2 <= chunk_end);
                    br.read_buf(&mut self.frame)?;
                    validate!(br.tell() + 4 <= chunk_end);
                    let start = br.read_u16le()? as usize;
                    let nclrs = br.read_u16le()? as usize;
                    validate!(start + nclrs <= 256 && br.tell() + (nclrs as u64) * 3 <= chunk_end);
                    if nclrs > 0 {
                        br.read_vga_pal_some(&mut self.pal[start * 3..][..nclrs * 3])?;
                    }
                    br.seek(SeekFrom::Start(chunk_end))?;

                    if last_frame { // end of block
                        br.seek(SeekFrom::Start(u64::from(self.next_blk)))?;
                        self.next_blk = 0;
                    }

                    self.audio = true;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                0x40 => {
                    br.read_skip(4)?;
                    br.read_vga_pal_some(&mut self.pal[..(cur_size - 14).min(768)])?;
                    br.read_skip(cur_size.saturating_sub(0x30E))?;
                },
                _ => { br.read_skip(cur_size - 10)?; },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let size = br.read_u32le()?;
    validate!(size == 40);
    let prev_size = br.read_u32le()?;
    validate!(prev_size == 0);
    let ctype = br.read_u16le()?;
    validate!(ctype == 2);
    let next_blk = u32::from(br.read_u16le()?);
    validate!(next_blk > 0);
    br.read_skip(28)?;

    let size = br.read_u32le()?;
    validate!(size == 28);
    let prev_size = br.read_u32le()?;
    validate!(prev_size == 40);
    let ctype = br.read_u16le()?;
    validate!(ctype == 1);
    br.read_u16le()?;
    br.read_u32le()?;
    let width  = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!((1..=320).contains(&width) && (1..=200).contains(&height));
    let fps = u32::from(br.read_u16le()?);
    validate!((1..=30).contains(&fps));
    let arate = u32::from(br.read_u16le()?);
    br.read_u16le()?;
    br.read_u16le()?;

    Ok(Box::new(IcomDecoder {
        fr: br,
        width, height, fps,
        frame: vec![0; width * height],
        data: Vec::new(),
        pal: [0; 768],
        arate,
        abuf: Vec::new(),
        audio: false,
        ablk_size: (arate / fps).max(1) as usize,
        next_blk: next_blk * 0x800,
    }))
}
