use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

#[derive(Clone,Copy,Debug,PartialEq)]
enum State {
    NewBlock,
    Frame(usize),
    End,
}

struct TsunamiDecoder {
    fr:         FileReader<BufReader<File>>,
    vdata:      Vec<u8>,
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    width:      usize,
    height:     usize,
    fps:        u16,
    fpb:        usize,
    offsets:    Vec<u32>,
    modes:      Vec<[u8; 2]>,
    pal:        [u8; 768],
    state:      State,
    nblocks:    u32,
    frameno:    u32,
    block_pos:  u64,
    has_audio:  bool,
    is_old:     bool,
}

impl TsunamiDecoder {
    fn unpack_rle(dst: &mut [u8], prev: &[u8], br: &mut dyn ByteIO) -> DecoderResult<()> {
        let mut pos = 0;
        while pos < dst.len() {
            let op = br.read_byte()?;
            if (op & 0x80) != 0 {
                let len = usize::from(op & 0x3F);
                validate!(pos + len <= dst.len());
                if (op & 0x40) != 0 {
                    let clr = br.read_byte()?;
                    for el in dst[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                } else {
                    dst[pos..][..len].copy_from_slice(&prev[pos..][..len]);
                }
                pos += len;
            } else {
                let len = usize::from(op);
                validate!(pos + len <= dst.len());
                br.read_buf(&mut dst[pos..][..len])?;
                pos += len;
            }
        }
        Ok(())
    }
    fn output_frame(&self) -> Vec<u8> {
        if self.is_old {
            return self.frame.clone();
        }
        let mut frame = vec![0; self.width * self.height];
        for (dline, sline) in frame.chunks_exact_mut(self.width)
                .zip(self.frame.chunks_exact(self.width)) {
            let (p01, p23) = sline.split_at(self.width / 2);
            let (p0, p1) = p01.split_at(self.width / 4);
            let (p2, p3) = p23.split_at(self.width / 4);
            for (dst, ((&a, &b), (&c, &d))) in dline.chunks_exact_mut(4)
                    .zip(p0.iter().zip(p1.iter()).zip(p2.iter().zip(p3.iter()))) {
                dst[0] = a;
                dst[1] = b;
                dst[2] = c;
                dst[3] = d;
            }
        }
        frame
    }
}

impl InputSource for TsunamiDecoder {
    fn get_num_streams(&self) -> usize { if self.has_audio { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no{
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                 }),
            1 if self.has_audio => StreamInfo::Audio(AudioInfo{
                    sample_rate: 11025,
                    sample_type: AudioSample::U8,
                    channels:    1,
                 }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut line_offs = [0; 256];
        loop {
            match self.state {
                State::NewBlock => {
                    if self.nblocks == 0 {
                        return Err(DecoderError::EOF);
                    }
                    self.nblocks -= 1;
                    self.block_pos = self.fr.tell();
                    let tag = self.fr.read_tag()?;
                    validate!(&tag == b"IVAS");
                    let blk_size = self.fr.read_u32le()?;
                    self.fr.read_skip(40)?;
                    let frm_base = 0x30 + (self.offsets.len() as u32) * 0xC;
                    for (off, mode) in self.offsets.iter_mut().zip(self.modes.iter_mut()) {
                        self.fr.read_u16le()?;
                        *off = self.fr.read_u32le()?;
                        validate!(*off >= frm_base && *off < blk_size);
                        self.fr.read_u32le()?;
                        self.fr.read_buf(mode)?;
                    }
                    let data_start = self.fr.tell();
                    validate!(self.block_pos + u64::from(self.offsets[0]) >= data_start);
                    let asize = (self.block_pos + u64::from(self.offsets[0]) - data_start) as usize;
                    self.state = State::Frame(0);
                    if asize > 0 && self.has_audio {
                        let mut audio = vec![0; asize];
                        self.fr.read_buf(&mut audio)?;
                        return Ok((1, Frame::AudioU8(audio)));
                    }
                },
                State::Frame(fno) => {
                    self.state = if fno + 1 < self.fpb {
                            State::Frame(fno + 1)
                        } else { State::NewBlock };

                    let frame_start = self.block_pos + u64::from(self.offsets[fno]);
                    self.fr.seek(SeekFrom::Start(frame_start)).map_err(|_| DecoderError::EOF)?;
                    for off in line_offs.iter_mut().take(self.height) {
                        *off = self.fr.read_u16le()?;
                    }
                    for ((line, pline), &off) in self.frame.chunks_exact_mut(self.width)
                            .zip(self.pframe.chunks_exact(self.width)).zip(line_offs.iter()) {
                        if off == 0 {
                            continue;
                        }
                        self.fr.seek(SeekFrom::Start(frame_start + u64::from(off)))?;
                        if self.modes[fno][0] == 0 {
                            self.fr.read_buf(line)?;
                        } else {
                            Self::unpack_rle(line, pline, &mut self.fr)
                                .map_err(|_| DecoderError::InvalidData)?;
                        }
                    }
                    if self.modes[fno][1] == 0xFF || self.modes[fno][1] == 0xFE {
                        self.pframe.copy_from_slice(&self.frame);
                    }
                    return Ok((0, Frame::VideoPal(self.output_frame(), self.pal)));
                }
                State::End => return Err(DecoderError::EOF),
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    fr.read_skip(6)?;
    let nblocks = fr.read_u32le()?;
    validate!(nblocks > 0);
    let mut fps = fr.read_u16le()?;
    let fpb = usize::from(fr.read_u16le()?);
    validate!(fpb > 0);
    let is_old = fpb == 1;
    if is_old {
        fps &= 0xF;
    }
    validate!((1..=30).contains(&fps));
    let _draw_type = fr.read_u16le()?;
    fr.read_u16le()?;
    let width = usize::from(fr.read_u16le()?);
    let height = usize::from(fr.read_u16le()?);
    validate!((1..=320).contains(&width) && (1..=240).contains(&height) && (width & 3) == 0);
    fr.read_u16le()?;
    fr.read_u16le()?;
    let _pal_start = usize::from(fr.read_u16le()?);
    let _pal_size  = usize::from(fr.read_u16le()?);
    fr.read_u16le()?;
    let mut pal = [0; 768];
    fr.read_buf(&mut pal)?;
    fr.read_skip(16)?;

    fr.seek(SeekFrom::Start(0x362))?;
    let blk_off = fr.read_u32le()?;
    let has_audio = blk_off > 0x360 + (fpb as u32) * 0xC;
    fr.seek(SeekFrom::Start(0x330))?;

    Ok(Box::new(TsunamiDecoder {
        fr,
        width, height, fps, pal, is_old,
        nblocks, fpb,
        frameno: 0,
        block_pos: 0,
        has_audio,
        offsets: vec![0; fpb],
        modes: vec![[0; 2]; fpb],
        state: State::NewBlock,
        frame: vec![0; width * height],
        pframe: vec![0; width * height],
        vdata: Vec::new(),
    }))
}
