use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const TILE_W: usize = 32;
const TILE_H: usize = 14;

struct KodaDecoder {
    fr:         FileReader<BufReader<File>>,
    audio:      bool,
    ablk_size:  usize,
    abuf:       Vec<u8>,
    vblk_size:  usize,
    vbuf:       Vec<u8>,
    tile_w:     usize,
    tile_h:     usize,
    frame_size: usize,
    fcount:     u8,
    next_off:   u64,
    warned:     bool,
    mode:       u16,
}

impl KodaDecoder {
    fn render_frame(&mut self) -> Vec<u16> {
        const WIDTH: usize = TILE_W * 8;
        let mut frame = vec![0; WIDTH * TILE_H * 8];

        if matches!(self.mode, 0x5253 | 0x5352) {
            return frame;
        }

        let mut pal = [0; 16];
        for (el, src) in pal.iter_mut().zip(self.vbuf.chunks_exact(2)) {
            let clr = read_u16be(src).unwrap_or_default();
            let b = (clr >> 8) & 0xF;
            let g = (clr >> 4) & 0xF;
            let r =  clr       & 0xF;
            *el = (r << 11) | ((r >> 3) << 10)
                | (g <<  6) | ((g >> 3) <<  5)
                | (b <<  1) |  (b >> 3);
        }

        for (strip, tiles) in frame.chunks_exact_mut(WIDTH * 8)
                .zip(self.vbuf[16 * 2..].chunks_exact(self.tile_w * 32)).take(self.tile_h) {
            for (x, tile) in tiles.chunks_exact(32).enumerate() {
                let dpos = x * 8;
                for (line, src) in strip.chunks_exact_mut(WIDTH).zip(tile.chunks_exact(4)) {
                    for (dst, &b) in line[dpos..].chunks_exact_mut(2).zip(src.iter()) {
                        dst[0] = pal[usize::from(b >> 4)];
                        dst[1] = pal[usize::from(b & 0xF)];
                    }
                }
            }
        }

        self.vbuf.drain(..self.frame_size);

        frame
    }
}

impl InputSource for KodaDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  TILE_W * 8,
                    height: TILE_H * 8,
                    bpp:    15,
                    tb_num: 2,
                    tb_den: 15,
                }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 16384,
                    sample_type: AudioSample::U8,
                    channels:    2,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        if self.audio {
            if self.abuf.is_empty() {
                br.seek(SeekFrom::Start(self.next_off)).map_err(|_| DecoderError::EOF)?;
                self.abuf.resize(self.ablk_size, 0);
                br.read_buf(&mut self.abuf)?;
                for el in self.abuf.iter_mut() {
                    if *el < 0x80 {
                        *el = 0x80 - *el;
                    }
                }
            }
            let out_size = self.ablk_size / 8;
            let mut audio = vec![0; out_size];
            audio.copy_from_slice(&self.abuf[..out_size]);
            self.abuf.drain(..out_size);
            self.audio = false;
            Ok((1, Frame::AudioU8(audio)))
        } else {
            if self.fcount == 0 {
                let cur_pos = br.tell();
                // known videos often end with empty 4kB chunk, so check for it first
                br.seek(SeekFrom::Current(0x1000)).map_err(|_| DecoderError::EOF)?;
                br.peek_byte().map_err(|_| DecoderError::EOF)?;
                br.seek(SeekFrom::Start(cur_pos))?;

                self.next_off = cur_pos + (self.vblk_size as u64);
                self.mode = br.peek_u16be()?;
                if self.mode > 0x1000 {
                    br.read_skip(16)?;
                    match self.mode {
                        0x3038 => {
                            self.tile_w = 32;
                            self.tile_h = 14;
                        },
                        0x3135 => {
                            self.tile_w = 15;
                            self.tile_h = 10;
                        },
                        0x4E4F => {
                            self.tile_w = 32;
                            self.tile_h = 14;
                        },
                        0x5253 | 0x5352 => {
                            if !self.warned {
                                println!("credits mode - not supported");
                                self.warned = true;
                            }
                        },
                        _ => return Err(DecoderError::NotImplemented),
                    }
                    self.frame_size = self.tile_w * self.tile_h * 32 + 16 * 2;
                }
                self.vbuf.resize(self.frame_size * 8, 0);
                br.read_buf(&mut self.vbuf)?;
                self.fcount = 8;
            }
            let frame = self.render_frame();
            self.audio = true;
            self.fcount -= 1;
            Ok((0, Frame::VideoRGB16(frame)))
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let fr = FileReader::new_read(BufReader::new(file));

    let (vblk_size, ablk_size) = (0x1D800, 0x8000);

    Ok(Box::new(KodaDecoder {
        fr,
        audio: true,
        abuf: Vec::with_capacity(ablk_size),
        vbuf: Vec::with_capacity(vblk_size),
        vblk_size,
        ablk_size,
        tile_w: 32,
        tile_h: 14,
        frame_size: 32 * 14 * 32 + 16 * 2,
        fcount: 0,
        next_off: 0,
        warned: false,
        mode: 0,
    }))
}
