use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 192;
const FPS: u32 = 14;

#[derive(Debug,PartialEq)]
enum M95State {
    NewBlock,
    Video(u8, u8),
    Audio(u8, u8),
}

#[derive(Clone,Copy)]
struct BlockHeader {
    start:  u16,
    length: u8,
    count:  u8,
}

struct FrameDecoder {
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    clr_buf:    Vec<u8>,
    pal:        [u8; 768],
}

macro_rules! copy_block {
    (selfcopy; $dst:expr, $dst_x:expr, $dst_y:expr, $src_off:expr, $size:expr) => {
        let mut dst_pos = $dst_x + $dst_y * WIDTH;
        validate!($src_off + ($size - 1) * WIDTH + $size <= $dst.len());
        let mut src_pos = $src_off;
        for _ in 0..$size {
            for i in 0..$size {
                $dst[dst_pos + i] = $dst[src_pos + i];
            }
            dst_pos += WIDTH;
            src_pos += WIDTH;
        }
    };
    (offset; $dst:expr, $dst_x:expr, $dst_y:expr, $src:expr, $src_pos:expr, $size:expr) => {
        let dst_pos = $dst_x + $dst_y * WIDTH;
        validate!($src_pos + ($size - 1) * WIDTH + $size <= $dst.len());
        for (dline, sline) in $dst[dst_pos..].chunks_mut(WIDTH)
                .zip($src[$src_pos..].chunks(WIDTH)).take($size) {
            dline[..$size].copy_from_slice(&sline[..$size]);
        }
    };
    (mv; $dst:expr, $dst_x:expr, $dst_y:expr, $src:expr, $mv:expr, $size:expr) => {
        let mut mv_x = ($mv & 0xF) as i8;
        if (mv_x & 8) != 0 { mv_x -= 16; }
        let mut mv_y = ($mv >> 4) as i8;
        if (mv_y & 8) != 0 { mv_y -= 16; }
        let src_pos = ($dst_x as isize) + isize::from(mv_x) + (($dst_y as isize) + isize::from(mv_y)) * (WIDTH as isize);
        validate!(src_pos >= 0);
        copy_block!(offset; $dst, $dst_x, $dst_y, $src, src_pos as usize, $size)
    };
    ($dst:expr, $dst_x:expr, $dst_y:expr, $src:expr, $src_x:expr, $src_y:expr, $size:expr) => {
    }
}

impl FrameDecoder {
    fn unpack_frame(&mut self, br: &mut dyn ByteIO) -> DecoderResult<()> {
        std::mem::swap(&mut self.frame, &mut self.pframe);

        let packed_off = br.read_u16le()?;
        let packed = packed_off != 0;
        if packed {
            let src_pos = br.tell();
            self.clr_buf.clear();
            validate!(packed_off > 2);

            br.read_skip(packed_off as usize - 2)?;
            'lz_unp: loop {
                validate!(self.clr_buf.len() < 1048576);
                let mut flags = br.read_u16le()?;
                for _ in 0..16 {
                    if (flags & 1) == 0 {
                        let word = br.read_u16le()? as usize;
                        let offset = word & 0x3FF;
                        let len = (word >> 10) + 3;
                        if offset == 0 {
                            break 'lz_unp;
                        }
                        validate!(offset <= self.clr_buf.len());
                        let start = self.clr_buf.len() - offset;
                        for i in start..(start + len) {
                            let c = self.clr_buf[i];
                            self.clr_buf.push(c);
                        }
                    } else {
                        let b = br.read_byte()?;
                        self.clr_buf.push(b);
                    }
                    flags >>= 1;
                }
            }
            validate!((self.clr_buf.len() & 0x3F) == 0);
            br.seek(SeekFrom::Start(src_pos))?;
        }
        let mut clr_br = MemoryReader::new_read(&self.clr_buf);

        let mut modes = 0;
        let mut modehi = false;
        let mut clr = [0; 16];
        let frame = &mut self.frame;
        let pframe = &self.pframe;
        for yoff in (0..HEIGHT).step_by(8) {
            for xoff in (0..WIDTH).step_by(8) {
                let mode = if !modehi {
                        modes = br.read_byte()?;
                        modes & 0xF
                    } else {
                        modes >> 4
                    };
                modehi = !modehi;

                match mode {
                    0x1 => {
                        br.read_buf(&mut clr[..8])?;
                        for line in frame[xoff + yoff * WIDTH..].chunks_mut(WIDTH).take(8) {
                            let mut mask = br.read_u24le()? as usize;
                            for el in line[..8].iter_mut() {
                                *el = clr[mask & 7];
                                mask >>= 3;
                            }
                        }
                    },
                    0x2 => {
                        let src_off = br.read_u16le()? as usize;
                        copy_block!(offset; frame, xoff, yoff, pframe, src_off, 8);
                    },
                    0x3 => {
                        copy_block!(offset; frame, xoff, yoff, pframe, xoff + yoff * WIDTH, 8);
                    },
                    0x4 => {
                        validate!(packed);
                        for line in frame[xoff + yoff * WIDTH..].chunks_mut(WIDTH).take(8) {
                            clr_br.read_buf(&mut line[..8])?;
                        }
                    },
                    0x6 => {
                        for sbno in 0..4 {
                            let cur_xoff = xoff + (sbno & 1) * 4;
                            let cur_yoff = yoff + (sbno & 2) * 2;
                            let src_pos = br.read_u16le()? as usize;
                            copy_block!(offset; frame, cur_xoff, cur_yoff, pframe, src_pos, 4);
                        }
                    },
                    0x7 => {
                        for sbno in 0..4 {
                            let cur_xoff = xoff + (sbno & 1) * 4;
                            let cur_yoff = yoff + (sbno & 2) * 2;

                            let src_pos = br.read_u16le()? as usize;
                            copy_block!(selfcopy; frame, cur_xoff, cur_yoff, src_pos, 4);
                        }
                    },
                    0x8 => {
                        br.read_buf(&mut clr[..2])?;
                        for line in frame[xoff + yoff * WIDTH..].chunks_mut(WIDTH).take(8) {
                            let mut mask = usize::from(br.read_byte()?);
                            for el in line[..8].iter_mut() {
                                *el = clr[mask & 1];
                                mask >>= 1;
                            }
                        }
                    },
                    0x9 => {
                        for sbno in 0..16 {
                            let cur_xoff = xoff + (sbno & 3) * 2;
                            let cur_yoff = yoff + (sbno >> 2) * 2;

                            let mv = br.read_byte()?;
                            copy_block!(mv; frame, cur_xoff, cur_yoff, pframe, mv, 2);
                        }
                    },
                    0xA => {
                        for sbno in 0..4 {
                            let cur_xoff = xoff + (sbno & 1) * 4;
                            let cur_yoff = yoff + (sbno & 2) * 2;

                            let dst_pos = cur_xoff + cur_yoff * WIDTH;
                            br.read_buf(&mut clr[..2])?;
                            let mut mask = br.read_u16le()? as usize;
                            for line in frame[dst_pos..].chunks_mut(WIDTH).take(4) {
                                for el in line[..4].iter_mut() {
                                    *el = clr[mask & 1];
                                    mask >>= 1;
                                }
                            }
                        }
                    },
                    0xB => {
                        for sbno in 0..4 {
                            let cur_xoff = xoff + (sbno & 1) * 4;
                            let cur_yoff = yoff + (sbno & 2) * 2;

                            let dst_pos = cur_xoff + cur_yoff * WIDTH;
                            br.read_buf(&mut clr[..4])?;
                            let mut mask = br.read_u16le()?;
                            for (y, line) in frame[dst_pos..].chunks_mut(WIDTH)
                                    .take(4).enumerate() {
                                let clr0 = if y < 2 { clr[0] } else { clr[3] };
                                for (x, el) in line[..4].iter_mut().enumerate() {
                                    let clr1 = if x < 2 { clr[1] } else { clr[2] };
                                    *el = if (mask & 1) != 0 { clr1 } else { clr0 };
                                    mask >>= 1;
                                }
                            }
                        }
                    },
                    0xC => {
                        for sbno in 0..4 {
                            let cur_xoff = xoff + (sbno & 1) * 4;
                            let cur_yoff = yoff + (sbno >> 1) * 4;

                            let mv = br.read_byte()?;
                            copy_block!(mv; frame, cur_xoff, cur_yoff, pframe, mv, 4);
                        }
                    },
                    0xD => {
                        for sbno in 0..4 {
                            let cur_xoff = xoff + (sbno & 1) * 4;
                            let cur_yoff = yoff + (sbno & 2) * 2;

                            let dst_pos = cur_xoff + cur_yoff * WIDTH;
                            br.read_buf(&mut clr[..4])?;
                            let mut mask = br.read_u32le()? as usize;
                            for line in frame[dst_pos..].chunks_mut(WIDTH).take(4) {
                                for el in line[..4].iter_mut() {
                                    *el = clr[mask & 3];
                                    mask >>= 2;
                                }
                            }
                        }
                    },
                    0xE => {},
                    0xF => {
                        for line in frame[xoff + yoff * WIDTH..].chunks_mut(WIDTH).take(8) {
                            br.read_buf(&mut line[..8])?;
                        }
                    },
                    _ => {
                        println!(" unknown mode {mode:X}");
                        return Err(DecoderError::InvalidData);
                    }
                }
            }
        }
        Ok(())
    }
}

struct M95Decoder {
    fr:         FileReader<File>,
    blk_tab:    Vec<BlockHeader>,
    cur_blk:    usize,
    blk_start:  u64,
    state:      M95State,
    new_pal:    [u8; 768],
    pal_frm:    u8,
    foffsets:   [u32; 14],
    fdec:       FrameDecoder,
    arate:      u32,
    abits:      u16,
    channels:   u16,
    abuf:       Vec<u8>,
    ablk_size:  usize,
}

fn load_wav(br: &mut dyn ByteIO, abuf: &mut Vec<u8>, arate: u32, abits: u16, channels: u16) -> DecoderResult<()> {
    let tag = br.read_tag()?;
    validate!(&tag == b"RIFF");
    br.read_u32le()?;
    let tag = br.read_tag()?;
    validate!(&tag == b"WAVE");
    let tag = br.read_tag()?;
    validate!(&tag == b"fmt ");
    let wfx_size = br.read_u32le()?;
    validate!(wfx_size == 16);
    let format_tag = br.read_u16le()?;
    if format_tag != 1 {
        return Err(DecoderError::NotImplemented);
    }
    let wchannels = br.read_u16le()?;
    let warate = br.read_u32le()?;
    br.read_u32le()?;
    br.read_u16le()?;
    let wabits = br.read_u16le()?;
    validate!(arate == warate && channels == wchannels && abits == wabits);
    let tag = br.read_tag()?;
    validate!(&tag == b"data");
    let size = br.read_u32le()? as usize;
    validate!((1..=1048576).contains(&size));
    br.read_extend(abuf, size)?;
    Ok(())
}

impl InputSource for M95Decoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: FPS,
                }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    sample_type: if self.abits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                    channels:    self.channels as u8,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        loop {
            match self.state {
                M95State::NewBlock => {
                    if self.cur_blk >= self.blk_tab.len() {
                        return Err(DecoderError::EOF);
                    }
                    let blk = &self.blk_tab[self.cur_blk];
                    self.cur_blk += 1;
                    self.blk_start = u64::from(blk.start) * 0x800;
                    validate!(br.tell() <= self.blk_start);
                    br.seek(SeekFrom::Start(self.blk_start))?;

                    for offs in self.foffsets.iter_mut() {
                        *offs = br.read_u32le()?;
                    }
                    br.read_u32le()?;
                    br.read_u32le()?;
                    let wav_offset = br.read_u32le()?;
                    br.read_u32le()?;
                    br.read_u32le()?;
                    br.read_u32le()?;
                    let pal_offset = br.read_u32le()?;
                    br.read_u32le()?;
                    br.read_u32le()?;

                    if pal_offset > 0 {
                        validate!(pal_offset > 0x5C);
                        br.seek(SeekFrom::Start(self.blk_start + u64::from(pal_offset)))?;
                        self.pal_frm = br.read_byte()?;
                        br.read_buf(&mut self.new_pal)?;
                    }

                    if wav_offset > 0 {
                        validate!(wav_offset > 0x5C);
                        validate!(self.arate > 0);
                        br.seek(SeekFrom::Start(self.blk_start + u64::from(wav_offset)))?;
                        load_wav(&mut *br, &mut self.abuf, self.arate, self.abits, self.channels)
                            .map_err(|_| DecoderError::InvalidData)?;
                    }

                    self.state = M95State::Video(0, blk.count);
                },
                M95State::Video(frameno, count) => {
                    let offs = self.foffsets[usize::from(frameno)];
                    validate!(offs >= 0x5C);
                    if frameno == self.pal_frm {
                        self.fdec.pal.copy_from_slice(&self.new_pal);
                    }
                    br.seek(SeekFrom::Start(self.blk_start + u64::from(offs)))?;
                    self.fdec.unpack_frame(br).map_err(|_| DecoderError::InvalidData)?;

                    self.state = M95State::Audio(frameno, count);
                    return Ok((0, Frame::VideoPal(self.fdec.frame.clone(), self.fdec.pal)));
                },
                M95State::Audio(frameno, count) => {
                    self.state = if count > 1 {
                            M95State::Video(frameno + 1, count - 1)
                        } else { M95State::NewBlock };
                    if self.arate > 0 && self.abuf.len() >= self.ablk_size {
                        let mut audio = vec![0; self.ablk_size];
                        audio.copy_from_slice(&self.abuf[..self.ablk_size]);
                        self.abuf.drain(..self.ablk_size);
                        return Ok((1, Frame::AudioU8(audio)));
                    }
                },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(file);

    let mut blk_tab = Vec::with_capacity(1024);
    let mut next_blk = 2;
    for _ in 0..1024 {
        let start  = br.read_u16le()?;
        let length = br.read_byte()?;
        let count  = br.read_byte()?;
        if start == 0 && length == 0 && count == 0 {
            break;
        }
        let end = start + u16::from(length);
        validate!(start >= next_blk);
        blk_tab.push(BlockHeader { start, length, count });
        next_blk = end;
    }
    validate!(!blk_tab.is_empty());

    // check for audio
    br.seek(SeekFrom::Start(u64::from(blk_tab[0].start) * 0x800 + 0x40))?;
    let audio_off = br.read_u32le()?;
    let mut arate = 0;
    let mut abits = 0;
    let mut channels = 0;
    if audio_off > 0 {
        validate!(audio_off > 0x5C);
        br.read_skip(audio_off as usize - 0x44)?;
        let tag = br.read_tag()?;
        validate!(&tag == b"RIFF");
        br.read_u32le()?;
        let tag = br.read_tag()?;
        validate!(&tag == b"WAVE");
        let tag = br.read_tag()?;
        validate!(&tag == b"fmt ");
        let wfx_size = br.read_u32le()?;
        validate!(wfx_size == 16);
        let format_tag = br.read_u16le()?;
        if format_tag == 1 {
            channels = br.read_u16le()?;
            arate = br.read_u32le()?;
            br.read_u32le()?;
            br.read_u16le()?;
            abits = br.read_u16le()?;
            validate!((8000..=32000).contains(&arate));
            validate!(channels == 1 || channels == 2);
            validate!(abits == 8 || abits == 16);
            if abits != 8 {
                println!("unexpected audio format {abits} bits, ignoring");
                arate = 0;
            }
        } else {
            println!("unexpected audio codec {format_tag}, ignoring");
        }
    }
    br.seek(SeekFrom::Start(0))?;

    Ok(Box::new(M95Decoder {
        fr: br,
        new_pal: [0; 768],
        pal_frm: 0,
        blk_tab,
        cur_blk: 0,
        blk_start: 0,
        state: M95State::NewBlock,
        foffsets: [0; 14],
        fdec: FrameDecoder {
            pal: [0; 768],
            frame: vec![0; WIDTH * HEIGHT],
            pframe: vec![0; WIDTH * HEIGHT],
            clr_buf: Vec::new(),
        },
        arate, abits, channels,
        abuf: Vec::new(),
        ablk_size: (arate / FPS).max(1) as usize * (channels as usize),
    }))
}
