use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 240;

struct FMVDecoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    vdata:      Vec<u8>,
    nframes:    u16,
    frameno:    u16,
}

impl FMVDecoder {
    fn decode_frame(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        let num_quads = br.read_u32le()? as usize;
        let mut dpos = 0;
        let mut opcodes = [0; 4];
        for _ in 0..num_quads {
            for el in opcodes.iter_mut() {
                *el = usize::from(br.read_byte()?) * 4;
            }
            dpos += opcodes[0];
            if dpos == self.frame.len() { break; }
            validate!(dpos + opcodes[1] <= self.frame.len());
            if opcodes[1] > 0 {
                br.read_buf(&mut self.frame[dpos..][..opcodes[1]])?;
            }
            dpos += opcodes[1];
            if dpos == self.frame.len() { break; }
            dpos += opcodes[2];
            if dpos == self.frame.len() { break; }
            validate!(dpos + opcodes[3] <= self.frame.len());
            if opcodes[3] > 0 {
                br.read_buf(&mut self.frame[dpos..][..opcodes[3]])?;
            }
            dpos += opcodes[3];
        }

        Ok(())
    }
}

impl InputSource for FMVDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 15,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        let size = br.read_u32le().map_err(|_| DecoderError::EOF)? as usize;
        self.vdata.resize(size + 4, 0);
        br.read_buf(&mut self.vdata)?;

        self.decode_frame().map_err(|_| DecoderError::InvalidData)?;

        let pal = self.pal;
        let frame = self.frame.clone();
        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);

    let tag = fr.read_tag()?;
    validate!(&tag == b"AFMV");
    let nframes = fr.read_u16le()?;
    let fill = fr.read_byte()?;
    let mut pal = [0; 768];
    fr.read_buf(&mut pal)?;
    fr.read_skip(1)?;

    Ok(Box::new(FMVDecoder {
        fr,
        pal,
        frame: vec![fill; WIDTH * HEIGHT],
        vdata: Vec::new(),
        frameno: 0,
        nframes,
    }))
}
