use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 180;

struct FMVDecoder {
    fr:         FileReader<File>,
    pal:        [u16; 256],
    frameno:    usize,
    fsizes:     Vec<usize>,
    frame:      Vec<u16>,
    vdata:      Vec<u8>,
    unp_buf:    Vec<u8>,
    fmv2:       bool,
}

struct PPBitReader<'a> {
    src: &'a [u8],
    pos: usize,
    bitpos: u8,
}

impl<'a> PPBitReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        let pos = src.len();
        Self {
            src, pos,
            bitpos: 0,
        }
    }
    fn read(&mut self, bits: u8) -> DecoderResult<u16> {
        let mut val = 0;
        for _ in 0..bits {
            if self.bitpos > 0 {
                self.bitpos -= 1;
            } else {
                self.bitpos = 7;
                if self.pos == 0 {
                    return Err(DecoderError::ShortData);
                }
                self.pos -= 1;
            }
            let bit = (self.src[self.pos] >> (7 - self.bitpos)) & 1;
            val = val * 2 + u16::from(bit);
        }
        Ok(val)
    }
}

impl FMVDecoder {
    // Power Packer 2 decrunching
    fn pp2_unpack(src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<()> {
        validate!(src.len() >= 16);
        validate!(&src[..4] == b"PP20");
        let len_bits = &src[4..8];
        let mut tail = src[src.len() - 1];
        let dst_size = read_u24be(&src[src.len() - 4..]).unwrap_or(0) as usize;
        validate!(dst_size > 0);

        dst.resize(dst_size, 0);

        let mut dpos = dst_size;
        let mut br = PPBitReader::new(&src[8..src.len() - 4]);
        while tail >= 8 {
            tail -= 8;
            br.read(8)?;
        }
        br.read(tail)?;
        while dpos > 0 {
            if br.read(1)? == 0 {
                let mut len = 1;
                loop {
                    let add = br.read(2)?;
                    len += add as usize;
                    if add != 3 {
                        break;
                    }
                }
                validate!(dpos >= len);
                for el in dst[dpos - len..dpos].iter_mut().rev() {
                    *el = br.read(8)? as u8;
                }
                dpos -= len;
                if dpos == 0 {
                    break;
                }
            }

            let mut len = br.read(2)? as usize + 2;
            let offlen = len_bits[len - 2];
            validate!((1..=16).contains(&offlen));
            let off = if len < 5 {
                    br.read(offlen)? as usize
                } else {
                    let offlen = if br.read(1)? == 0 { 7 } else { offlen };
                    let offset = br.read(offlen)? as usize;

                    loop {
                        let add = br.read(3)?;
                        len += add as usize;
                        if add != 7 {
                            break;
                        }
                    }
                    offset
                };
            validate!(dpos >= len);
            validate!(dpos + off < dst.len());
            for i in 0..len {
                dst[dpos - i - 1] = dst[dpos + off - i];
            }
            dpos -= len;
        }

        Ok(())
    }
    fn decode_frame(src: &[u8], dst: &mut [u16], pal: &mut [u16; 256], fmv2: bool) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(src);

        let (stride, height, tsize) = if fmv2 {
                (WIDTH * 2, HEIGHT * 2, 8)
            } else {
                (WIDTH, HEIGHT, 4)
            };

        if br.peek_byte()? == 0x80 {
            br.read_skip(1)?;
            let nentries = usize::from(br.read_byte()?) + 1;
            let mut rgb = [0; 3];
            for el in pal[..nentries].iter_mut() {
                br.read_buf(&mut rgb)?;
                *el = (u16::from(rgb[0]) << 10) | (u16::from(rgb[1]) << 5) | u16::from(rgb[2]);
            }
        }

        if br.peek_byte()? == 0x20 { // skip frame
            return Ok(());
        }

        let mut xpos = 0;
        let mut ypos = 0;
        while br.left() > 0 {
            let op = br.read_byte()?;
            match op {
                0x00 => {
                    xpos += tsize;
                    if xpos == stride {
                        xpos = 0;
                        ypos += tsize;
                    }
                },
                0x21 => {
                    let nrows = usize::from(br.read_byte()?);
                    validate!((nrows & 7) == 0);
                    ypos += if fmv2 { nrows } else { nrows / 2 };
                    xpos = 0;
                },
                0x42 => {
                    validate!(ypos < height);
                    for line in dst[xpos + ypos * stride..].chunks_mut(stride).take(tsize) {
                        for el in line[..tsize].iter_mut() {
                            *el = pal[usize::from(br.read_byte()?)];
                        }
                    }
                    xpos += tsize;
                    if xpos == stride {
                        xpos = 0;
                        ypos += tsize;
                    }
                },
                0x43 => {
                    validate!(ypos < height);
                    for quarter in 0..4 {
                        let subop = br.read_byte()?;
                        let dst_pos = xpos + ypos * stride + (quarter & 1) * (tsize / 2)
                                     + (quarter >> 1) * (tsize / 2) * stride;
                        match subop {
                            0x42 => {
                                for line in dst[dst_pos..].chunks_mut(stride).take(tsize / 2) {
                                    for el in line[..tsize / 2].iter_mut() {
                                        *el = pal[usize::from(br.read_byte()?)];
                                    }
                                }
                            },
                            0x0..=0xF if !fmv2 => {
                                if (subop & 0x1) != 0 {
                                    dst[dst_pos] = pal[usize::from(br.read_byte()?)];
                                }
                                if (subop & 0x2) != 0 {
                                    dst[dst_pos + 1] = pal[usize::from(br.read_byte()?)];
                                }
                                if (subop & 0x4) != 0 {
                                    dst[dst_pos + stride] = pal[usize::from(br.read_byte()?)];
                                }
                                if (subop & 0x8) != 0 {
                                    dst[dst_pos + stride + 1] = pal[usize::from(br.read_byte()?)];
                                }
                            },
                            0x0..=0xF => {
                                if (subop & 0x1) != 0 {
                                    dst[dst_pos] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + 1] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + stride] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + stride + 1] = pal[usize::from(br.read_byte()?)];
                                }
                                if (subop & 0x2) != 0 {
                                    dst[dst_pos + 2] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + 3] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + 2 + stride] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + 3 + stride] = pal[usize::from(br.read_byte()?)];
                                }
                                if (subop & 0x4) != 0 {
                                    dst[dst_pos + stride * 2] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + stride * 2 + 1] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + stride * 3] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + stride * 3 + 1] = pal[usize::from(br.read_byte()?)];
                                }
                                if (subop & 0x8) != 0 {
                                    dst[dst_pos + stride * 2 + 2] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + stride * 2 + 3] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + stride * 3 + 2] = pal[usize::from(br.read_byte()?)];
                                    dst[dst_pos + stride * 3 + 3] = pal[usize::from(br.read_byte()?)];
                                }
                            },
                            _ => return Err(DecoderError::InvalidData),
                        }
                    }
                    xpos += tsize;
                    if xpos == stride {
                        xpos = 0;
                        ypos += tsize;
                    }
                },
                _ => return Err(DecoderError::InvalidData),
            }
        }

        Ok(())
    }
}

impl InputSource for FMVDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  if !self.fmv2 { WIDTH } else { WIDTH * 2 },
                    height: if !self.fmv2 { HEIGHT } else { HEIGHT * 2 },
                    bpp:    15,
                    tb_num: 1,
                    tb_den: 10,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frameno >= self.fsizes.len() {
            return Err(DecoderError::EOF);
        }

        let br = &mut self.fr;
        let size = self.fsizes[self.frameno];
        self.frameno += 1;
        self.vdata.resize(size, 0);
        br.read_buf(&mut self.vdata)?;
        if self.fmv2 {
            Self::pp2_unpack(&self.vdata, &mut self.unp_buf).map_err(|_| DecoderError::InvalidData)?;
            Self::decode_frame(&self.unp_buf, &mut self.frame, &mut self.pal, true)
                .map_err(|_| DecoderError::InvalidData)?;
        } else {
            Self::decode_frame(&self.vdata, &mut self.frame, &mut self.pal, false)
                .map_err(|_| DecoderError::InvalidData)?;
        }

        let frame = self.frame.clone();
        Ok((0, Frame::VideoRGB16(frame)))
    }
}

pub fn open1(name: &str) -> DecoderResult<Box<dyn InputSource>> { open(name, false) }
pub fn open2(name: &str) -> DecoderResult<Box<dyn InputSource>> { open(name, true) }

fn open(name: &str, fmv2: bool) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);

    let nframes = fr.read_u32le()? as usize;
    let mut fsizes = Vec::with_capacity(nframes);
    for _ in 0..nframes {
        fsizes.push(fr.read_u32le()? as usize);
    }
    let (width, height) = if !fmv2 { (WIDTH, HEIGHT) } else { (WIDTH * 2, HEIGHT * 2) };

    Ok(Box::new(FMVDecoder {
        fr,
        pal: [0; 256],
        frame: vec![0; width * height],
        vdata: Vec::new(),
        unp_buf: Vec::new(),
        frameno: 0,
        fsizes,
        fmv2,
    }))
}
