use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const SAMPLERATE: u32 = 11025;

struct PacoDecoder {
    fr:         FileReader<BufReader<File>>,
    data:       Vec<u8>,
    frame:      Vec<u8>,
    pal:        [u8; 768],
    has_video:  bool,
    fps:        u16,
    width:      usize,
    height:     usize,
    pkt_sizes:  Vec<u32>,
    pkt_no:     usize,
    has_audio:  bool,
    audio:      Vec<u8>,
    afrm_size:  usize,
}

macro_rules! rle_op {
    (copy; $frame:expr, $stride:expr, $x:expr, $y:expr, $right:expr, $len:expr, $br:expr) => ({
        validate!($x + $len <= $right);
        $br.read_buf(&mut $frame[$x + $y * $stride..][..$len])?;
        $x += $len;
    });
    (run; $frame:expr, $stride:expr, $x:expr, $y:expr, $right:expr, $len:expr, $c:expr) => ({
        validate!($x + $len <= $right);
        for el in $frame[$x + $y * $stride..][..$len].iter_mut() {
            *el = $c;
        }
        $x += $len;
    });
    (pair; $frame:expr, $stride:expr, $x:expr, $y:expr, $right:expr, $len:expr, $clrs:expr) => ({
        validate!($x + $len * 2 <= $right);
        for pair in $frame[$x + $y * $stride..][..$len * 2].chunks_exact_mut(2) {
            pair.copy_from_slice(&$clrs);
        }
        $x += $len * 2;
    });
    (quad; $frame:expr, $stride:expr, $x:expr, $y:expr, $right:expr, $len:expr, $clrs:expr) => ({
        validate!($x + $len * 4 <= $right);
        for quad in $frame[$x + $y * $stride..][..$len * 4].chunks_exact_mut(4) {
            quad.copy_from_slice(&$clrs);
        }
        $x += $len * 4;
    })
}

impl PacoDecoder {
    fn adjust_line(ypos: &mut usize, add: usize, top: usize, height: usize, mode: u8) {
        if mode == 1 {
            *ypos += add;
        } else {
            *ypos += add * 2;
            if *ypos >= top + height {
                *ypos -= height - (!height & 1);
            }
        }
    }
    fn decode_video(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = MemoryReader::new_read(&self.data);

        let left   = br.read_u16be()? as usize;
        let top    = br.read_u16be()? as usize;
        let width  = br.read_u16be()? as usize;
        let height = br.read_u16be()? as usize;
        validate!(left + width <= self.width && top + height <= self.height);
        let method = br.read_byte()?;
                     br.read_byte()?;
        if !matches!(method & 0x1F, 1 | 2) {
            return Err(DecoderError::NotImplemented);
        }

        let mut cclrs = [0; 16];

        let mut pos_y = top;
        let right = left + width;
        let mut lineno = 0;
        'line_loop: while lineno < height {
            let mut pos_x = left;
            let mut compact = false;
            while pos_x < right {
                let op = br.read_byte()?;
                if !compact {
                    match op {
                        0x00 => { // long mode
                            let op = br.read_u16be()?;
                            let len = (op & 0xFFF) as usize;
                            match op >> 12 {
                                0 => rle_op!(copy; self.frame, self.width, pos_x, pos_y, right, len, br),
                                1 => {
                                    let clr = br.read_byte()?;
                                    rle_op!(run; self.frame, self.width, pos_x, pos_y, right, len, clr);
                                },
                                2 => {
                                    let mut clrs = [0; 2];
                                    br.read_buf(&mut clrs)?;
                                    rle_op!(pair; self.frame, self.width, pos_x, pos_y, right, len, clrs);
                                },
                                3 => {
                                    let mut clrs = [0; 4];
                                    br.read_buf(&mut clrs)?;
                                    rle_op!(quad; self.frame, self.width, pos_x, pos_y, right, len, clrs);
                                },
                                4 => {
                                    validate!(pos_x + len <= right);
                                    pos_x += len;
                                },
                                5 => {
                                    validate!(lineno + len < height);
                                    Self::adjust_line(&mut pos_y, len + 1, top, height, method & 0xF);
                                    lineno += len + 1;
                                    continue 'line_loop;
                                },
                                15 => break 'line_loop,
                                _ => unimplemented!(),
                            }
                        },
                        0x01..=0x7F => {
                            rle_op!(copy; self.frame, self.width, pos_x, pos_y, right, usize::from(op), br);
                        },
                        0x80..=0xFD => {
                            let c = br.read_byte()?;
                            rle_op!(run; self.frame, self.width, pos_x, pos_y, right, 256 - usize::from(op), c);
                        },
                        0xFE => { // skip or compact mode
                            let val = usize::from(br.read_byte()?);
                            if val > 0 {
                                validate!(pos_x + val <= right);
                                pos_x += val;
                            } else {
                                validate!((method & 0x20) != 0);
                                compact = true;

                                let mut mask = br.read_u16be()?;
                                for clr in cclrs.iter_mut() {
                                    if (mask & 1) == 1 {
                                        *clr = br.read_byte()?;
                                    }
                                    mask >>= 1;
                                }
                            }
                        },
                        0xFF => { // pair/quad run
                            let val = br.read_byte()?;
                            if (val & 0x80) == 0 {
                                let mut clrs = [0; 2];
                                br.read_buf(&mut clrs)?;
                                rle_op!(pair; self.frame, self.width, pos_x, pos_y, right, usize::from(val), clrs);
                            } else {
                                let mut clrs = [0; 4];
                                br.read_buf(&mut clrs)?;
                                rle_op!(quad; self.frame, self.width, pos_x, pos_y, right, 256 - usize::from(val), clrs);
                            }
                        },
                    }
                } else {
                    let len = usize::from(op & 0xF);
                    match op >> 4 {
                        0x0 => {
                            let op = len;
                            let len = usize::from(br.read_byte()?);
                            match op {
                                0 => {
                                    let mut idx = 0;
                                    let mut hi_nib = true;
                                    validate!(pos_x + len <= right);
                                    for el in self.frame[pos_x + pos_y * self.width..][..len].iter_mut() {
                                        if hi_nib {
                                            idx = usize::from(br.read_byte()?);
                                            *el = cclrs[idx >> 4];
                                        } else {
                                            *el = cclrs[idx & 0xF];
                                        }
                                        hi_nib = !hi_nib;
                                    }
                                    pos_x += len;
                                },
                                1 => {
                                    let clr = cclrs[usize::from(br.read_byte()?) & 0xF];
                                    rle_op!(run; self.frame, self.width, pos_x, pos_y, right, len, clr);
                                },
                                2 => {
                                    let idx = usize::from(br.read_byte()?);
                                    let clrs = [cclrs[idx >> 4], cclrs[idx & 0xF]];
                                    rle_op!(pair; self.frame, self.width, pos_x, pos_y, right, len, clrs);
                                },
                                3 => {
                                    let idx0 = usize::from(br.read_byte()?);
                                    let idx1 = usize::from(br.read_byte()?);
                                    let clrs = [cclrs[idx0 >> 4], cclrs[idx0 & 0xF], cclrs[idx1 >> 4], cclrs[idx1 & 0xF]];
                                    rle_op!(quad; self.frame, self.width, pos_x, pos_y, right, len, clrs);
                                },
                                4 => {
                                    validate!(pos_x + len <= right);
                                    pos_x += len;
                                },
                                5 => {
                                    validate!(lineno + len < height);
                                    Self::adjust_line(&mut pos_y, len + 1, top, height, method & 0xF);
                                    lineno += len + 1;
                                    continue 'line_loop;
                                },
                                _ => return Err(DecoderError::NotImplemented),
                            }
                        },
                        0x1..=0x7 => {
                            let mut idx = len;
                            let mut hi_nib = false;
                            let len = usize::from(op >> 4);
                            validate!(pos_x + len <= right);
                            for el in self.frame[pos_x + pos_y * self.width..][..len].iter_mut() {
                                if hi_nib {
                                    idx = usize::from(br.read_byte()?);
                                    *el = cclrs[idx >> 4];
                                } else {
                                    *el = cclrs[idx & 0xF];
                                }
                                hi_nib = !hi_nib;
                            }
                            pos_x += len;
                        },
                        0x8..=0xD => {
                            rle_op!(run; self.frame, self.width, pos_x, pos_y, right, 16 - usize::from(op >> 4), cclrs[len]);
                        },
                        0xE => {
                            validate!(pos_x + len <= right);
                            pos_x += len;
                        },
                        _ if len < 8 => {
                            let idx = usize::from(br.read_byte()?);
                            let clrs = [cclrs[idx >> 4], cclrs[idx & 0xF]];
                            rle_op!(pair; self.frame, self.width, pos_x, pos_y, right, len, clrs);
                        },
                        _ => {
                            let idx0 = usize::from(br.read_byte()?);
                            let idx1 = usize::from(br.read_byte()?);
                            let clrs = [cclrs[idx0 >> 4], cclrs[idx0 & 0xF], cclrs[idx1 >> 4], cclrs[idx1 & 0xF]];
                            rle_op!(quad; self.frame, self.width, pos_x, pos_y, right, 16 - len, clrs);
                        }
                    }
                }
            }
            Self::adjust_line(&mut pos_y, 1, top, height, method & 0xF);
            lineno += 1;
        }

        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
    fn get_audio(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut frm = vec![0; self.afrm_size];
        frm.copy_from_slice(&self.audio[..self.afrm_size]);
        self.audio.drain(..self.afrm_size);
        Ok((if self.has_video { 1 } else { 0 }, Frame::AudioU8(frm)))
    }
}

impl InputSource for PacoDecoder {
    fn get_num_streams(&self) -> usize { if self.has_audio { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 if self.has_video => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                 }),
            0 if !self.has_video => StreamInfo::Audio(AudioInfo{
                    sample_rate: SAMPLERATE,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            1 if self.has_audio => StreamInfo::Audio(AudioInfo{
                    sample_rate: SAMPLERATE,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.has_audio && self.audio.len() >= self.afrm_size {
            return self.get_audio();
        }

        if self.pkt_no >= self.pkt_sizes.len() {
            return Err(DecoderError::EOF);
        }

        let br = &mut self.fr;
        let pkt_end = br.tell() + u64::from(self.pkt_sizes[self.pkt_no]);
        self.pkt_no += 1;

        while br.tell() < pkt_end {
            let ctype = br.read_byte()?;
            let csize = br.read_u24be()? as usize;
            validate!(csize >= 4);
            let csize = csize - 4;
            match ctype {
                2 => {
                    validate!(csize >= 8);
                    let pal_type = br.read_byte()?;
                                   br.read_skip(7)?;
                    match pal_type {
                        0x10 => {
                            validate!(csize > 8 && csize <= 768 + 8);
                            br.read_buf(&mut self.pal[..csize - 8])?;
                        },
                        0x30 => {
                            self.pal = PACO_PAL;
                            br.read_skip(csize - 8)?;
                        },
                        _ => {
                            println!("Unknown palette mode {pal_type}");
                            br.read_skip(csize - 8)?;
                        }
                    }
                },
                4 if self.has_audio => {
                    br.read_skip(4)?;
                    br.read_extend(&mut self.audio, csize - 4)?;
                },
                8 => {
                    self.data.resize(csize, 0);
                    br.read_buf(&mut self.data)?;
                },
                11 => break,
                _ => br.read_skip(csize)?,
            }
        }
        validate!(br.tell() == pkt_end);

        if self.has_video {
            self.decode_video().map_err(|err|
                    if err == DecoderError::ShortData { DecoderError::InvalidData } else { err })
        } else if self.has_audio {
            self.get_audio()
        } else {
            Err(DecoderError::Bug)
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let magic1                      = br.read_u16be()?;
    let magic26                     = br.read_u16be()?;
    validate!(magic1 == 1 && magic26 == 0x26);

    let width                       = br.read_u16be()? as usize;
    let height                      = br.read_u16be()? as usize;
    let has_video = width > 0 && height > 0;
    validate!(width <= 1024 && height <= 768);
    let fps                         = br.read_u16be()? as i16;
    let has_audio = fps < 0;
    if !has_video && !has_audio {
        return Err(DecoderError::InvalidData);
    }
    let fps = if fps != 0 { fps.unsigned_abs() } else { 10 };
                                      br.read_u16be()?; // flags
                                      br.read_u32be()?; // maximum chunk size
                                      br.read_u32be()?;
                                      br.read_u32be()?;
    let nframes                     = br.read_u16be()? as usize;
                                      br.read_u16be()?;
    validate!(nframes > 0);
                                      br.read_u16be()?; // always 8?
                                      br.read_u16be()?; // always 0x600?
                                      br.read_u32be()?; // flags
                                      br.read_u16be()?; // always 0?

    let mut pkt_sizes = Vec::with_capacity(nframes);
    for _ in 0..nframes {
        let size                    = br.read_u32be()?;
        pkt_sizes.push(size);
    }

    let afrm_size = ((SAMPLERATE / u32::from(fps) + 3) & !3) as usize;

    Ok(Box::new(PacoDecoder {
        fr: br,
        frame: vec![0; width * height],
        data: Vec::new(),
        has_video, width, height, fps,
        pal: PACO_PAL,
        pkt_sizes,
        pkt_no: 0,
        has_audio,
        audio: Vec::new(),
        afrm_size,
    }))
}

// standard QuickTime palette
const PACO_PAL: [u8; 768] = [
    0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xCC,
    0xFF, 0xFF, 0x99,
    0xFF, 0xFF, 0x66,
    0xFF, 0xFF, 0x33,
    0xFF, 0xFF, 0x00,
    0xFF, 0xCC, 0xFF,
    0xFF, 0xCC, 0xCC,
    0xFF, 0xCC, 0x99,
    0xFF, 0xCC, 0x66,
    0xFF, 0xCC, 0x33,
    0xFF, 0xCC, 0x00,
    0xFF, 0x99, 0xFF,
    0xFF, 0x99, 0xCC,
    0xFF, 0x99, 0x99,
    0xFF, 0x99, 0x66,
    0xFF, 0x99, 0x33,
    0xFF, 0x99, 0x00,
    0xFF, 0x66, 0xFF,
    0xFF, 0x66, 0xCC,
    0xFF, 0x66, 0x99,
    0xFF, 0x66, 0x66,
    0xFF, 0x66, 0x33,
    0xFF, 0x66, 0x00,
    0xFF, 0x33, 0xFF,
    0xFF, 0x33, 0xCC,
    0xFF, 0x33, 0x99,
    0xFF, 0x33, 0x66,
    0xFF, 0x33, 0x33,
    0xFF, 0x33, 0x00,
    0xFF, 0x00, 0xFF,
    0xFF, 0x00, 0xCC,
    0xFF, 0x00, 0x99,
    0xFF, 0x00, 0x66,
    0xFF, 0x00, 0x33,
    0xFF, 0x00, 0x00,
    0xCC, 0xFF, 0xFF,
    0xCC, 0xFF, 0xCC,
    0xCC, 0xFF, 0x99,
    0xCC, 0xFF, 0x66,
    0xCC, 0xFF, 0x33,
    0xCC, 0xFF, 0x00,
    0xCC, 0xCC, 0xFF,
    0xCC, 0xCC, 0xCC,
    0xCC, 0xCC, 0x99,
    0xCC, 0xCC, 0x66,
    0xCC, 0xCC, 0x33,
    0xCC, 0xCC, 0x00,
    0xCC, 0x99, 0xFF,
    0xCC, 0x99, 0xCC,
    0xCC, 0x99, 0x99,
    0xCC, 0x99, 0x66,
    0xCC, 0x99, 0x33,
    0xCC, 0x99, 0x00,
    0xCC, 0x66, 0xFF,
    0xCC, 0x66, 0xCC,
    0xCC, 0x66, 0x99,
    0xCC, 0x66, 0x66,
    0xCC, 0x66, 0x33,
    0xCC, 0x66, 0x00,
    0xCC, 0x33, 0xFF,
    0xCC, 0x33, 0xCC,
    0xCC, 0x33, 0x99,
    0xCC, 0x33, 0x66,
    0xCC, 0x33, 0x33,
    0xCC, 0x33, 0x00,
    0xCC, 0x00, 0xFF,
    0xCC, 0x00, 0xCC,
    0xCC, 0x00, 0x99,
    0xCC, 0x00, 0x66,
    0xCC, 0x00, 0x33,
    0xCC, 0x00, 0x00,
    0x99, 0xFF, 0xFF,
    0x99, 0xFF, 0xCC,
    0x99, 0xFF, 0x99,
    0x99, 0xFF, 0x66,
    0x99, 0xFF, 0x33,
    0x99, 0xFF, 0x00,
    0x99, 0xCC, 0xFF,
    0x99, 0xCC, 0xCC,
    0x99, 0xCC, 0x99,
    0x99, 0xCC, 0x66,
    0x99, 0xCC, 0x33,
    0x99, 0xCC, 0x00,
    0x99, 0x99, 0xFF,
    0x99, 0x99, 0xCC,
    0x99, 0x99, 0x99,
    0x99, 0x99, 0x66,
    0x99, 0x99, 0x33,
    0x99, 0x99, 0x00,
    0x99, 0x66, 0xFF,
    0x99, 0x66, 0xCC,
    0x99, 0x66, 0x99,
    0x99, 0x66, 0x66,
    0x99, 0x66, 0x33,
    0x99, 0x66, 0x00,
    0x99, 0x33, 0xFF,
    0x99, 0x33, 0xCC,
    0x99, 0x33, 0x99,
    0x99, 0x33, 0x66,
    0x99, 0x33, 0x33,
    0x99, 0x33, 0x00,
    0x99, 0x00, 0xFF,
    0x99, 0x00, 0xCC,
    0x99, 0x00, 0x99,
    0x99, 0x00, 0x66,
    0x99, 0x00, 0x33,
    0x99, 0x00, 0x00,
    0x66, 0xFF, 0xFF,
    0x66, 0xFF, 0xCC,
    0x66, 0xFF, 0x99,
    0x66, 0xFF, 0x66,
    0x66, 0xFF, 0x33,
    0x66, 0xFF, 0x00,
    0x66, 0xCC, 0xFF,
    0x66, 0xCC, 0xCC,
    0x66, 0xCC, 0x99,
    0x66, 0xCC, 0x66,
    0x66, 0xCC, 0x33,
    0x66, 0xCC, 0x00,
    0x66, 0x99, 0xFF,
    0x66, 0x99, 0xCC,
    0x66, 0x99, 0x99,
    0x66, 0x99, 0x66,
    0x66, 0x99, 0x33,
    0x66, 0x99, 0x00,
    0x66, 0x66, 0xFF,
    0x66, 0x66, 0xCC,
    0x66, 0x66, 0x99,
    0x66, 0x66, 0x66,
    0x66, 0x66, 0x33,
    0x66, 0x66, 0x00,
    0x66, 0x33, 0xFF,
    0x66, 0x33, 0xCC,
    0x66, 0x33, 0x99,
    0x66, 0x33, 0x66,
    0x66, 0x33, 0x33,
    0x66, 0x33, 0x00,
    0x66, 0x00, 0xFF,
    0x66, 0x00, 0xCC,
    0x66, 0x00, 0x99,
    0x66, 0x00, 0x66,
    0x66, 0x00, 0x33,
    0x66, 0x00, 0x00,
    0x33, 0xFF, 0xFF,
    0x33, 0xFF, 0xCC,
    0x33, 0xFF, 0x99,
    0x33, 0xFF, 0x66,
    0x33, 0xFF, 0x33,
    0x33, 0xFF, 0x00,
    0x33, 0xCC, 0xFF,
    0x33, 0xCC, 0xCC,
    0x33, 0xCC, 0x99,
    0x33, 0xCC, 0x66,
    0x33, 0xCC, 0x33,
    0x33, 0xCC, 0x00,
    0x33, 0x99, 0xFF,
    0x33, 0x99, 0xCC,
    0x33, 0x99, 0x99,
    0x33, 0x99, 0x66,
    0x33, 0x99, 0x33,
    0x33, 0x99, 0x00,
    0x33, 0x66, 0xFF,
    0x33, 0x66, 0xCC,
    0x33, 0x66, 0x99,
    0x33, 0x66, 0x66,
    0x33, 0x66, 0x33,
    0x33, 0x66, 0x00,
    0x33, 0x33, 0xFF,
    0x33, 0x33, 0xCC,
    0x33, 0x33, 0x99,
    0x33, 0x33, 0x66,
    0x33, 0x33, 0x33,
    0x33, 0x33, 0x00,
    0x33, 0x00, 0xFF,
    0x33, 0x00, 0xCC,
    0x33, 0x00, 0x99,
    0x33, 0x00, 0x66,
    0x33, 0x00, 0x33,
    0x33, 0x00, 0x00,
    0x00, 0xFF, 0xFF,
    0x00, 0xFF, 0xCC,
    0x00, 0xFF, 0x99,
    0x00, 0xFF, 0x66,
    0x00, 0xFF, 0x33,
    0x00, 0xFF, 0x00,
    0x00, 0xCC, 0xFF,
    0x00, 0xCC, 0xCC,
    0x00, 0xCC, 0x99,
    0x00, 0xCC, 0x66,
    0x00, 0xCC, 0x33,
    0x00, 0xCC, 0x00,
    0x00, 0x99, 0xFF,
    0x00, 0x99, 0xCC,
    0x00, 0x99, 0x99,
    0x00, 0x99, 0x66,
    0x00, 0x99, 0x33,
    0x00, 0x99, 0x00,
    0x00, 0x66, 0xFF,
    0x00, 0x66, 0xCC,
    0x00, 0x66, 0x99,
    0x00, 0x66, 0x66,
    0x00, 0x66, 0x33,
    0x00, 0x66, 0x00,
    0x00, 0x33, 0xFF,
    0x00, 0x33, 0xCC,
    0x00, 0x33, 0x99,
    0x00, 0x33, 0x66,
    0x00, 0x33, 0x33,
    0x00, 0x33, 0x00,
    0x00, 0x00, 0xFF,
    0x00, 0x00, 0xCC,
    0x00, 0x00, 0x99,
    0x00, 0x00, 0x66,
    0x00, 0x00, 0x33,
    0xEE, 0x00, 0x00,
    0xDD, 0x00, 0x00,
    0xBB, 0x00, 0x00,
    0xAA, 0x00, 0x00,
    0x88, 0x00, 0x00,
    0x77, 0x00, 0x00,
    0x55, 0x00, 0x00,
    0x44, 0x00, 0x00,
    0x22, 0x00, 0x00,
    0x11, 0x00, 0x00,
    0x00, 0xEE, 0x00,
    0x00, 0xDD, 0x00,
    0x00, 0xBB, 0x00,
    0x00, 0xAA, 0x00,
    0x00, 0x88, 0x00,
    0x00, 0x77, 0x00,
    0x00, 0x55, 0x00,
    0x00, 0x44, 0x00,
    0x00, 0x22, 0x00,
    0x00, 0x11, 0x00,
    0x00, 0x00, 0xEE,
    0x00, 0x00, 0xDD,
    0x00, 0x00, 0xBB,
    0x00, 0x00, 0xAA,
    0x00, 0x00, 0x88,
    0x00, 0x00, 0x77,
    0x00, 0x00, 0x55,
    0x00, 0x00, 0x44,
    0x00, 0x00, 0x22,
    0x00, 0x00, 0x11,
    0xEE, 0xEE, 0xEE,
    0xDD, 0xDD, 0xDD,
    0xBB, 0xBB, 0xBB,
    0xAA, 0xAA, 0xAA,
    0x88, 0x88, 0x88,
    0x77, 0x77, 0x77,
    0x55, 0x55, 0x55,
    0x44, 0x44, 0x44,
    0x22, 0x22, 0x22,
    0x11, 0x11, 0x11,
    0x00, 0x00, 0x00,
];
