use std::fs::File;
use std::collections::VecDeque;
use crate::io::byteio::*;
use super::super::*;

struct PHFrame {
    frame:      Vec<u8>,
    pal:        [u8; 768],
    unp_buf:    Vec<u8>,
    vdata:      Vec<u8>,
    width:      usize,
    height:     usize,
}

impl PHFrame {
    fn unpack(&mut self, intra: bool) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.vdata);

        br.read_skip(2)?;
        let height = br.read_u16le()? as usize;
        let width  = br.read_u16le()? as usize;
        validate!(width <= self.width && height <= self.height);
        br.read_skip(2)?;
        let tdata_len = (((width + 1) / 2 + 7) / 8) * ((height + 1) / 2);

        if !intra {
            br.read_skip(2)?; // always 514?
            br.read_skip(tdata_len)?;
        }

        self.unp_buf.clear();
        let mut run = 0;
        let mut hinib = false;
        let mut clrs = [0; 2];
        while br.left() >= 2 {
            br.read_buf(&mut clrs)?;
            for &clr in clrs.iter() {
                if (clr & 0x80) != 0 {
                    self.unp_buf.push(clr);
                } else {
                    let len = if !hinib {
                            run = br.read_byte()?;
                            run & 0xF
                        } else {
                            run >> 4
                        };
                    if len == 0 {
                        break;
                    }
                    hinib = !hinib;
                    for _ in 0..=len {
                        self.unp_buf.push(clr | 0x80);
                    }
                }
            }
        }

        if intra {
            for (dline, sline) in self.frame.chunks_exact_mut(self.width)
                    .zip(self.unp_buf.chunks_exact(width)) {
                dline[..width].copy_from_slice(sline);
            }
        } else {
            let tdata = &self.vdata[10..];
            let mut br = MemoryReader::new_read(&self.unp_buf);

            for (dlines, stats) in self.frame.chunks_exact_mut(self.width * 2)
                    .zip(tdata.chunks_exact(((width + 1) / 2 + 7) / 8)) {
                for x in (0..width).step_by(2) {
                    if (stats[(x / 2) / 8] & (1 << ((x / 2) & 7))) != 0 {
                        dlines[x] = br.read_byte()?;
                        dlines[x + 1] = br.read_byte()?;
                        dlines[x + self.width] = br.read_byte()?;
                        dlines[x + self.width + 1] = br.read_byte()?;
                    }
                }
            }
        }

        Ok(())
    }
    fn get(&self) -> Frame {
        let pal = self.pal;
        Frame::VideoPal(self.frame.clone(), pal)
    }
}

type FrameQueue = VecDeque<(u64, Frame)>;

struct PHDecoder {
    fr:         FileReader<File>,
    frame:      PHFrame,
    block_end:  u64,
    nchunks:    usize,
    arate:      u32,
    channels:   u8,
    vpts_end:   u64,
    vpts:       u64,
    apts:       u64,
    vqueue:     FrameQueue,
    aqueue:     FrameQueue,
}

impl PHDecoder {
    fn get_frame(aqueue: &mut FrameQueue, vqueue: &mut FrameQueue, has_audio: bool, flush: bool) -> Option<(usize, Frame)> {
        if has_audio && !flush {
            if !aqueue.is_empty() && !vqueue.is_empty() {
                if aqueue[0].0 < vqueue[0].0 {
                    let (_, frm) = aqueue.pop_front().unwrap();
                    return Some((1, frm));
                } else {
                    let (_, frm) = vqueue.pop_front().unwrap();
                    return Some((0, frm));
                }
            }
        } else {
            if let Some((_pts, frm)) = vqueue.pop_front() {
                return Some((0, frm));
            }
            if let Some((_pts, frm)) = aqueue.pop_front() {
                return Some((1, frm));
            }
        }
        None
    }
}

impl InputSource for PHDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.frame.width,
                    height: self.frame.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 30,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    sample_type: AudioSample::U8,
                    channels:    self.channels,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        loop {
            if let Some(ret) = Self::get_frame(&mut self.aqueue, &mut self.vqueue, self.arate != 0, false) {
                return Ok(ret);
            }
            if self.nchunks > 0 {
                while (br.tell() & 3) != 0 {
                    br.read_skip(1)?;
                }
                let chunk_id = br.read_u16le()?;
                let chunk_size = br.read_u16le()? as usize;
                self.nchunks -= 1;

//println!("chunk {:04X} size {:X} @ {:X}", chunk_id, chunk_size, br.tell() - 4);
                match chunk_id {
                    0x1010 => {
                        let _frameno = br.read_u32le()?;
                        let delay = br.read_u32le()?; // in 1/60s
                        let _unk = br.read_u32le()?;
                        self.vpts_end += u64::from(delay);
                    },
                    0xA016 | 0xA024 | 0xA032 | 0xA044 | 0xA0A8 => {
                        validate!(self.channels == 1);
                        let mut adata = vec![0; chunk_size];
                        br.read_buf(&mut adata)?;
                        self.aqueue.push_back((self.apts * 60, Frame::AudioU8(adata)));
                        self.apts += chunk_size as u64;
                    },
                    0xA116 | 0xA124 | 0xA132 | 0xA144 | 0xA1A8 => {
                        validate!(self.channels == 2);
                        validate!((chunk_size & 1) == 0);
                        let mut adata = vec![0; chunk_size];
                        br.read_buf(&mut adata)?;
                        self.aqueue.push_back((self.apts * 60, Frame::AudioU8(adata)));
                        self.apts += chunk_size as u64 / 2;
                    },
                    0xB0B2 => {
                        validate!((5..=780).contains(&chunk_size) && (chunk_size - 2) % 3 == 0);
                        let start = usize::from(br.read_byte()?);
                        let nentries = usize::from(br.read_byte()?);
                        validate!(nentries * 3 + 2 <= chunk_size);
                        validate!(128 + start + nentries <= 256);
                        br.read_vga_pal_some(&mut self.frame.pal[(128 + start) * 3..][..nentries * 3])?;
                        br.read_skip(chunk_size - nentries * 3 - 2)?;
                    },
                    0xC0C1 => return Err(DecoderError::NotImplemented),
                    0xC0C2 | 0xC0C3 => {
                        self.frame.vdata.resize(chunk_size, 0);
                        br.read_buf(&mut self.frame.vdata)?;
                        self.frame.unpack(chunk_id == 0xC0C2).map_err(|_| DecoderError::InvalidData)?;

                        let mul = if self.arate > 0 { u64::from(self.arate) } else { 1 };
                        while self.vpts < self.vpts_end {
                            self.vqueue.push_back((self.vpts * mul, self.frame.get()));
                            self.vpts += 2;
                        }
                    },
                    _ => br.read_skip(chunk_size)?,
                }
            } else {
                validate!(br.tell() <= self.block_end);
                br.seek(SeekFrom::Start(self.block_end))?;
                let ret = br.read_byte();
                if ret.is_err() {
                    if let Some(ret) = Self::get_frame(&mut self.aqueue, &mut self.vqueue, self.arate != 0, true) {
                        return Ok(ret);
                    } else {
                        return Err(DecoderError::EOF);
                    }
                }
                let tag0 = ret.unwrap();
                let tag1 = br.read_byte()?;
                let _psize = br.read_u32le()?;
                let blk_size = br.read_u32le()?;
                let _nsize = br.read_u16le()?;
                let _smth = br.read_u16le()?;
                let nchunks = br.read_u16le()? as usize;
//println!("PH size {:X} / {:X} / {:X} chunks {} smth {} end {:X}", _psize, blk_size, _nsize, nchunks, _smth, br.tell() + u64::from(blk_size));
                validate!(tag0 == b'P' && tag1 == b'H');
                validate!(blk_size > 0 && nchunks > 0);
                self.block_end = br.tell() + u64::from(blk_size);
                self.nchunks = nchunks;
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(file);

    // scan whole file for audio/video chunks
    let mut width = 0;
    let mut height = 0;
    let mut arate = 0;
    let mut channels = 0;
    let mut nblocks = 0;
    while width == 0 || channels == 0 || nblocks < 5 {
        let mut hdr = [0; 2];
        if br.read_buf(&mut hdr).is_err() {
            break;
        }
        validate!(&hdr == b"PH");
        let _psize = br.read_u32le()?;
        let size = br.read_u32le()?;
        let _nsize = br.read_u16le()?;
        let _smth = br.read_u16le()?;
        let nchunks = br.read_u16le()? as usize;
        validate!(size > 0);
        validate!(nchunks > 0);

        let end = br.tell() + u64::from(size);
        for _ in 0..nchunks {
            let id = br.read_u16le()?;
            let csize = br.read_u16le()? as usize;
            if br.tell() + (csize as u64) > end {
                break;
            }
            match id {
                0xC0C2 => {
                    validate!(csize > 8);
                    br.read_skip(2)?;
                    let frm_h = br.read_u16le()? as usize;
                    let frm_w = br.read_u16le()? as usize;
                    validate!(frm_w > 0 && frm_h > 0);
                    height = height.max(frm_h);
                    width = width.max(frm_w);
                    br.read_skip(csize - 6)?;
                },
                0xA016 | 0xA024 | 0xA032 | 0xA044 | 0xA0A8 |
                0xA116 | 0xA124 | 0xA132 | 0xA144 | 0xA1A8 => {
                    let cur_ch = if (id & 0x0F00) != 0 { 2 } else { 1 };
                    let cur_rate = match id & 0xFF {
                            0x16 => 16000,
                            0x24 => 22050,
                            0x32 => 32000,
                            0x44 => 44100,
                            0xA8 => 8000,
                            _ => return Err(DecoderError::NotImplemented),
                        };
                    if arate == 0 {
                        arate = cur_rate;
                        channels = cur_ch;
                    } else {
                        validate!(arate == cur_rate);
                        validate!(channels == cur_ch);
                    }
                    br.read_skip(csize)?;
                },
                _ => br.read_skip(csize)?,
            }
            while (br.tell() & 3) != 0 {
                br.read_skip(1)?;
            }
        }
        validate!(br.tell() <= end);
        br.seek(SeekFrom::Start(end))?;
        nblocks += 1;
    }
    validate!(width > 0 && height > 0);

    br.seek(SeekFrom::Start(0))?;

    Ok(Box::new(PHDecoder {
        fr: br,
        arate, channels,
        frame:      PHFrame {
            width, height,
            frame:      vec![0; width * height],
            unp_buf:    Vec::with_capacity(width * height),
            vdata:      Vec::with_capacity(65536),
            pal:        [0; 768],
        },
        block_end:  0,
        nchunks:    0,
        vpts:       0,
        vpts_end:   0,
        apts:       0,
        vqueue:     VecDeque::new(),
        aqueue:     VecDeque::new(),
    }))
}
