use std::fs::File;
use std::io::BufReader;
use crate::input::util::lzs::lzs_unpack;
use crate::io::byteio::*;
use super::super::*;

const AFRAME_HDR_SIZE: usize = 16;

const FRAME_HEADER: usize = 24;

struct RobotDecoder {
    fr:         FileReader<BufReader<File>>,
    data:       Vec<u8>,
    adata:      Vec<u8>,
    frame:      Vec<u8>,
    cell_buf:   Vec<u8>,
    pal:        [u8; 768],
    fps:        u16,
    width:      usize,
    height:     usize,
    nframes:    usize,
    vframe_len: Vec<u32>,
    pkt_len:    Vec<u32>,
    pkt_no:     usize,
    version:    u16,
    has_audio:  bool,
    afrm_size:  usize,
    audio:      Vec<i16>,
}

fn unpack_cell(br: &mut dyn ByteIO, cell_size: usize, nchunks: usize, dst: &mut Vec<u8>, limit: usize) -> DecoderResult<()> {
    let mut data_left = cell_size;
    dst.clear();
    dst.reserve(limit);
    for _ in 0..nchunks {
        validate!(data_left >= 10);
        let csize       = br.read_u32le()? as usize;
        validate!(csize <= data_left);
        let rsize       = br.read_u32le()? as usize;
        validate!(rsize + dst.len() <= limit);
        let method      = br.read_u16le()?;

        data_left -= 10;

        let cur_size = dst.len();
        dst.resize(cur_size + rsize, 0);
        match method {
            0 => { lzs_unpack(br, csize, &mut dst[cur_size..])?; },
            2 => {
                validate!(rsize == csize);
                br.read_buf(&mut dst[cur_size..])?;
            },
            _ => return Err(DecoderError::NotImplemented),
        }
        data_left -= csize;
    }
    Ok(())
}

fn blit(dst: &mut [u8], dstride: usize, src: &[u8], sstride: usize) {
    for (dline, sline) in dst.chunks_mut(dstride).zip(src.chunks(sstride)) {
        dline[..sstride].copy_from_slice(sline);
    }
}

fn blit_scaled(dst: &mut [u8], dstride: usize, src: &[u8], sstride: usize, scale: u8) {
    let mut slines = src.chunks(sstride);
    let mut acc = 0;

    let mut cur_line = slines.next().unwrap();

    for dline in dst.chunks_mut(dstride) {
        dline[..sstride].copy_from_slice(cur_line);
        acc += scale;
        if acc >= 100 {
            acc -= 100;
            if let Some(line) = slines.next() {
                cur_line = line;
            } else {
                break;
            }
        }
    }
}

fn pred16(pred: i32, val: u8) -> i32 {
    if (val & 0x80) != 0 {
        pred - i32::from(SOL_AUD_STEPS16[(val & 0x7F) as usize])
    } else {
        pred + i32::from(SOL_AUD_STEPS16[(val & 0x7F) as usize])
    }
}

impl RobotDecoder {
    fn decode_video(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = MemoryReader::new_read(&self.data);

        let ncells              = br.read_u16le()? as usize;
        validate!(ncells > 0 && ncells <= 10);
        for el in self.frame.iter_mut() { *el = 0xFF; }
        for _ in 0..ncells {
                                  br.read_byte()?;
            let scale           = br.read_byte()?;
            let width           = br.read_u16le()? as usize;
            let height          = br.read_u16le()? as usize;
                                  br.read_skip(4)?;
            let xoff            = br.read_u16le()? as usize;
            let yoff            = br.read_u16le()? as usize;
            validate!(xoff + width <= self.width && yoff + height <= self.height);
            let mut cell_size   = br.read_u16le()? as usize;
            // hack
            if self.version == 6 && ncells == 1 && (self.data.len() - 18 >= 0x10000) {
                cell_size += (self.data.len() - 18) & !0xFFFF;
            }
            if self.version > 4 {
                let nchunks     = br.read_u16le()? as usize;
                validate!(nchunks > 0);
                                  br.read_skip(4)?;

                unpack_cell(&mut br, cell_size, nchunks, &mut self.cell_buf, width * height)?;
            } else {
                                  br.read_skip(6)?;
                self.cell_buf.resize(width * height, 0);
                lzs_unpack(&mut br, cell_size, &mut self.cell_buf)?;
            }
            if scale == 100 {
                blit(&mut self.frame[xoff + yoff * self.width..], self.width, &self.cell_buf, width);
            } else {
                blit_scaled(&mut self.frame[xoff + yoff * self.width..], self.width, &self.cell_buf, width, scale);
            }
        }

        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

impl InputSource for RobotDecoder {
    fn get_num_streams(&self) -> usize { if self.has_audio { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                 }),
            1 if self.has_audio => StreamInfo::Audio(AudioInfo{
                    sample_rate: 11025,
                    channels:    1,
                    sample_type: AudioSample::S16,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        if self.has_audio && self.audio.len() >= self.afrm_size {
            let mut frm = vec![0; self.afrm_size];
            frm.copy_from_slice(&self.audio[..self.afrm_size]);
            self.audio.drain(..self.afrm_size);
            return Ok((1, Frame::AudioS16(frm)));
        }

        if self.pkt_no >= self.nframes {
            return Err(DecoderError::EOF);
        }

        self.data.resize(self.vframe_len[self.pkt_no] as usize, 0);
                                  br.read_buf(&mut self.data)?;

        let asize = (self.pkt_len[self.pkt_no] - self.vframe_len[self.pkt_no]) as usize;
        if self.has_audio && asize > 0 {
            validate!(asize >= AFRAME_HDR_SIZE);
            let _ref_apts           = u64::from(br.read_u32le()?);
            let ref_asize           = br.read_u32le()? as usize;
            validate!(asize == ref_asize + 8);

            self.adata.resize(ref_asize, 0);
                                      br.read_buf(&mut self.adata)?;

            let (prime, data) = self.adata.split_at(8);
            let mut pred = 0;
            for &b in prime.iter() {
                pred = pred16(pred, b);
            }
            for &b in data.iter().take(data.len() / 2) {
                pred = pred16(pred, b);
                self.audio.push(pred as i16);
            }
        } else {
                                  br.read_skip(asize)?;
        }
        self.pkt_no += 1;

        self.decode_video().map_err(|_| DecoderError::InvalidData)
    }
}

fn unpack_pal_data(pal: &mut [u8; 768], src: &[u8]) -> DecoderResult<()> {
    validate!(src.len() > 37);
    let pal_start = read_u16le(&src[25..])? as usize;
    let pal_len   = read_u16le(&src[29..])? as usize;
    validate!(pal_len > 0 && pal_start + pal_len <= 256);
    match src[32] {
        0 => {
            let dpal = pal[pal_start * 3..].chunks_exact_mut(3);
            for (dst, quad) in dpal.zip(src[37..].chunks_exact(4)) {
                dst.copy_from_slice(&quad[1..]);
            }
        },
        1 => pal[pal_start * 3..][..pal_len * 3].copy_from_slice(&src[37..][..pal_len * 3]),
        _ => return Err(DecoderError::NotImplemented),
    }

    Ok(())
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let mut hdr = [0; 60];
                                      br.read_buf(&mut hdr)?;
    validate!(hdr[0] == 0x16 || hdr[0] == 0x3D);
    validate!(&hdr[2..6] == b"SOL\0");
    let version = read_u16le(&hdr[6..])?;
    let afrm_size = read_u16le(&hdr[8..])? as usize;
    validate!((4..=6).contains(&version));

    let nframes = read_u16le(&hdr[14..])? as usize;
    validate!(nframes > 0);
    let pal_size = read_u16le(&hdr[16..])? as usize;
    let audio_pre_size = read_u16le(&hdr[18..])? as usize;

    let mut width = read_u16le(&hdr[20..])? as usize;
    if width == 0 { width = 640; }
    let mut height = read_u16le(&hdr[22..])? as usize;
    if height == 0 { height = 480; }
    let has_pal = hdr[24] != 0;
    let has_audio = hdr[25] != 0 && afrm_size > 0;
    let fps = read_u16le(&hdr[28..])?;

    let mut audio = Vec::new();
    if !has_audio || audio_pre_size == 0 {
                                      br.read_skip(audio_pre_size)?;
    } else {
        let end_pos = br.tell() + (audio_pre_size as u64);
        validate!(audio_pre_size >= 12);
        validate!(afrm_size > AFRAME_HDR_SIZE);
        let pre_size                = br.read_u32le()? as usize;
        validate!(pre_size <= audio_pre_size - 14);
        let method                  = br.read_u16le()?;
        validate!(method == 0);
        let size1                   = br.read_u32le()? as usize;
        let size2                   = br.read_u32le()? as usize;
        validate!(size1 + size2 <= pre_size);
        let to_skip = (afrm_size - AFRAME_HDR_SIZE) / 2;
        let mut initial = Vec::new();
        if size1 + size2 > to_skip {
            initial.resize(size1 + size2 - to_skip, 0);
                                      br.read_buf(&mut initial)?;
        }
                                      br.seek(SeekFrom::Start(end_pos))?;
        // Looks like audio channels are decoded semi-independently
        // so the second channel may have more data initially
        // and follow-up audio blocks have channel data shifted by that amount.
        // Thus I don't bother to decode the second channel at all.
        let mut pred = 0;
        for &b in initial[..size1].iter() {
            pred = pred16(pred, b);
            audio.push(pred as i16);
        }
    }

    let mut pal = [0; 768];
    if pal_size > 0 {
        let mut pal_data = vec![0; pal_size];
                                      br.read_buf(&mut pal_data)?;
        if has_pal {
            unpack_pal_data(&mut pal, &pal_data).map_err(|_| DecoderError::InvalidData)?;
        }
    }
    if pal_size == 0 || !has_pal {
        for (i, clr) in pal.chunks_exact_mut(3).enumerate() {
            clr[0] = i as u8;
            clr[1] = i as u8;
            clr[2] = i as u8;
        }
    }

    let mut vframe_len = Vec::with_capacity(nframes);
    if version < 6 {
        for _ in 0..nframes {
            let size                = br.read_u16le()?;
            vframe_len.push(u32::from(size));
        }
    } else {
        for _ in 0..nframes {
            let size                = br.read_u32le()?;
            vframe_len.push(size);
        }
    }

    let mut pkt_len = Vec::with_capacity(nframes);
    if version < 6 {
        for _ in 0..nframes {
            let size                = br.read_u16le()?;
            pkt_len.push(u32::from(size));
        }
    } else {
        for _ in 0..nframes {
            let size                = br.read_u32le()?;
            pkt_len.push(size);
        }
    }
                                      br.read_skip(256 * 4)?; // cues
                                      br.read_skip(256 * 2)?; // smth
    let pos = (br.tell() & 0x7FF) as usize;
    if pos != 0 {
                                      br.read_skip(0x800 - pos)?;
    }

    Ok(Box::new(RobotDecoder {
        fr: br,
        frame: vec![0; width * height],
        cell_buf: Vec::new(),
        adata: Vec::new(),
        data: Vec::new(),
        width, height, fps, pal,
        version, pkt_len, vframe_len, nframes,
        pkt_no: 0,
        has_audio, afrm_size, audio,
    }))
}

const SOL_AUD_STEPS16: [i16; 128] = [
     0x00,   0x08,   0x10,   0x20,   0x30,   0x40,   0x50,   0x60,
     0x70,   0x80,   0x90,   0xA0,   0xB0,   0xC0,   0xD0,   0xE0,
     0xF0,  0x100,  0x110,  0x120,  0x130,  0x140,  0x150,  0x160,
    0x170,  0x180,  0x190,  0x1A0,  0x1B0,  0x1C0,  0x1D0,  0x1E0,
    0x1F0,  0x200,  0x208,  0x210,  0x218,  0x220,  0x228,  0x230,
    0x238,  0x240,  0x248,  0x250,  0x258,  0x260,  0x268,  0x270,
    0x278,  0x280,  0x288,  0x290,  0x298,  0x2A0,  0x2A8,  0x2B0,
    0x2B8,  0x2C0,  0x2C8,  0x2D0,  0x2D8,  0x2E0,  0x2E8,  0x2F0,
    0x2F8,  0x300,  0x308,  0x310,  0x318,  0x320,  0x328,  0x330,
    0x338,  0x340,  0x348,  0x350,  0x358,  0x360,  0x368,  0x370,
    0x378,  0x380,  0x388,  0x390,  0x398,  0x3A0,  0x3A8,  0x3B0,
    0x3B8,  0x3C0,  0x3C8,  0x3D0,  0x3D8,  0x3E0,  0x3E8,  0x3F0,
    0x3F8,  0x400,  0x440,  0x480,  0x4C0,  0x500,  0x540,  0x580,
    0x5C0,  0x600,  0x640,  0x680,  0x6C0,  0x700,  0x740,  0x780,
    0x7C0,  0x800,  0x900,  0xA00,  0xB00,  0xC00,  0xD00,  0xE00,
    0xF00, 0x1000, 0x1400, 0x1800, 0x1C00, 0x2000, 0x3000, 0x4000
];
