use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;
use crate::input::util::jpeg::*;
use crate::input::util::yuv::*;

#[derive(Clone,Copy,Debug,Default,PartialEq)]
enum State {
    #[default]
    Unknown,
    True,
    False
}

const OLD_RANGE: std::ops::RangeInclusive<u16> = 0x60..=0xA0;

struct RoQDecoder {
    fr:         FileReader<BufReader<File>>,
    width:      usize,
    height:     usize,
    fps:        u16,
    old:        State,
    data:       Vec<u8>,
    jpeg:       JPEGDecoder,
    yuv2rgb:    YUV2RGB,
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    alpha:      bool,
    mc_2x:      bool,
    cb:         [[u8; 3 * 4]; 256],
    num_2x2:    usize,
    cb4:        [[u8; 4]; 256],
    num4:       usize,
    frameno:    usize,
    abuf:       Vec<i16>,
    audio:      bool,
    ablk_len:   usize,
    channels:   u8,
}

struct MCParams {
    width:      usize,
    height:     usize,
    mean_dx:    isize,
    mean_dy:    isize,
    mc_2x:      bool,
    mc_halve:   bool,
}

impl RoQDecoder {
    fn swap_frames(&mut self) {
        std::mem::swap(&mut self.frame, &mut self.pframe);
    }
    fn do_mc(dst: &mut [u8], pframe: &[u8], xpos: usize, ypos: usize, mv: u8, mc_params: &MCParams, bsize: usize) -> DecoderResult<()> {
        let stride = mc_params.width * 3;
        let mut dx = 8 - isize::from(mv >> 4) - mc_params.mean_dx;
        let mut dy = 8 - isize::from(mv & 0xF) - mc_params.mean_dy;
        if mc_params.mc_2x {
            dx *= 2;
            if !mc_params.mc_halve {
                dy *= 2;
            }
        }
        let src_x = (xpos as isize) + dx;
        let src_y = (ypos as isize) + dy;
        validate!(src_x >= 0 && src_y >= 0);
        let src_x = src_x as usize;
        let src_y = src_y as usize;
        validate!(src_x + bsize <= mc_params.width && src_y + bsize <= mc_params.height);
        for (dline, sline) in dst.chunks_mut(stride).take(bsize)
                .zip(pframe[src_x * 3 + src_y * stride..].chunks(stride)) {
            dline[..bsize * 3].copy_from_slice(&sline[..bsize * 3]);
        }
        Ok(())
    }
    fn paint_4x4(dst: &mut [u8], stride: usize, idx: &[u8; 4], cb: &[[u8; 4 * 3]; 256], num_cb: usize) -> DecoderResult<()> {
        for (blk_no, &cb_idx) in idx.iter().enumerate() {
            let cb_idx = usize::from(cb_idx);
            validate!(cb_idx < num_cb);
            let src = &cb[cb_idx];
            for (dline, sline) in dst[(blk_no & 1) * 2 * 3 + (blk_no >> 1) * 2 * stride..]
                    .chunks_mut(stride).zip(src.chunks_exact(2 * 3)) {
                dline[..2 * 3].copy_from_slice(sline);
            }
        }
        Ok(())
    }
    fn decode_vq(&mut self, mean_mv: u16) -> DecoderResult<()> {
        const BTYPE_SKIP: u16 = 0;
        const BTYPE_MOTION: u16 = 1;
        const BTYPE_VQ: u16 = 2;
        const BTYPE_SPLIT: u16 = 3;
        let mc_parms = MCParams {
                width:      self.width,
                height:     self.height,
                mean_dx:    isize::from((mean_mv >> 8) as i8),
                mean_dy:    isize::from((mean_mv & 0xFF) as i8),
                mc_2x:      self.mc_2x,
                mc_halve:   self.mc_2x && self.width == 640 && self.height == 160, // TLC hack
            };

        let mut br = MemoryReader::new_read(&self.data);

        let stride = self.width * 3;

        let mut types = 0;
        let mut ntypes = 0;
        for (yrow, strip) in self.frame.chunks_exact_mut(stride * 16).enumerate() {
            for x in (0..self.width).step_by(16) {
                for blk_no in 0..4 {
                    let dst_x = (blk_no & 1) * 8 + x;
                    let dst_y = (blk_no >> 1) * 8;

                    if ntypes == 0 {
                        types = br.read_u16le()?;
                        ntypes = 8;
                    }
                    let btype = types >> 14;
                    types <<= 2;
                    ntypes -= 1;
                    match btype {
                        BTYPE_SKIP => {},
                        BTYPE_MOTION => {
                            let mv = br.read_byte()?;
                            Self::do_mc(&mut strip[dst_x * 3 + dst_y * stride..],
                                        &self.pframe, dst_x, dst_y + yrow * 16, mv, &mc_parms, 8)?;
                        },
                        BTYPE_VQ => {
                            let superidx = usize::from(br.read_byte()?);
                            validate!(superidx < self.num4);
                            for (sblk_no, &idx) in self.cb4[superidx].iter().enumerate() {
                                let idx = usize::from(idx);
                                validate!(idx < self.num_2x2);
                                let cb = &self.cb[idx];
                                for (sy, dline) in strip.chunks_exact_mut(stride)
                                        .skip((blk_no >> 1) * 8 + (sblk_no >> 1) * 4).take(4)
                                        .enumerate() {
                                    for (sx, dst) in dline[(x + (blk_no & 1) * 8 + (sblk_no & 1) * 4) * 3..][..4 * 3]
                                            .chunks_exact_mut(3).enumerate() {
                                        dst.copy_from_slice(&cb[((sx / 2) + (sy / 2) * 2) * 3..][..3]);
                                    }
                                }
                            }
                        },
                        BTYPE_SPLIT => {
                            for sblk_no in 0..4 {
                                let sdst_x = dst_x + (sblk_no & 1) * 4;
                                let sdst_y = dst_y + (sblk_no >> 1) * 4;

                                if ntypes == 0 {
                                    types = br.read_u16le()?;
                                    ntypes = 8;
                                }
                                let btype = types >> 14;
                                types <<= 2;
                                ntypes -= 1;
                                match btype {
                                    BTYPE_SKIP => {},
                                    BTYPE_MOTION => {
                                        let mv = br.read_byte()?;
                                        Self::do_mc(&mut strip[sdst_x * 3 + sdst_y * stride..],
                                                    &self.pframe, sdst_x, sdst_y + yrow * 16,
                                                    mv, &mc_parms, 4)?;
                                    },
                                    BTYPE_VQ => {
                                        let sidx = usize::from(br.read_byte()?);
                                        validate!(sidx < self.num4);
                                        Self::paint_4x4(&mut strip[sdst_x * 3 + sdst_y * stride..],
                                                stride, &self.cb4[sidx], &self.cb, self.num_2x2)?;
                                    },
                                    BTYPE_SPLIT => {
                                        let mut idx4 = [0; 4];
                                        br.read_buf(&mut idx4)?;
                                        Self::paint_4x4(&mut strip[sdst_x * 3 + sdst_y * stride..],
                                                stride, &idx4, &self.cb, self.num_2x2)?;
                                    },
                                    _ => unreachable!(),
                                }
                            }
                        },
                        _ => unreachable!()
                    }
                }
            }
        }
        Ok(())
    }
    fn decode_mono(dst: &mut Vec<i16>, src: &[u8], arg: u16) {
        let mut pred = arg as i16;
        for &b in src.iter() {
            let mant = i16::from(b & 0x7F);
            let sign = (b & 0x80) != 0;
            if !sign {
                pred += mant * mant;
            } else {
                pred -= mant * mant;
            }
            dst.push(pred);
        }
    }
    fn decode_stereo(dst: &mut Vec<i16>, src: &[u8], arg: u16) {
        let mut lpred = (arg & 0xFF00) as i16;
        let mut rpred = (arg << 8) as i16;
        for pair in src.chunks_exact(2) {
            let mant = i16::from(pair[0] & 0x7F);
            let sign = (pair[0] & 0x80) != 0;
            if !sign {
                lpred = lpred.wrapping_add(mant * mant);
            } else {
                lpred = lpred.wrapping_sub(mant * mant);
            }
            dst.push(lpred);
            let mant = i16::from(pair[1] & 0x7F);
            let sign = (pair[1] & 0x80) != 0;
            if !sign {
                rpred = rpred.wrapping_add(mant * mant);
            } else {
                rpred = rpred.wrapping_sub(mant * mant);
            }
            dst.push(rpred);
        }
    }
}

impl InputSource for RoQDecoder {
    fn get_num_streams(&self) -> usize {
        (if self.width > 0 { 1 } else { 0 }) + (if self.channels > 0 { 1 } else { 0 })
    }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        let has_video = self.width > 0;
        let has_audio = self.channels > 0;
        match (stream_no, has_video, has_audio) {
            (0, true, _) => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    24,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                }),
            (0, false, true) | (1, true, true) => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    sample_type: AudioSample::S16,
                    channels:    self.channels,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        loop {
            if self.audio && self.abuf.len() >= self.ablk_len {
                let has_video = self.width > 0;
                let mut ret = vec![0; self.ablk_len];
                ret.copy_from_slice(&self.abuf[..self.ablk_len]);
                self.abuf.drain(..self.ablk_len);
                self.audio = !has_video;
                return Ok((if has_video { 1 } else { 0 }, Frame::AudioS16(ret)));
            }
            let tag = self.fr.read_u16le().map_err(|_| DecoderError::EOF)?;
            let size = self.fr.read_u32le()? as usize;
            let arg = self.fr.read_u16le()?;
            match tag {
                0x1084 => return Err(DecoderError::EOF),
                0x1001 => { // configuration - already processed
                    self.fr.read_skip(size)?;
                },
                0x1000 => { // quad
                    return Err(DecoderError::NotImplemented);
                },
                0x1002 => { // quad cb
                    let mut num_2x2 = usize::from(arg >> 8);
                    if num_2x2 == 0 {
                        num_2x2 = 256;
                    }
                    let mut num4 = usize::from(arg & 0xFF);
                    if num4 == 0 && size > num_2x2 * 6 {
                        num4 = 256;
                    }

                    let mut elen = if self.alpha { 10 } else { 6 };
                    // sometimes codebooks with alpha occur in the middle of normal files
                    if elen == 6 && size == num_2x2 * 10 + num4 * 4 {
                        elen = 10;
                    }
                    validate!(size == elen * num_2x2 + 4 * num4);
                    let mut entry = [0; 10];
                    for el in self.cb[..num_2x2].iter_mut() {
                        self.fr.read_buf(&mut entry[..elen])?;
                        if elen == 6 {
                            let u = entry[4];
                            let v = entry[5];
                            for (pix, &y) in el.chunks_exact_mut(3).zip(entry[..4].iter()) {
                                self.yuv2rgb.convert(y, u, v, pix);
                            }
                        } else {
                            let u = entry[8];
                            let v = entry[9];
                            for (pix, ya) in el.chunks_exact_mut(3).zip(entry[..8].chunks_exact(2)) {
                                self.yuv2rgb.convert(ya[0], u, v, pix);
                            }
                        }
                    }
                    for el in self.cb4[..num4].iter_mut() {
                        self.fr.read_buf(el)?;
                    }
                    self.num_2x2 = num_2x2;
                    self.num4 = num4;
                },
                0x1011 => { // quad VQ
                    self.data.resize(size, 0);
                    self.fr.read_buf(&mut self.data)?;
                    self.swap_frames();
                    self.decode_vq(arg).map_err(|_| DecoderError::InvalidData)?;
                    if self.frameno == 0 {
                        self.pframe.copy_from_slice(&self.frame);
                    }
                    self.frameno += 1;
                    self.audio = true;
                    return Ok((0, Frame::VideoRGB24(self.frame.clone())));
                },
                0x1012 => { // JPEG
                    self.data.resize(size, 0);
                    self.fr.read_buf(&mut self.data)?;
                    self.swap_frames();
                    let jfrm = self.jpeg.decode(&self.data).map_err(|_| DecoderError::InvalidData)?;
                    validate!(jfrm.width == self.width && jfrm.height == self.height);
                    self.frame = jfrm.to_rgb(&self.yuv2rgb);
                    if self.frameno == 0 {
                        self.pframe.copy_from_slice(&self.frame);
                    }
                    self.frameno += 1;
                    self.audio = true;
                    return Ok((0, Frame::VideoRGB24(self.frame.clone())));
                },
                0x1013 => { // skip
                    validate!(size == 0);
                    self.frameno += 1;
                    self.audio = true;
                    return Ok((0, Frame::VideoRGB24(self.frame.clone())));
                },
                0x1020 if self.channels > 0 => { // mono
                    self.data.resize(size, 0);
                    self.fr.read_buf(&mut self.data)?;
                    // audio format autodetect hack
                    if self.old == State::Unknown {
                        let p0 = arg >> 8;
                        self.old = if OLD_RANGE.contains(&p0) {
                                State::True
                            } else {
                                State::False
                            };
                    }
                    let pred = if self.old == State::True { arg ^ 0x8000 } else { arg };
                    Self::decode_mono(&mut self.abuf, &self.data, pred);
                },
                0x1021 if self.channels > 0 => { // stereo
                    validate!((size & 1) == 0);
                    self.data.resize(size, 0);
                    self.fr.read_buf(&mut self.data)?;
                    // audio format autodetect hack
                    if self.old == State::Unknown {
                        let p0 = arg >> 8;
                        let p1 = arg & 0xFF;
                        self.old = if OLD_RANGE.contains(&p0) && OLD_RANGE.contains(&p1) {
                                State::True
                            } else {
                                State::False
                            };
                    }
                    let pred = if self.old == State::True { arg ^ 0x8080 } else { arg };
                    Self::decode_stereo(&mut self.abuf, &self.data, pred);
                },
                0x1030 => { // container chunk
                },
                _ => {
                    self.fr.read_skip(size)?;
                }
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_u16le()?;
    validate!(tag == 0x1084);
    let size = fr.read_u32le()?;
    validate!(size == 0 || size == 0xFFFFFFFF);
    let mc_2x = size == 0;
    let mut fps = fr.read_u16le()?;
    if fps == 0 {
        fps = 30;
    }
    validate!((1..=30).contains(&fps));

    let mut channels = 0;
    let mut nchunks = 0;
    let mut config_found = false;
    let mut width = 0;
    let mut height = 0;
    let mut alpha = false;
    while let Ok(tag) = fr.read_u16le() {
        match tag {
            0x1001 => {
                let size = fr.read_u32le()?;
                validate!(size >= 8);
                let arg = fr.read_u16le()?;
                if arg > 1 {
                    return Err(DecoderError::NotImplemented);
                }
                alpha = arg == 1;
                width = fr.read_u16le()? as usize;
                height = fr.read_u16le()? as usize;
                validate!((1..=1024).contains(&width) && (1..=1024).contains(&height) && ((width | height) & 3) == 0);
                fr.read_skip((size - 4) as usize)?;
                config_found = true;
            },
            0x1030 => {
                fr.read_skip(6)?;
            },
            0x1020 => {
                channels = 1;
                if config_found {
                    break;
                }
                let size = fr.read_u32le()? as usize;
                fr.read_u16le()?;
                fr.read_skip(size)?;
            },
            0x1021 => {
                channels = 2;
                if config_found {
                    break;
                }
                let size = fr.read_u32le()? as usize;
                fr.read_u16le()?;
                fr.read_skip(size)?;
            },
            _ => {
                let size = fr.read_u32le()? as usize;
                fr.read_u16le()?;
                fr.read_skip(size)?;
            }
        }
        nchunks += 1;
        if nchunks > 16 {
            break;
        }
    }
    validate!(config_found || channels > 0);

    fr.seek(SeekFrom::Start(8))?;

    Ok(Box::new(RoQDecoder {
        fr,
        width, height, fps, alpha, mc_2x,
        old: State::Unknown,
        data: Vec::new(),
        jpeg: JPEGDecoder::new(),
        yuv2rgb: YUV2RGB::new(),
        frame: vec![0; width * 3 * height],
        pframe: vec![0; width * 3 * height],
        cb: [[0; 3 * 4]; 256],
        num_2x2: 0,
        cb4: [[0; 4]; 256],
        num4: 0,
        frameno: 0,
        abuf: Vec::new(),
        audio: width == 0,
        ablk_len: (22050 / usize::from(fps)) * usize::from(channels.max(1)),
        channels,
    }))
}
