use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::input::util::lzss::*;
use super::super::*;

struct ShadowcasterDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    palettes:   Vec<[u8; 768]>,
    frame:      Vec<u8>,
    data:       Vec<u8>,
    udata:      Vec<u8>,
    width:      usize,
    height:     usize,
    fps:        u16,
    repeat:     u8,
}

impl ShadowcasterDecoder {
    fn unpack_intra(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        let _buf_no = usize::from(br.read_byte()?);
        let xoff   = br.read_u16le()? as usize;
        let yoff   = br.read_u16le()? as usize;
        let width  = br.read_u16le()? as usize;
        let height = br.read_u16le()? as usize;
        validate!(xoff + width <= self.width && yoff + height <= self.height);

        for line in self.frame.chunks_exact_mut(self.width).skip(yoff).take(height) {
            let mut pos = xoff;
            while pos < xoff + width {
                let op = br.read_byte()? as i8;
                if op <= 0 {
                    let copy_len = -op as usize;
                    validate!(pos + copy_len <= xoff + width);
                    br.read_buf(&mut line[pos..][..copy_len])?;
                    pos += copy_len;
                } else {
                    let run_len = op as usize;
                    let clr = br.read_byte()?;
                    validate!(pos + run_len <= xoff + width);
                    for el in line[pos..][..run_len].iter_mut() {
                        *el = clr;
                    }
                    pos += run_len;
                }
            }
        }

        Ok(())
    }

    fn unpack_inter(&mut self) -> DecoderResult<()> {
        validate!(self.data.len() > 4);
        let len = read_u32le(&self.data)? as usize;
        let src = if len == 0xFFFFFFFF {
                &self.data[4..]
            } else if len > 0 {
                validate!(self.data.len() > 8 && len > self.data.len() - 8 && len < (1 << 20));
                self.udata.resize(len, 0);
                lzss_unpack(&self.data[4..][..self.data.len() - 8], &mut self.udata)?;
                &self.udata
            } else {
                return Ok(())
            };

        let mut br = MemoryReader::new_read(src);

        let _buf_no = usize::from(br.read_byte()?);
        let yoff   = br.read_u16le()? as usize;
        let height = br.read_u16le()? as usize;
        validate!(yoff + height <= self.height);

        for line in self.frame.chunks_exact_mut(self.width).skip(yoff).take(height) {
            let b = usize::from(br.read_byte()?);
            let (mut pos, nsegs) = if b < 0x80 { (0, b) } else { (0x100, b & 0x7F) };
            for _ in 0..nsegs {
                pos += usize::from(br.read_byte()?);
                let num_ops = usize::from(br.read_byte()?);
                validate!(pos <= self.width);
                for _ in 0..num_ops {
                    let op = br.read_byte()? as i8;
                    if op <= 0 {
                        let copy_len = -op as usize;
                        validate!(pos + copy_len <= self.width);
                        br.read_buf(&mut line[pos..][..copy_len])?;
                        pos += copy_len;
                    } else {
                        let run_len = op as usize;
                        let clr = br.read_byte()?;
                        validate!(pos + run_len <= self.width);
                        for el in line[pos..][..run_len].iter_mut() {
                            *el = clr;
                        }
                        pos += run_len;
                    }
                }
            }
        }

        Ok(())
    }

    fn update_palette(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        let pal_no = usize::from(br.read_byte()?);
        while self.palettes.len() <= pal_no {
            self.palettes.push([0; 768]);
        }
        let nsegs = usize::from(br.read_byte()?);
        let mut idx = 0;
        for _ in 0..nsegs {
            let skip = usize::from(br.read_byte()?);
            let mut count = usize::from(br.read_byte()?);
            if count == 0 {
                count = 256;
            }
            idx += skip;
            validate!(idx + count <= 256);
            br.read_vga_pal_some(&mut self.palettes[pal_no][idx * 3..][..count * 3])?;
            idx += count;
        }

        Ok(())
    }
}

impl InputSource for ShadowcasterDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: u32::from(self.fps),
                tb_den: 70,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = &mut self.fr;
        loop {
            if self.repeat > 0 {
                self.repeat -= 1;
                return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
            }

            let pos = br.tell();
            let tag = br.read_u16le().map_err(|_| DecoderError::EOF)?;
            let size = br.read_u32le()? as usize;
            validate!(size < (1 << 20));
            self.data.resize(size, 0);
            if size > 0 {
                br.read_buf(&mut self.data)?;
            }

            match tag {
                0x01 => {
                    self.unpack_intra().map_err(|_| DecoderError::InvalidData)?;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                0x02 => {
                    self.unpack_inter().map_err(|_| DecoderError::InvalidData)?;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                0x03 => { // untested
                    validate!(self.width == 320 && self.height == 200);
                    validate!(self.data.len() > 8);
                    let len = read_u32le(&self.data)? as usize;
                    if len == 320 * 200 {
                        validate!(self.data.len() == self.frame.len() + 4);
                        self.frame.copy_from_slice(&self.data[4..]);
                    } else {
                        validate!(self.data.len() >= len + 4);
                        lzss_unpack(&self.data[4..][..len], &mut self.frame)
                            .map_err(|_| DecoderError::InvalidData)?;
                    }
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                0x04 => {
                    self.update_palette().map_err(|_| DecoderError::InvalidData)?;
                    br = &mut self.fr;
                },
                0x10 => {}, // some CD command?
                0x11 => {}, // unknown
                0x12 => {}, // load music track
                0x13 => {}, // unknown
                0x14 => {}, // unload music track
                0x15 => { // some buffer IDs
                    validate!(self.data.len() >= 2);
                    validate!(self.data.len() == usize::from(self.data[0]) + 1);
                },
                0x16 => {
                    validate!(self.data.len() == 1);
                    let pal_idx = usize::from(self.data[0]);
                    validate!(pal_idx < self.palettes.len());
                    self.pal = self.palettes[pal_idx];
                },
                0x17 => {}, // unknown
                0x18 => {}, // unknown
                0x19 => {
                    validate!(self.data.len() == 1);
                    self.repeat = self.data[0];
                },
                0x20 => {
                    self.pal = [0; 768];
                    for el in self.frame.iter_mut() {
                        *el = 0;
                    }
                },
                0x21 => { // fade in
                    validate!(self.data.len() == 1);
                    let pal_idx = usize::from(self.data[0]);
                    validate!(pal_idx == 0 || pal_idx < self.palettes.len());
                    if pal_idx != 0 {
                        self.pal = self.palettes[pal_idx];
                    }
                },
                0x22 => { // fade out
                    validate!(self.data.len() == 1);
                    let pal_idx = usize::from(self.data[0]);
                    validate!(pal_idx == 0 || pal_idx < self.palettes.len());
                    self.pal = [0; 768];
                },
                0x23 => {}, // unknown
                _ => { println!(" unknown command {tag} size {size} @ {pos:X}"); },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag = br.read_u16le()?;
    validate!(tag == 0x105);
    let width  = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!((1..=320).contains(&width) && (1..=200).contains(&height));
    let fps = br.read_u16le()?;
    validate!(fps > 1);
    br.seek(SeekFrom::Start(0x40))?;

    Ok(Box::new(ShadowcasterDecoder {
        fr: br,
        pal: [0; 768],
        palettes: Vec::with_capacity(4),
        data: Vec::new(),
        udata: Vec::new(),
        frame: vec![0; width * height],
        width, height, fps,
        repeat: 0,
    }))
}
