use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const DEFAULT_VBV_DELAY: u32 = 80;

const VB_FLAG_GMC:      u16 = 0x0001;
const VB_FLAG_AUDIO:    u16 = 0x0004;
const VB_FLAG_VIDEO:    u16 = 0x0008;
const VB_FLAG_PALETTE:  u16 = 0x0010;
const VB_FLAG_DELAY:    u16 = 0x0020;

fn check_size(x: usize, row_no: usize, dx: i16, dy: i16, stride: usize, len: usize) -> Option<usize> {
    let start = (x as i32) + i32::from(dx) + ((row_no * 4) as i32 + i32::from(dy)) * (stride as i32);
    if start >= 0 {
        let start = start as usize;
        let end = start + 4 + stride * 3;

        if end <= len {
            Some(start)
        } else {
            None
        }
    } else {
        None
    }
}

fn decode_video8(br: &mut dyn ByteIO, dst: &mut [u8], prev_frame: &[u8], width: usize, gmv: [i16; 2]) -> DecoderResult<()> {
    let mut btc = 0;
    let mut btypes = 0;
    for (row_no, strip) in dst.chunks_mut(width * 4).enumerate() {
        for x in (0..width).step_by(4) {
            if btc == 0 {
                btypes                          = br.read_byte()?;
                btc = 4;
            }
            match btypes & 0xC0 {
                0xC0 => {
                    let t                       = br.read_byte()?;
                    let mut pattern = VB_PATTERNS[(t & 0x3F) as usize];
                    let op = t >> 6;
                    validate!(op != 3);
                    if op == 0 {
                        let mut clr = [0; 2];
                                              br.read_buf(&mut clr)?;
                        for dline in strip[x..].chunks_mut(width).take(4) {
                            for el in dline[..4].iter_mut() {
                                *el = clr[(pattern & 1) as usize];
                                pattern >>= 1;
                            }
                        }
                    } else {
                        if op == 2 {
                            pattern = !pattern;
                        }
                        let clr             = br.read_byte()?;

                        if let Some(start) = check_size(x, row_no, gmv[0], gmv[1], width, prev_frame.len()) {
                            for (dline, sline) in strip[x..].chunks_mut(width).zip(prev_frame[start..].chunks(width)).take(4) {
                                for (dst, &src) in dline[..4].iter_mut().zip(sline.iter()) {
                                    *dst = if (pattern & 1) != 0 { clr } else { src };
                                    pattern >>= 1;
                                }
                            }
                        } else {
                            return Err(DecoderError::InvalidData);
                        }
                    }
                },
                0x80 => {
                    let clr                 = br.read_byte()?;
                    for dline in strip[x..].chunks_mut(width).take(4) {
                        for el in dline[..4].iter_mut() {
                            *el = clr;
                        }
                    }
                },
                0x40 => {
                    let mv                  = br.read_byte()?;
                    if mv == 0 {
                        for dline in strip[x..].chunks_mut(width).take(4) {
                                          br.read_buf(&mut dline[..4])?;
                        }
                    } else {
                        let mvx = (((mv & 0xF) ^ 8) as i16) - 8;
                        let mvy = (((mv >>  4) ^ 8) as i16) - 8;
                        if let Some(start) = check_size(x, row_no, gmv[0] + mvx, gmv[1] + mvy, width, prev_frame.len()) {
                            for (dline, sline) in strip[x..].chunks_mut(width).zip(prev_frame[start..].chunks(width)).take(4) {
                                dline[..4].copy_from_slice(&sline[..4]);
                            }
                        } else {
                            return Err(DecoderError::InvalidData);
                        }
                    }
                },
                _ => {
                    if let Some(start) = check_size(x, row_no, gmv[0], gmv[1], width, prev_frame.len()) {
                        for (dline, sline) in strip[x..].chunks_mut(width).zip(prev_frame[start..].chunks(width)).take(4) {
                            dline[..4].copy_from_slice(&sline[..4]);
                        }
                    } else {
                        return Err(DecoderError::InvalidData);
                    }
                },
            }

            btypes <<= 2;
            btc     -= 1;
        }
    }

    Ok(())
}

struct FCPDecoder {
    fr:         FileReader<BufReader<File>>,
    size:       u32,
    nframes:    usize,
    cframe:     usize,
    width:      usize,
    height:     usize,
    flags:      u16,
    abuf:       Vec<u8>,
    pal:        [u8; 768],
    cur_frame:  Vec<u8>,
    prev_frame: Vec<u8>,
    audio:      bool,
}

impl InputSource for FCPDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 2,
                    tb_den: 25,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        let is_end = br.tell() >= u64::from(self.size) || self.cframe >= self.nframes;

        let afrm_len = 1765;
        if self.abuf.len() >= afrm_len && (self.audio || is_end) {
            let mut audio = vec![0; afrm_len];
            audio.copy_from_slice(&self.abuf[..afrm_len]);
            self.abuf.drain(..afrm_len);
            self.audio = false;
            return Ok((1, Frame::AudioU8(audio)));
        }

        if is_end {
            return Err(DecoderError::EOF);
        }

        let size                        = br.read_u16le()? as usize;
        validate!(size > 8);
        let end = br.tell() + (size as u64) - 2;
        self.cframe += 1;

        let asize                       = br.read_u16le()? as usize;
        let _duration                   = br.read_u16le()? as u64;
        let nclrs                       = br.read_u16le()? as usize;
        if nclrs > 0 {
            let pal_start               = br.read_u16le()? as usize;
            validate!(pal_start + nclrs <= 256);
            br.read_buf(&mut self.pal[pal_start * 3..][..nclrs * 3])?;
        }
        if asize > 0 {
            br.read_extend(&mut self.abuf, asize)?;
        }

        let (gmv_x, gmv_y) = if (self.flags & 0x2000) != 0 {
                let mvx                         = br.read_u16le()? as i16;
                let mvy                         = br.read_u16le()? as i16;
                (mvx, mvy)
            } else {
                (0, 0)
            };

        std::mem::swap(&mut self.cur_frame, &mut self.prev_frame);
        decode_video8(&mut *br, &mut self.cur_frame, &self.prev_frame, self.width, [gmv_x, gmv_y])?;
        validate!(br.tell() == end);

        self.audio = true;

        Ok((0, Frame::VideoPal(self.cur_frame.clone(), self.pal)))
    }
}

struct AudioData {
    abuf:       Vec<u8>,
    srate:      u32,
    channels:   u8,
    bits:       u8,
    frm_size:   usize,
}

impl AudioData {
    fn get_stream_info(&self) -> StreamInfo {
        StreamInfo::Audio(AudioInfo{
            sample_rate: self.srate,
            channels:    self.channels,
            sample_type: if self.bits == 8 { AudioSample::U8 } else { AudioSample::S16 },
        })
    }
    fn get_frame(&mut self, last: bool) -> DecoderResult<(usize, Frame)> {
        if self.abuf.is_empty() || (!last && self.abuf.len() < self.frm_size) {
            return Err(DecoderError::InvalidData);
        }
        let mut raw_size = self.frm_size;
        if last {
            raw_size = raw_size.min(self.abuf.len());
        }
        match self.bits {
            8 => {
                let mut audio = vec![0; raw_size];
                audio.copy_from_slice(&self.abuf[..raw_size]);
                self.abuf.drain(..raw_size);
                Ok((1, Frame::AudioU8(audio)))
            },
            16 => {
                let mut audio = vec![0; raw_size / 2];
                for (dst, src) in audio.iter_mut().zip(self.abuf.chunks_exact(2)) {
                    *dst = read_u16le(src).unwrap_or(0) as i16;
                }
                self.abuf.drain(..raw_size);
                Ok((1, Frame::AudioS16(audio)))
            }
            _ => Err(DecoderError::NotImplemented),
        }
    }
}

struct VBV1Decoder {
    fr:         FileReader<BufReader<File>>,
    size:       u32,
    adata:      AudioData,
    audio:      bool,
    nframes:    usize,
    cframe:     usize,
    width:      usize,
    height:     usize,
    pal:        [u8; 768],
    cur_frame:  Vec<u8>,
    prev_frame: Vec<u8>,
    tb_num:     u32,
    tb_den:     u32,
}

impl InputSource for VBV1Decoder {
    fn get_num_streams(&self) -> usize { if self.adata.srate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: self.tb_num,
                    tb_den: self.tb_den,
                 }),
            1 if self.adata.srate > 0 => self.adata.get_stream_info(),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        if self.audio {
            if let Ok(ret) = self.adata.get_frame(false) {
                self.audio = false;
                return Ok(ret);
            }
        }

        if self.cframe >= self.nframes {
            if let Ok(ret) = self.adata.get_frame(true) {
                return Ok(ret);
            }
            return Err(DecoderError::EOF);
        }

        let size                        = br.read_u32le()? as usize;
        validate!(size > 6);
        let end = br.tell() + (size as u64) - 4;

        let flags                       = br.read_u16le()?;
        let (gmv_x, gmv_y) = if (flags & VB_FLAG_GMC) != 0 {
                let mvx                         = br.read_u16le()? as i16;
                let mvy                         = br.read_u16le()? as i16;
                (mvx, mvy)
            } else {
                (0, 0)
            };
        if (flags & VB_FLAG_AUDIO) != 0 {
            let asize                   = br.read_u32le()? as usize;
            validate!(asize > 4);
            validate!(br.tell() + (asize as u64) - 4 <= end);
                                          br.read_extend(&mut self.adata.abuf, asize - 4)?;
        }
        if (flags & VB_FLAG_VIDEO) != 0 {
            let vsize                   = br.read_u32le()? as usize;
            validate!(vsize > 4);
            let vend = br.tell() + (vsize as u64) - 4;
            validate!(vend <= end);

            std::mem::swap(&mut self.cur_frame, &mut self.prev_frame);
            decode_video8(&mut *br, &mut self.cur_frame, &self.prev_frame, self.width, [gmv_x, gmv_y])?;
            validate!(br.tell() == vend);
        }
        if (flags & VB_FLAG_PALETTE) != 0 {
            let pal_size                        = br.read_u32le()? as usize;
            validate!(pal_size > 6);
            validate!(br.tell() + (pal_size as u64) - 4 <= end);
            let start                           = br.read_byte()? as usize;
            let mut size                        = br.read_byte()? as usize;
            if size == 0 {
                size = 256;
            }
            validate!(start + size <= 256);
            validate!(size * 3 + 6 == pal_size);
                                                  br.read_buf(&mut self.pal[start * 3..][..size * 3])?;
        }
        let _delay = if (flags & VB_FLAG_DELAY) != 0 {
                                          br.read_u16le()? as u64
            } else { 0 };

        validate!(br.tell() <= end);
        br.seek(SeekFrom::Start(end))?;

        self.cframe += 1;
        self.audio = true;

        Ok((0, Frame::VideoPal(self.cur_frame.clone(), self.pal)))
    }
}

struct VBV2Decoder {
    fr:         FileReader<BufReader<File>>,
    size:       u32,
    adata:      AudioData,
    audio:      bool,
    nframes:    usize,
    cframe:     usize,
    width:      usize,
    height:     usize,
    cur_frame:  Vec<u16>,
    prev_frame: Vec<u16>,
    pal:        [u16; 256],
    tb_num:     u32,
    tb_den:     u32,
}

impl InputSource for VBV2Decoder {
    fn get_num_streams(&self) -> usize { if self.adata.srate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    15,
                    tb_num: self.tb_num,
                    tb_den: self.tb_den,
                 }),
            1 if self.adata.srate > 0 => self.adata.get_stream_info(),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        if self.audio {
            if let Ok(ret) = self.adata.get_frame(false) {
                self.audio = false;
                return Ok(ret);
            }
        }

        if self.cframe >= self.nframes {
            if let Ok(ret) = self.adata.get_frame(true) {
                return Ok(ret);
            }
            return Err(DecoderError::EOF);
        }

        let size                        = br.read_u32le()? as usize;
        validate!(size > 6);
        let end = br.tell() + (size as u64) - 4;

        let flags                       = br.read_u16le()?;
        let (gmv_x, gmv_y) = if (flags & VB_FLAG_GMC) != 0 {
                let mvx                         = br.read_u16le()? as i16;
                let mvy                         = br.read_u16le()? as i16;
                (mvx, mvy)
            } else {
                (0, 0)
            };
        if (flags & VB_FLAG_AUDIO) != 0 {
            let asize                   = br.read_u32le()? as usize;
            validate!(asize > 4);
            validate!(br.tell() + (asize as u64) - 4 <= end);
                                          br.read_extend(&mut self.adata.abuf, asize - 4)?;
        }
        if (flags & VB_FLAG_VIDEO) != 0 {
            let vsize                   = br.read_u32le()? as usize;
            validate!(vsize > 4);
            let vend = br.tell() + (vsize as u64) - 4;
            validate!(vend <= end);

            let has_pal = (flags & VB_FLAG_PALETTE) != 0;
            if has_pal {
                let cur_off = br.tell();
                                                  br.seek(SeekFrom::Current((vsize - 4) as i64))?;
                let psize                       = br.read_u32le()? as usize;
                validate!(psize > 4 && psize <= 0x204 && (psize & 1) == 0);
                for el in self.pal[..(psize - 4)/ 2].iter_mut() {
                    *el                         = br.read_u16le()?;
                }
                                                  br.seek(SeekFrom::Start(cur_off))?;
            }

            std::mem::swap(&mut self.cur_frame, &mut self.prev_frame);

            let mut btc = 0;
            let mut btypes = 0;
            for (row_no, strip) in self.cur_frame.chunks_mut(self.width * 4).enumerate() {
                for x in (0..self.width).step_by(4) {
                    if btc == 0 {
                        btypes                  = br.read_byte()?;
                        btc = 4;
                    }
                    match btypes & 0xC0 {
                        0xC0 => {
                            let t               = br.read_byte()?;
                            let mut pattern = VB_PATTERNS[(t & 0x3F) as usize];
                            let op = t >> 6;
                            validate!(op != 3);
                            if op == 0 {
                                let mut clr = [0; 2];
                                if has_pal {
                                    clr[0]      = self.pal[br.read_byte()? as usize];
                                    clr[1]      = self.pal[br.read_byte()? as usize];
                                } else {
                                    clr[0]      = br.read_u16le()?;
                                    clr[1]      = br.read_u16le()?;
                                }
                                for dline in strip[x..].chunks_mut(self.width).take(4) {
                                    for el in dline[..4].iter_mut() {
                                        *el = clr[(pattern & 1) as usize];
                                        pattern >>= 1;
                                    }
                                }
                            } else {
                                if op == 2 {
                                    pattern = !pattern;
                                }
                                let clr = if has_pal {
                                                  self.pal[br.read_byte()? as usize]
                                    } else {
                                                  br.read_u16le()?
                                    };

                                if let Some(start) = check_size(x, row_no, gmv_x, gmv_y, self.width, self.prev_frame.len()) {
                                    for (dline, sline) in strip[x..].chunks_mut(self.width).zip(self.prev_frame[start..].chunks(self.width)).take(4) {
                                        for (dst, &src) in dline[..4].iter_mut().zip(sline.iter()) {
                                            *dst = if (pattern & 1) != 0 { clr } else { src };
                                            pattern >>= 1;
                                        }
                                    }
                                } else {
                                    return Err(DecoderError::InvalidData);
                                }
                            }
                        },
                        0x80 => {
                            let clr = if has_pal {
                                                  self.pal[br.read_byte()? as usize]
                                } else {
                                                  br.read_u16le()?
                                };
                            for dline in strip[x..].chunks_mut(self.width).take(4) {
                                for el in dline[..4].iter_mut() {
                                    *el = clr;
                                }
                            }
                        },
                        0x40 => {
                            let mv              = br.read_byte()?;
                            if mv == 0 {
                                if has_pal {
                                    for dline in strip[x..].chunks_mut(self.width).take(4) {
                                        for el in dline[..4].iter_mut() {
                                            *el = self.pal[br.read_byte()? as usize];
                                        }
                                    }
                                } else {
                                    for dline in strip[x..].chunks_mut(self.width).take(4) {
                                        for el in dline[..4].iter_mut() {
                                            *el = br.read_u16le()?;
                                        }
                                    }
                                }
                            } else {
                                let mvx = (((mv & 0xF) ^ 8) as i16) - 8;
                                let mvy = (((mv >>  4) ^ 8) as i16) - 8;
                                if let Some(start) = check_size(x, row_no, gmv_x + mvx, gmv_y + mvy, self.width, self.prev_frame.len()) {
                                    for (dline, sline) in strip[x..].chunks_mut(self.width).zip(self.prev_frame[start..].chunks(self.width)).take(4) {
                                        dline[..4].copy_from_slice(&sline[..4]);
                                    }
                                } else {
                                    return Err(DecoderError::InvalidData);
                                }
                            }
                        },
                        _ => {
                            if let Some(start) = check_size(x, row_no, gmv_x, gmv_y, self.width, self.prev_frame.len()) {
                                for (dline, sline) in strip[x..].chunks_mut(self.width).zip(self.prev_frame[start..].chunks(self.width)).take(4) {
                                    dline[..4].copy_from_slice(&sline[..4]);
                                }
                            } else {
                                return Err(DecoderError::InvalidData);
                            }
                        },
                    }

                    btypes <<= 2;
                    btc     -= 1;
                }
            }
            validate!(br.tell() == vend);
        }
        if (flags & VB_FLAG_PALETTE) != 0 {
            let pal_size                        = br.read_u32le()? as usize;
            validate!(pal_size > 6);
            validate!(br.tell() + (pal_size as u64) - 4 <= end);
                                                  br.read_skip(pal_size - 4)?;
        }
        let _delay = if (flags & VB_FLAG_DELAY) != 0 {
                                          br.read_u16le()? as u64
            } else { 0 };

        validate!(br.tell() <= end);
        br.seek(SeekFrom::Start(end))?;

        self.cframe += 1;
        self.audio = true;

        Ok((0, Frame::VideoRGB16(self.cur_frame.clone())))
    }
}

struct SndDecoder {
    fr:         FileReader<BufReader<File>>,
    size:       u32,
    srate:      u32,
    abits:      u8,
    channels:   u8,
}

impl InputSource for SndDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Audio(AudioInfo{
                sample_rate: self.srate,
                channels:    self.channels,
                sample_type: if self.abits > 8 { AudioSample::S16 } else { AudioSample::U8 },
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;

        if self.size == 0 {
            return Err(DecoderError::EOF);
        }

        let blk_size = self.size.min(3072);
        self.size -= blk_size;
        match self.abits {
            8 => {
                let mut audio = vec![0; blk_size as usize];
                br.read_buf(&mut audio)?;
                Ok((0, Frame::AudioU8(audio)))
            },
            12 => {
                let mut audio = vec![0; blk_size as usize * 2 / 3];
                for pair in audio.chunks_exact_mut(2) {
                    let val = br.read_u24le()? as i32;
                    pair[0] = ((val & 0xFFF) << 20 >> 16) as i16;
                    pair[1] = ((val & 0xFFF000) << 8 >> 16) as i16;
                }
                Ok((0, Frame::AudioS16(audio)))
            },
            16 => {
                let mut audio = vec![0; blk_size as usize / 2];
                for el in audio.iter_mut() {
                    *el = br.read_u16le()? as i16;
                }
                Ok((0, Frame::AudioS16(audio)))
            },
            _ => Err(DecoderError::NotImplemented),
        }
    }
}

fn scan_for_framerate(br: &mut dyn ByteIO, nframes: usize, def_delay: u32) -> DecoderResult<(u32, u32)> {
    let pos = br.tell();

    let count = nframes.min(16);
    let mut delays = 0;
    for _ in 0..count {
        let size = br.read_u32le()? as usize;
        validate!(size > 6);
        let next = br.tell() + (size as u64) - 4;
        let flags                       = br.read_u16le()?;
        if (flags & VB_FLAG_GMC) != 0 {
                                          br.read_skip(4)?;
        }
        if (flags & VB_FLAG_AUDIO) != 0 {
            let asize                   = br.read_u32le()? as usize;
            validate!(asize > 4);
                                          br.read_skip(asize - 4)?;
        }
        if (flags & VB_FLAG_VIDEO) != 0 {
            let vsize                   = br.read_u32le()? as usize;
            validate!(vsize > 4);
                                          br.read_skip(vsize - 4)?;
        }
        if (flags & VB_FLAG_PALETTE) != 0 {
            let psize                   = br.read_u32le()? as usize;
            validate!(psize > 4);
                                          br.read_skip(psize - 4)?;
        }
        let delay = if (flags & VB_FLAG_DELAY) != 0 {
                                          br.read_u16le()?
            } else { 0 };
        br.seek(SeekFrom::Start(next))?;

        if delay > 0 {
            delays += u32::from(delay);
        } else {
            delays += def_delay;
        }
    }

    br.seek(SeekFrom::Start(pos))?;
    if count > 0 {
        let avg = (delays + (count as u32) / 2) / (count as u32);
        match avg {
            33 | 34 => Ok((1, 30)),
            40 => Ok((1, 25)),
            41 => Ok((1, 24)),
            50 => Ok((1, 20)),
            66 | 67 => Ok((1, 15)),
            77 => Ok((1, 13)),
            80 => Ok((2, 25)),
            83 => Ok((1, 12)),
            91 => Ok((1, 11)),
            100 => Ok((1, 10)),
            _ => Ok((avg, 1000)),
        }
    } else {
        Ok((def_delay, 1000))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let magic                       = br.read_tag()?;
    validate!(&magic == b"SIFF");
    let full_size                   = br.read_u32be()?;
    let tag                         = br.read_tag()?;

    match &tag {
        b"FCPK" => {
            let tag                         = br.read_tag()?;
            validate!(&tag == b"FCHD");
            let hdr_size                    = br.read_u32be()? as usize;
            validate!(hdr_size >= 16);
            let flags                       = br.read_u16le()?;
            let width                       = br.read_u16le()? as usize;
            let height                      = br.read_u16le()? as usize;
            validate!(width > 0 && height > 0);
            let nframes                     = br.read_u16le()? as usize;
                                              br.read_skip(8)?;
                                              br.read_skip(hdr_size - 16)?;

            let tag                         = br.read_tag()?;
            validate!(&tag == b"BODY");
            let size                        = br.read_u32be()?;
            validate!(br.tell() + u64::from(size) <= u64::from(full_size) + 8);

            Ok(Box::new(FCPDecoder {
                fr: br,
                width, height, flags,
                pal: [0; 768],
                cur_frame: vec![0; width * height],
                prev_frame: vec![0; width * height],
                abuf: Vec::with_capacity(22050),
                audio: false,
                nframes, size,
                cframe: 0,
            }))
        },
        b"VBV1" => {
            let tag                         = br.read_tag()?;
            validate!(&tag == b"VBHD");
            let hdr_size                    = br.read_u32be()? as usize;
            validate!(hdr_size >= 32);
            let version                     = br.read_u16le()?;
            validate!(version == 1 || version == 2);
            let width                       = br.read_u16le()? as usize;
            let height                      = br.read_u16le()? as usize;
            validate!(width > 0 && height > 0);
                                              br.read_skip(4)?;
            let nframes                     = br.read_u16le()? as usize;
            let flags                       = br.read_u16le()?;
            let bits = flags as u8;
            let channels = if (flags & 0x100) != 0 { 2 } else { 1 };
            validate!(bits == 0 || bits >= 8);
            let srate                       = br.read_u16le()? as u32;
                                              br.read_skip(16)?;
                                              br.read_skip(hdr_size - 32)?;

            let tag                         = br.read_tag()?;
            validate!(&tag == b"BODY");
            let size                        = br.read_u32be()?;
            validate!(br.tell() + u64::from(size) <= u64::from(full_size) + 8);

            let (tb_num, tb_den) = scan_for_framerate(&mut br, nframes, DEFAULT_VBV_DELAY)?;

            let frm_size = srate * tb_num * u32::from(channels) * u32::from(bits) / 8 / tb_den;
            let adata = AudioData {
                    srate, channels, bits,
                    frm_size: frm_size as usize,
                    abuf: Vec::new(),
                };

            if version == 1 {
                Ok(Box::new(VBV1Decoder {
                    fr: br,
                    width, height,
                    pal: [0; 768],
                    audio: false,
                    cur_frame: vec![0; width * height],
                    prev_frame: vec![0; width * height],
                    nframes, size,
                    cframe: 0,
                    adata,
                    tb_num, tb_den,
                }))
            } else {
                Ok(Box::new(VBV2Decoder {
                    fr: br,
                    width, height,
                    cur_frame: vec![0; width * height],
                    prev_frame: vec![0; width * height],
                    pal: [0; 256],
                    nframes, size,
                    cframe: 0,
                    adata,
                    audio: false,
                    tb_num, tb_den,
                }))
            }
        },
        b"SOUN" => {
            let tag                         = br.read_tag()?;
            validate!(&tag == b"SHDR");
            let hdr_size                    = br.read_u32be()? as usize;
            validate!(hdr_size >= 8);
            let _duration                   = br.read_u32le()? as u64;
            let srate                       = u32::from(br.read_u16le()?);
            let flags                       = br.read_u16le()?;
            let abits = flags as u8;
            validate!(matches!(abits, 8 | 12 | 16));
            let channels = if (flags & 0x100) != 0 { 2 } else { 1 };
                                              br.read_skip(hdr_size - 8)?;

            let tag                         = br.read_tag()?;
            validate!(&tag == b"BODY");
            let size                        = br.read_u32be()?;
            validate!(br.tell() + u64::from(size) <= u64::from(full_size) + 8);

            Ok(Box::new(SndDecoder {
                fr: br,
                srate, abits, channels, size,
            }))
        },
        _ => unreachable!(),
    }
}

const VB_PATTERNS: [u16; 64] = [
    0x0660, 0xFF00, 0xCCCC, 0xF000, 0x8888, 0x000F, 0x1111, 0xFEC8,
    0x8CEF, 0x137F, 0xF731, 0xC800, 0x008C, 0x0013, 0x3100, 0xCC00,
    0x00CC, 0x0033, 0x3300, 0x0FF0, 0x6666, 0x00F0, 0x0F00, 0x2222,
    0x4444, 0xF600, 0x8CC8, 0x006F, 0x1331, 0x318C, 0xC813, 0x33CC,
    0x6600, 0x0CC0, 0x0066, 0x0330, 0xF900, 0xC88C, 0x009F, 0x3113,
    0x6000, 0x0880, 0x0006, 0x0110, 0xCC88, 0xFC00, 0x00CF, 0x88CC,
    0x003F, 0x1133, 0x3311, 0xF300, 0x6FF6, 0x0603, 0x08C6, 0x8C63,
    0xC631, 0x6310, 0xC060, 0x0136, 0x136C, 0x36C8, 0x6C80, 0x324C
];
