use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;
use crate::input::util::jpeg::*;
use crate::input::util::yuv::*;

struct SKSDecoder {
    fr:         FileReader<BufReader<File>>,
    offsets:    Vec<u32>,
    sizes:      Vec<u32>,
    cur_frm:    usize,
    width:      usize,
    height:     usize,
    fps:        u32,
    data:       Vec<u8>,
    jpeg:       JPEGDecoder,
    yuv2rgb:    YUV2RGB,
}

impl InputSource for SKSDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    24,
                tb_num: 1,
                tb_den: self.fps,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frm >= self.offsets.len() {
            return Err(DecoderError::EOF);
        }
        self.fr.seek(SeekFrom::Start(u64::from(self.offsets[self.cur_frm])))?;
        self.data.resize(self.sizes[self.cur_frm] as usize, 0);
        self.cur_frm += 1;

        self.fr.read_buf(&mut self.data)?;
        let mut jfrm = self.jpeg.decode(&self.data).map_err(|_| DecoderError::InvalidData)?;
        validate!(jfrm.width == self.width && jfrm.height == self.height);
        // colourplanes are swapped, unswap them back
        jfrm.offset.swap(1, 2);
        jfrm.stride.swap(1, 2);
        let frame = jfrm.to_rgb(&self.yuv2rgb);
        Ok((0, Frame::VideoRGB24(frame)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    validate!(&tag == b"SK10");
    let width = fr.read_u32le()? as usize;
    let height = fr.read_u32le()? as usize;
    validate!(width > 0 && height > 0 && ((width | height) & 3) == 0);
    let fps = fr.read_u32le()?;
    validate!((1..=30).contains(&fps));
    let nframes = fr.read_u32le()? as usize;
    validate!((1..=10000).contains(&nframes));
    fr.read_skip(12)?;

    let mut offsets = Vec::with_capacity(nframes);
    let mut sizes = Vec::with_capacity(nframes);
    let mut expected_off = (nframes as u32) * 8 + 0x28;
    for _ in 0..nframes {
        let offset = fr.read_u32le()?;
        let size = fr.read_u32le()?;
        validate!(offset >= expected_off && offset < 0x80000000 && size < 0x1000000);
        offsets.push(offset);
        sizes.push(size);
        expected_off = offset + size;
    }
    let last_off = fr.read_u32le()?;
    validate!(last_off >= expected_off);
    let last_size = fr.read_u32le()?;
    validate!(last_size == 0xCDCDCDCD);

    Ok(Box::new(SKSDecoder {
        fr,
        width, height, fps,
        offsets, sizes,
        cur_frm: 0,
        data: Vec::new(),
        jpeg: JPEGDecoder::new(),
        yuv2rgb: YUV2RGB::new(),
    }))
}
