use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 172;
const HEIGHT: usize = 108;
const FPS: u32 = 6;

struct S2StrDecoder {
    fr:         FileReader<File>,
    frame:      Vec<u8>,
    pal:        [u8; 768],
    arate:      u32,
    abuf:       Vec<u8>,
    audio:      bool,
}

impl S2StrDecoder {
    fn load_pal(fr: &mut dyn ByteIO, pal: &mut [u8; 768], csize: usize) -> DecoderResult<()> {
        let pal_size = fr.read_u32le()? as usize;
        validate!(pal_size <= csize - 0x14 && pal_size > 0 && pal_size <= pal.len());
        fr.read_skip(0xC)?;
        fr.read_vga_pal_some(&mut pal[..pal_size])?;
        fr.read_skip(csize - pal_size - 0x14)?;
        Ok(())
    }
    fn load_frame(&mut self, csize: usize) -> DecoderResult<()> {
        validate!(csize > 0x24);
        let img_size = self.fr.read_u32le()? as usize;
        self.fr.read_skip(16)?;
        let width = self.fr.read_u32le()? as usize;
        let height = self.fr.read_u32le()? as usize;
        self.fr.read_u32le()?;
        validate!(img_size == width * height && img_size <= csize + 0x26);
        validate!((4..=WIDTH).contains(&width) && (width & 3) == 0);
        validate!((4..=HEIGHT).contains(&height) && (height & 3) == 0);
        self.fr.read_u16le()?;

        let mut qline = [0; WIDTH / 4];
        for plane in 0..4 {
            for strip in self.frame.chunks_exact_mut(WIDTH * 4).take(height / 4) {
                for quarter in 0..4 {
                    self.fr.read_buf(&mut qline[..width / 4])?;
                    for (quad, &pix) in strip[quarter * WIDTH..][..width].chunks_exact_mut(4)
                            .zip(qline.iter()) {
                        quad[plane] = pix;
                    }
                }
            }
        }
        self.fr.read_skip(csize - img_size - 0x26)?;

        Ok(())
    }
}

impl InputSource for S2StrDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 }}
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: FPS,
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo {
                    sample_rate: self.arate,
                    sample_type: AudioSample::U8,
                    channels:    1,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let ablk_len = ((self.arate + FPS - 1) / FPS).max(1) as usize;
        loop {
            if self.audio && self.abuf.len() >= ablk_len {
                self.audio = false;
                let mut ret = vec![0; ablk_len];
                ret.copy_from_slice(&self.abuf[..ablk_len]);
                self.abuf.drain(..ablk_len);
                return Ok((1, Frame::AudioU8(ret)));
            }
            let tag = self.fr.read_tag().map_err(|_| DecoderError::EOF)?;
            let csize = self.fr.read_u32le()? as usize;
            validate!(csize >= 8);
            match &tag {
                b"GRPH" => {
                    let gtype = self.fr.read_tag()?;
                    match &gtype {
                        b"PAL " => {
                            Self::load_pal(&mut self.fr, &mut self.pal, csize)?;
                        },
                        b"PIXX" => {
                            self.load_frame(csize)?;
                            self.audio = true;
                            return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                        },
                        _ => return Err(DecoderError::NotImplemented),
                    }
                },
                b"SND " if self.arate > 0 => {
                    let subtype = self.fr.read_tag()?;
                    validate!(&subtype == b"SSMP");
                    let ssize = self.fr.read_u32le()? as usize;
                    validate!(ssize > 0 && ssize <= csize - 0x1C);
                    self.fr.read_skip(0x14)?;
                    self.fr.read_extend(&mut self.abuf, ssize)?;
                    self.fr.read_skip(csize - ssize - 0x1C)?;
                },
                _ => {
                    self.fr.read_skip(csize)?;
                },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let tag = fr.read_tag()?;
    validate!(&tag == b"STHD");
    let version = fr.read_u32le()?;
    validate!(version == 0x100);
    let _smth = fr.read_u32le()?;
    let _zero = fr.read_u32le()?;
    let mut arate = fr.read_u32le()?;
    let abits = fr.read_u16le()?;
                fr.read_u16le()?;
    if arate != 22050 || abits != 8 {
        return Err(DecoderError::NotImplemented);
    }

    let mut pal = [0; 768];

    let tag = fr.read_tag()?;
    let csize = fr.read_u32le()? as usize;
    validate!(&tag == b"GRPH");
    let gtype = fr.read_tag()?;
    validate!(&gtype == b"PAL ");
    S2StrDecoder::load_pal(&mut fr, &mut pal, csize)?;

    let next_tag = fr.peek_tag()?;
    if &next_tag != b"SND " {
        arate = 0;
    }

    Ok(Box::new(S2StrDecoder{
        fr,
        frame: vec![0; WIDTH * HEIGHT],
        pal,
        arate,
        abuf: Vec::new(),
        audio: false,
    }))
}
