use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 288;
const HEIGHT: usize = 168;
const FPS: u32 = 15;

struct VideoData {
    frame:      Vec<u8>,
    pal:        [u8; 768],
    tile1:      [[u8; 4]; 256],
    tile4:      [[u8; 4]; 256],
}

impl VideoData {
    fn decode_intra(&mut self, src: &[u8], xoff: usize, yoff: usize, w: usize, h: usize) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(src);

        let mut bitbuf = 0;
        let mut bits = 0;
        for strip in self.frame[yoff * WIDTH..].chunks_exact_mut(WIDTH * 4).take(h / 4) {
            for x in (xoff..xoff + w).step_by(4) {
                if bits == 0 {
                    bitbuf = br.read_u32be()?;
                    bits = 32;
                }
                if (bitbuf >> 31) == 0 {
                    let idx = usize::from(br.read_byte()?);
                    let tile = &self.tile1[idx];
                    for (yy, line) in strip.chunks_exact_mut(WIDTH).enumerate() {
                        for (xx, el) in line[x..][..4].iter_mut().enumerate() {
                            *el = tile[xx / 2 + yy / 2 * 2];
                        }
                    }
                } else {
                    let mut idc = [0; 4];
                    for el in idc.iter_mut() {
                        *el = usize::from(br.read_byte()?);
                    }
                    for (yy, line) in strip.chunks_exact_mut(WIDTH).enumerate() {
                        for (xx, el) in line[x..][..4].iter_mut().enumerate() {
                            *el = self.tile4[idc[yy / 2 * 2 + xx / 2]][(yy & 1) * 2 + (xx & 1)];
                        }
                    }
                }
                bitbuf <<= 1;
                bits    -= 1;
            }
        }

        Ok(())
    }
    fn decode_inter(&mut self, src: &[u8], xoff: usize, yoff: usize, w: usize, h: usize) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(src);

        let mut bitbuf = 0;
        let mut bits = 0;
        for strip in self.frame[yoff * WIDTH..].chunks_exact_mut(WIDTH * 4).take(h / 4) {
            for x in (xoff..xoff + w).step_by(4) {
                if bits == 0 {
                    bitbuf = br.read_u32be()?;
                    bits = 32;
                }
                if (bitbuf >> 31) != 0 {
                    bitbuf <<= 1;
                    bits    -= 1;
                    if bits == 0 {
                        bitbuf = br.read_u32be()?;
                        bits = 32;
                    }
                    if (bitbuf >> 31) == 0 {
                        let idx = usize::from(br.read_byte()?);
                        let tile = &self.tile1[idx];
                        for (yy, line) in strip.chunks_exact_mut(WIDTH).enumerate() {
                            for (xx, el) in line[x..][..4].iter_mut().enumerate() {
                                *el = tile[xx / 2 + yy / 2 * 2];
                            }
                        }
                    } else {
                        let mut idc = [0; 4];
                        for el in idc.iter_mut() {
                            *el = usize::from(br.read_byte()?);
                        }
                        for (yy, line) in strip.chunks_exact_mut(WIDTH).enumerate() {
                            for (xx, el) in line[x..][..4].iter_mut().enumerate() {
                                *el = self.tile4[idc[yy / 2 * 2 + xx / 2]][(yy & 1) * 2 + (xx & 1)];
                            }
                        }
                    }
                }
                bitbuf <<= 1;
                bits    -= 1;
            }
        }

        Ok(())
    }
}

struct WingNutsDecoder {
    fr:         FileReader<BufReader<File>>,
    data:       Vec<u8>,
    vdata:      VideoData,
}

impl WingNutsDecoder {
    fn unpack_frame(&mut self) -> DecoderResult<()> {
        let src = &self.data;
        if src.len() < 0x100 { // one of those short chunks with unknown content
            return Ok(());
        }
        let mut br = MemoryReader::new_read(src);

        br.read_skip(8)?;

        let mode = br.read_byte()?;
        validate!(mode < 2);
        let mut size = br.read_u24be()? as usize;
        validate!(size >= 10 && size + 8 <= src.len());
        let _width = br.read_u16be()? as usize;
        let _height = br.read_u16be()? as usize;
        let _smth = br.read_u16be()?;

        size -= 10;
        let mut pal_found = false;
        while size > 0 {
            let ctype = br.read_byte()?;
            let csize = br.read_u24be()? as usize;
            validate!(csize >= 4 && csize <= size);
            match ctype {
                0x10 | 0x11 => {
                    validate!(csize >= 12);
                    let xoff = br.read_u16be()? as usize;
                    let yoff = br.read_u16be()? as usize;
                    let hei  = br.read_u16be()? as usize;
                    let wid  = br.read_u16be()? as usize;
                    validate!(xoff + wid <= WIDTH && yoff + hei <= HEIGHT);
                    validate!((wid | hei) & 3 == 0);
                    let mut vdata_left = csize - 12;
                    while vdata_left >= 4 {
                        let vtype = br.read_byte()?;
                        let vsize = br.read_u24be()? as usize;
                        validate!(vsize >= 4 && vsize <= vdata_left);
                        match vtype {
                            0x20 => {
                                validate!(vsize == 0x404);
                                for quad in self.vdata.tile4.iter_mut() {
                                    br.read_buf(quad)?;
                                }
                            },
                            0x21 => {
                                let end = br.tell() + ((vsize - 4) as u64);
                                for quads in self.vdata.tile4.chunks_exact_mut(32) {
                                    let mut flags = br.read_u32be()?;
                                    for quad in quads.iter_mut() {
                                        if (flags >> 31) != 0 {
                                            br.read_buf(quad)?;
                                        }
                                        flags <<= 1;
                                    }
                                }
                                validate!(br.tell() == end);
                            },
                            0x22 => {
                                validate!(vsize == 0x404);
                                for quad in self.vdata.tile1.iter_mut() {
                                    br.read_buf(quad)?;
                                }
                            },
                            0x23 => {
                                let end = br.tell() + ((vsize - 4) as u64);
                                for quads in self.vdata.tile1.chunks_exact_mut(32) {
                                    let mut flags = br.read_u32be()?;
                                    for quad in quads.iter_mut() {
                                        if (flags >> 31) != 0 {
                                            br.read_buf(quad)?;
                                        }
                                        flags <<= 1;
                                    }
                                }
                                validate!(br.tell() == end);
                            },
                            0x30 => {
                                let pos = br.tell() as usize;
                                let vsrc = &src[pos..][..vsize - 4];
                                self.vdata.decode_intra(vsrc, xoff, yoff, wid, hei)
                                    .map_err(|_| DecoderError::InvalidData)?;
                                br.read_skip(vsize - 4)?;
                            },
                            0x31 => {
                                let pos = br.tell() as usize;
                                let vsrc = &src[pos..][..vsize - 4];
                                self.vdata.decode_inter(vsrc, xoff, yoff, wid, hei)
                                    .map_err(|_| DecoderError::InvalidData)?;
                                br.read_skip(vsize - 4)?;
                            },
                            _ => {
                                br.read_skip(vsize - 4)?;
                            }
                        }
                        vdata_left -= vsize;
                    }
                    br.read_skip(vdata_left)?;
                },
                0x40 => {
                    validate!(csize - 4 <= self.vdata.pal.len());
                    br.read_buf(&mut self.vdata.pal[..csize - 4])?;
                    pal_found = true;
                },
                _ => br.read_skip(csize - 4)?,
            }
            size -= csize;
        }
        if mode == 0 {
            validate!(pal_found);
        }

        Ok(())
    }
}

impl InputSource for WingNutsDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: FPS,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let br = &mut self.fr;
        loop {
            let csize = br.read_u32be().map_err(|_| DecoderError::EOF)? as usize;
            let tag = br.read_tag()?;

            /*print!("chunk ");
            for el in tag.iter() {
                if (0x20..0x80).contains(el) {
                    print!("{}", *el as char);
                } else {
                    print!("<{el:02X}>");
                }
            }
            println!(" size {csize:X} @ {:X}", br.tell() - 8);*/

            validate!(csize >= 12);

            if &tag[1..] == b"vid" {
                self.data.resize(csize - 8, 0);
                br.read_buf(&mut self.data)?;
                self.unpack_frame().map_err(|_| DecoderError::InvalidData)?;
                return Ok((0, Frame::VideoPal(self.vdata.frame.clone(), self.vdata.pal)));
            } else {
                br.read_skip(csize - 8)?;
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let size = br.read_u32be()?;
    let tag  = br.read_u32be()?;
               br.read_u32be()?;
    let tag2 = br.read_tag()?;
    validate!(size == 0x1C && tag == 0x08000001 && &tag2 == b"WinD");
    br.read_skip(12)?;

    Ok(Box::new(WingNutsDecoder {
        fr: br,
        data: Vec::new(),
        vdata: VideoData {
                tile1: [[0; 4]; 256],
                tile4: [[0; 4]; 256],
                frame: vec![0; WIDTH * HEIGHT],
                pal: [0; 768],
            },
    }))
}
