use std::fs::File;
use crate::io::bitreader::*;
use crate::io::byteio::*;
use super::super::*;
use super::super::util::imaadpcm::*;

const WIDTH:  usize = 320;
const CWIDTH: usize = WIDTH / 2;
const HEIGHT: usize = 200;
const CHEIGHT: usize = HEIGHT / 2;
const ARATE: u32 = 22050;
const ABLK_SIZE: usize = 1470 * 2;

struct YUV2RGB {
    table:  [u16; 1 << 15],
}

impl YUV2RGB {
    #[allow(clippy::excessive_precision)]
    fn new() -> Self {
        let mut table = [0; 1 << 15];
        for (i, el) in table.iter_mut().enumerate() {
            let y = (i & 0x1F) as f32;
            let u = ((i >> 5) & 0x1F) as f32 - 15.0;
            let v = (i >> 10)         as f32 - 15.0;

            let r = (y * 1.00309    + u * 0.964849 + v * 0.61786   + 0.5).max(0.0).min(31.0) as u16;
            let g = (y * 0.99677598 - u * 0.270706 - v * 0.644788  + 0.5).max(0.0).min(31.0) as u16;
            let b = (y * 1.0085     - u * 1.11049  + v * 1.6995699 + 0.5).max(0.0).min(31.0) as u16;
            *el = (r << 10) | (g << 5) | b;
        }
        Self { table }
    }
    fn convert(&self, y: u8, u: u8, v: u8) -> u16 {
        let idx = usize::from(y) | (usize::from(u) << 5) | (usize::from(v) << 10);
        self.table[idx]
    }
}

struct ComponentReader<'a> {
    src:    &'a [u8],
    pix:    u16,
    pos:    usize,
    cpos:   u8,
}

impl<'a> ComponentReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self {
            src,
            pix:    0,
            pos:    0,
            cpos:   0,
        }
    }
    fn read(&mut self) -> DecoderResult<u8> {
        if self.pos >= self.src.len() {
            return Err(DecoderError::ShortData);
        }
        self.cpos = (self.cpos + 1) % 3;
        match self.cpos {
            1 => {
                self.pix = read_u16le(&self.src[self.pos..])?;
                self.pos += 2;
                Ok((self.pix & 0x1F) as u8)
            },
            2 => {
                Ok((self.pix >> 5) as u8 & 0x1F)
            },
            _ => {
                Ok((self.pix >> 10) as u8 & 0x1F)
            }
        }
    }
}

struct YQSDecoder {
    fr:         FileReader<File>,
    yframe:     [u8; WIDTH * HEIGHT],
    uframe:     [u8; WIDTH * HEIGHT / 4],
    vframe:     [u8; WIDTH * HEIGHT / 4],
    vdata:      Vec<u8>,
    abuf:       Vec<i16>,
    audio:      bool,
    arate:      u32,
    adpcm:      IMAState,
    yuv2rgb:    YUV2RGB,
    nframes:    u16,
    cur_frame:  u16,
}

impl YQSDecoder {
    fn luma_block_2x2(dst: &mut [u8], bits: &mut BitReader, pix: &mut ComponentReader, vecs: &[u8]) -> DecoderResult<()> {
        let mode = bits.read(2)?;
        let (line0, line1) = dst.split_at_mut(WIDTH);
        match mode {
            0 => {
                let val = pix.read()?;
                line0[0] = val;
                line0[1] = val;
                line1[0] = val;
                line1[1] = val;
            },
            2 => {
                line0[0] = pix.read()?;
                line0[1] = pix.read()?;
                line1[0] = pix.read()?;
                line1[1] = pix.read()?;
            },
            1 => {
                let idx = usize::from(pix.read()?);
                validate!((idx + 1) * 4 <= vecs.len());
                let quad = &vecs[idx * 4..][..4];
                line0[0] = quad[0];
                line0[1] = quad[1];
                line1[0] = quad[2];
                line1[1] = quad[3];
            },
            3 => {
                let idx = usize::from(pix.read()?);
                validate!((idx + 1) * 4 <= vecs.len());
                let quad = &vecs[idx * 4..][..4];
                line0[0] = quad[2];
                line0[1] = quad[3];
                line1[0] = quad[0];
                line1[1] = quad[1];
            },
            _ => unreachable!(),
        }
        Ok(())
    }
    fn luma_block_4x4(dst: &mut [u8], bits: &mut BitReader, pix: &mut ComponentReader, vecs: &[u8]) -> DecoderResult<()> {
        if !bits.read_bool()? {
            let val = pix.read()?;
            for line in dst.chunks_mut(WIDTH).take(4) {
                for el in line[..4].iter_mut() {
                    *el = val;
                }
            }
        } else {
            for i in 0..4 {
                if bits.read_bool()? {
                    let off = (i & 1) * 2 + (i >> 1) * WIDTH * 2;
                    Self::luma_block_2x2(&mut dst[off..], bits, pix, vecs)?;
                }
            }
        }
        Ok(())
    }
    fn unpack_luma(dst: &mut [u8; WIDTH * HEIGHT], bits: &mut BitReader, pix: &mut ComponentReader, vecs: &[u8]) -> DecoderResult<()> {
        for stripe in dst.chunks_exact_mut(WIDTH * 8) {
            for x in (0..WIDTH).step_by(8) {
                if !bits.read_bool()? {
                    continue;
                }
                if !bits.read_bool()? {
                    let val = pix.read()?;
                    for line in stripe.chunks_exact_mut(WIDTH) {
                        for el in line[x..][..8].iter_mut() {
                            *el = val;
                        }
                    }
                } else {
                    for i in 0..4 {
                        if bits.read_bool()? {
                            let off = x + (i & 1) * 4 + (i >> 1) * WIDTH * 4;
                            Self::luma_block_4x4(&mut stripe[off..], bits, pix, vecs)?;
                        }
                    }
                }
            }
        }
        Ok(())
    }
    fn chroma_block_2x2(dst: &mut [u8], bits: &mut BitReader, pix: &mut ComponentReader) -> DecoderResult<()> {
        let (line0, line1) = dst.split_at_mut(CWIDTH);
        if !bits.read_bool()? {
            let val = pix.read()?;
            line0[0] = val;
            line0[1] = val;
            line1[0] = val;
            line1[1] = val;
        } else {
            line0[0] = pix.read()?;
            line0[1] = pix.read()?;
            line1[0] = pix.read()?;
            line1[1] = pix.read()?;
        }
        Ok(())
    }
    fn unpack_chroma(dst: &mut [u8; CWIDTH * CHEIGHT], bits: &mut BitReader, pix: &mut ComponentReader) -> DecoderResult<()> {
        for stripe in dst.chunks_exact_mut(CWIDTH * 4) {
            for x in (0..CWIDTH).step_by(4) {
                if !bits.read_bool()? {
                    continue;
                }
                if !bits.read_bool()? {
                    let val = pix.read()?;
                    for line in stripe.chunks_exact_mut(CWIDTH) {
                        for el in line[x..][..4].iter_mut() {
                            *el = val;
                        }
                    }
                } else {
                    for i in 0..4 {
                        if bits.read_bool()? {
                            let off = x + (i & 1) * 2 + (i >> 1) * CWIDTH * 2;
                            Self::chroma_block_2x2(&mut stripe[off..], bits, pix)?;
                        }
                    }
                }
            }
        }
        Ok(())
    }
    fn convert_frame(&self) -> Vec<u16> {
        let mut frame = vec![0; WIDTH * HEIGHT];
        for ((dstrip, ystrip), (uline, vline)) in frame.chunks_exact_mut(WIDTH * 2)
                .zip(self.yframe.chunks_exact(WIDTH * 2))
                .zip(self.uframe.chunks_exact(CWIDTH).zip(self.vframe.chunks_exact(CWIDTH))) {
            let (dline0, dline1) = dstrip.split_at_mut(WIDTH);
            let (yline0, yline1) = ystrip.split_at(WIDTH);
            for ((dst, luma), (&u, &v)) in dline0.chunks_exact_mut(2).zip(yline0.chunks_exact(2))
                    .zip(uline.iter().zip(vline.iter())) {
                dst[0] = self.yuv2rgb.convert(luma[0], u, v);
                dst[1] = self.yuv2rgb.convert(luma[1], u, v);
            }
            for ((dst, luma), (&u, &v)) in dline1.chunks_exact_mut(2).zip(yline1.chunks_exact(2))
                    .zip(uline.iter().zip(vline.iter())) {
                dst[0] = self.yuv2rgb.convert(luma[0], u, v);
                dst[1] = self.yuv2rgb.convert(luma[1], u, v);
            }
        }
        frame
    }
}

impl InputSource for YQSDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: 15,
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    1,
                    sample_type: AudioSample::S16,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.audio && self.abuf.len() >= ABLK_SIZE {
            let mut audio = vec![0; ABLK_SIZE];
            audio.copy_from_slice(&self.abuf[..ABLK_SIZE]);
            self.abuf.drain(..ABLK_SIZE);
            self.audio = false;
            return Ok((1, Frame::AudioS16(audio)));
        }

        if self.cur_frame >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.cur_frame += 1;

        let br = &mut self.fr;
        let size  = br.read_u32le()? as usize;
        validate!((size & 0x7FF) == 0 && (0x800..=0x20000).contains(&size));
        self.vdata.resize(size, 0);
        br.read_buf(&mut self.vdata[4..])?;

        const DATA_OFF: usize = 36 + 32 * 4;

        let ypix_off = read_u32le(&self.vdata[4..])? as usize;
        let ybit_off = read_u32le(&self.vdata[8..])? as usize;
        let upix_off = read_u32le(&self.vdata[12..])? as usize;
        let ubit_off = read_u32le(&self.vdata[16..])? as usize;
        let vpix_off = read_u32le(&self.vdata[20..])? as usize;
        let vbit_off = read_u32le(&self.vdata[24..])? as usize;
        let aud_off  = read_u32le(&self.vdata[28..])? as usize;
        let aud_len  = read_u32le(&self.vdata[32..])? as usize;

        if aud_len > 0 {
            validate!(aud_off >= DATA_OFF);
            validate!(aud_off + aud_len <= self.vdata.len());
            for &b in self.vdata[aud_off..][..aud_len].iter() {
                self.abuf.push(self.adpcm.expand_sample(b >> 4));
                self.abuf.push(self.adpcm.expand_sample(b & 0xF));
            }
        }

        validate!(ypix_off >= DATA_OFF && ybit_off >= DATA_OFF);
        let mut br = BitReader::new(&self.vdata[ybit_off..], BitReaderMode::LE);
        let mut pix = ComponentReader::new(&self.vdata[ypix_off..]);
        Self::unpack_luma(&mut self.yframe, &mut br, &mut pix, &self.vdata[36..])?;

        validate!(upix_off >= DATA_OFF && ubit_off >= DATA_OFF);
        let mut br = BitReader::new(&self.vdata[ubit_off..], BitReaderMode::LE);
        let mut pix = ComponentReader::new(&self.vdata[upix_off..]);
        Self::unpack_chroma(&mut self.uframe, &mut br, &mut pix)?;

        validate!(vpix_off >= DATA_OFF && vbit_off >= DATA_OFF);
        let mut br = BitReader::new(&self.vdata[vbit_off..], BitReaderMode::LE);
        let mut pix = ComponentReader::new(&self.vdata[vpix_off..]);
        Self::unpack_chroma(&mut self.vframe, &mut br, &mut pix)?;

        let frame = self.convert_frame();
        self.audio = self.arate > 0;

        Ok((0, Frame::VideoRGB16(frame)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);

    let nframes = fr.read_u16le()?;
    validate!(nframes > 0);
    let arate = u32::from(fr.read_u16le()?);
    fr.seek(SeekFrom::Start(0x800))?;

    Ok(Box::new(YQSDecoder {
        fr,
        yframe:     [0x00; WIDTH * HEIGHT],
        uframe:     [0xF; CWIDTH * CHEIGHT],
        vframe:     [0xF; CWIDTH * CHEIGHT],
        vdata:      Vec::new(),
        abuf:       Vec::with_capacity(ABLK_SIZE),
        audio:      false,
        arate,
        adpcm:      IMAState::new(),
        yuv2rgb:    YUV2RGB::new(),
        nframes,
        cur_frame:  0,
    }))
}
